package de.lxtools.noteshop.ui

import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.WindowInsetsSides
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.navigationBarsPadding
import androidx.compose.foundation.layout.only
import androidx.compose.foundation.layout.safeDrawing
import androidx.compose.foundation.layout.windowInsetsPadding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.List
import androidx.compose.material.icons.automirrored.filled.PlaylistAdd
import androidx.compose.material.icons.automirrored.filled.PlaylistAddCheck
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.ExposurePlus1
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.BottomAppBar
import androidx.compose.material3.Button
import androidx.compose.material3.DrawerValue
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalNavigationDrawer
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.rememberDrawerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.core.content.edit
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.viewmodel.compose.viewModel
import de.lxtools.noteshop.AppViewModelProvider
import de.lxtools.noteshop.BiometricAuthenticator
import de.lxtools.noteshop.MainActivityViewModel
import de.lxtools.noteshop.R
import de.lxtools.noteshop.Screen
import de.lxtools.noteshop.data.AppContainer
import de.lxtools.noteshop.findActivity
import de.lxtools.noteshop.security.hashString
import de.lxtools.noteshop.ui.appshell.AppContent
import de.lxtools.noteshop.ui.appshell.AppDialogs
import de.lxtools.noteshop.ui.appshell.AppDrawer
import de.lxtools.noteshop.ui.appshell.AppTopBar
import de.lxtools.noteshop.ui.notes.NotesViewModel
import de.lxtools.noteshop.ui.recipes.RecipesViewModel
import de.lxtools.noteshop.ui.shopping.ShoppingListsViewModel
import de.lxtools.noteshop.ui.theme.ColorTheme
import de.lxtools.noteshop.ui.webapp.WebAppIntegrationViewModel
import kotlinx.coroutines.launch

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun AppShell(
    container: AppContainer,
    mainActivityViewModel: MainActivityViewModel,
    notesViewModel: NotesViewModel = viewModel(factory = AppViewModelProvider.Factory(container)),
    shoppingListsViewModel: ShoppingListsViewModel = viewModel(factory = AppViewModelProvider.Factory(container)),
    recipesViewModel: RecipesViewModel = viewModel(factory = AppViewModelProvider.Factory(container)),
    colorTheme: ColorTheme,
    onThemeChange: (ColorTheme) -> Unit,
    isAppLockEnabled: Boolean,
    onAppLockChange: (Boolean) -> Unit,
    sharedPrefs: android.content.SharedPreferences,
    biometricAuthenticator: BiometricAuthenticator,
    canUseBiometrics: Boolean,
    createDatabaseLauncher: androidx.activity.result.ActivityResultLauncher<String>,
    openDatabaseLauncher: androidx.activity.result.ActivityResultLauncher<Array<String>>
) {
    val context = LocalContext.current
    val scope = rememberCoroutineScope()

    val drawerState = rememberDrawerState(initialValue = DrawerValue.Closed)
    val (currentScreen, setCurrentScreen) = rememberSaveable {
        val defaultStartScreenRoute = sharedPrefs.getString("default_start_screen", "last_screen")
        val initialScreen = when (defaultStartScreenRoute) {
            Screen.ShoppingLists.route -> Screen.ShoppingLists
            Screen.Notes.route -> Screen.Notes
            Screen.Recipes.route -> Screen.Recipes
            else -> { // "last_screen" or unset
                val lastScreenRoute =
                    sharedPrefs.getString("last_screen_route", Screen.ShoppingLists.route)
                when (lastScreenRoute) {
                    Screen.Notes.route -> Screen.Notes
                    Screen.Recipes.route -> Screen.Recipes
                    else -> Screen.ShoppingLists
                }
            }
        }
        mutableStateOf(initialScreen)
    }

    LaunchedEffect(currentScreen) {
        if (currentScreen == Screen.ShoppingLists || currentScreen == Screen.Notes || currentScreen == Screen.Recipes) {
            sharedPrefs.edit {
                putString("last_screen_route", currentScreen.route)
            }
        }
    }

    LaunchedEffect(Unit) {
        val firstLaunchCompleted = sharedPrefs.getBoolean("first_launch_completed", false)
        if (!firstLaunchCompleted) {
            setCurrentScreen(Screen.GuidedTour)
            sharedPrefs.edit { putBoolean("first_launch_completed", true) }
        }
    }

    val (selectedListId, setSelectedListId) = rememberSaveable { mutableStateOf<Int?>(null) }
    val (selectedNoteId, setSelectedNoteId) = rememberSaveable { mutableStateOf<Int?>(null) }
    val (selectedRecipeId, setSelectedRecipeId) = rememberSaveable { mutableStateOf<Int?>(null) }

    val (unlockedItemIds, setUnlockedItemIds) = rememberSaveable { mutableStateOf<Set<Int>>(emptySet()) }

    val (showListDialog, setShowListDialog) = rememberSaveable { mutableStateOf(false) }
    val listDetails by shoppingListsViewModel.listDetails.collectAsState()

    val (showNoteDialog, setShowNoteDialog) = rememberSaveable { mutableStateOf(false) }
    val noteDetails by notesViewModel.noteDetails.collectAsState()

    val (showRecipeDialog, setShowRecipeDialog) = rememberSaveable { mutableStateOf(false) }
    val recipeDetails by recipesViewModel.recipeDetails.collectAsState()

    val (showJsonDialog, setShowJsonDialog) = rememberSaveable { mutableStateOf(false) }
    val (showSetPasswordDialogState, setShowSetPasswordDialogState) = rememberSaveable { mutableStateOf(false) }
    val (showSetRecipePasswordDialog, setShowSetRecipePasswordDialog) = rememberSaveable { mutableStateOf(false) }
    val (showSetListPasswordDialog, setShowSetListPasswordDialog) = rememberSaveable { mutableStateOf(false) }

    val (showUnlockPasswordDialog, setShowUnlockPasswordDialog) = rememberSaveable { mutableStateOf(false) }
    val (unlockErrorMessage, setUnlockErrorMessage) = rememberSaveable { mutableStateOf<String?>(null) }
    val (itemToUnlockId, setItemToUnlockId) = rememberSaveable { mutableStateOf<Int?>(null) }
    val (itemToUnlockType, setItemToUnlockType) = rememberSaveable { mutableStateOf<Screen?>(null) }

    val (showChooseLockMethodDialog, setShowChooseLockMethodDialog) = rememberSaveable { mutableStateOf(false) }
    val (itemToLockId, setItemToLockId) = rememberSaveable { mutableStateOf<Int?>(null) }
    val (itemToLockType, setItemToLockType) = rememberSaveable { mutableStateOf<LockableItemType?>(null) }

    val onUnlockItem: (Int, Screen, Int) -> Unit = { id, type, protectionType ->
        when (protectionType) {
            1 -> { // Password protected
                setItemToUnlockId(id)
                setItemToUnlockType(type)
                setShowUnlockPasswordDialog(true)
            }
            2 -> { // Biometric protected
                biometricAuthenticator.promptBiometricAuth(
                    title = context.getString(R.string.unlock_item),
                    subtitle = "",
                    fragmentActivity = context.findActivity() as FragmentActivity,
                    onSuccess = { _ ->
                        setUnlockedItemIds(unlockedItemIds + id)
                        when (type) {
                            is Screen.ShoppingListDetail -> setSelectedListId(id)
                            is Screen.NoteDetail -> setSelectedNoteId(id)
                            is Screen.RecipeDetail -> setSelectedRecipeId(id)
                            else -> {}
                        }
                        setCurrentScreen(type)
                    }, onFailed = {
                        setItemToUnlockId(id)
                        setItemToUnlockType(type)
                        setShowUnlockPasswordDialog(true)
                    },
                    onError = { _, _ ->
                        setItemToUnlockId(id)
                        setItemToUnlockType(type)
                        setShowUnlockPasswordDialog(true)
                    }
                )
            }
        }
    }

    val newItemName by shoppingListsViewModel.newItemName.collectAsState()
    val showCompletedItems by shoppingListsViewModel.showCompletedItems.collectAsState()
    val shoppingListWithItems by shoppingListsViewModel.getShoppingListWithItemsStream(
        selectedListId ?: 0
    )
        .collectAsState(initial = null)

    val (shoppingListsTitle, setShoppingListsTitle) = remember {
        mutableStateOf(
            sharedPrefs.getString(
                "shopping_lists_title",
                context.getString(R.string.menu_shopping_lists)
            ) ?: context.getString(R.string.menu_shopping_lists)
        )
    }
    val (showRenameDialog, setShowRenameDialog) = rememberSaveable { mutableStateOf(false) }

    val (recipesTitle, setRecipesTitle) = remember {
        mutableStateOf(
            sharedPrefs.getString(
                "recipes_title",
                context.getString(R.string.menu_recipes)
            ) ?: context.getString(R.string.menu_recipes)
        )
    }
    val (showRecipeRenameDialog, setShowRecipeRenameDialog) = rememberSaveable { mutableStateOf(false) }

    val (startScreen, setStartScreen) = remember {
        mutableStateOf(
            sharedPrefs.getString(
                "default_start_screen",
                "last_screen"
            ) ?: "last_screen"
        )
    }

    val (showChooseDatabaseLocationDialog, setShowChooseDatabaseLocationDialog) = rememberSaveable { mutableStateOf(false) }

    if (showChooseDatabaseLocationDialog) {
        AlertDialog(
            onDismissRequest = { setShowChooseDatabaseLocationDialog(false) },
            title = { Text(stringResource(R.string.database_location)) },
            text = { Text(stringResource(R.string.database_location_dialog_text)) },
            confirmButton = {
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.SpaceEvenly
                ) {
                    Button(onClick = {
                        setShowChooseDatabaseLocationDialog(false)
                        createDatabaseLauncher.launch("noteshop.db")
                    }) {
                        Text(stringResource(R.string.create_database))
                    }
                    Button(onClick = {
                        setShowChooseDatabaseLocationDialog(false)
                        openDatabaseLauncher.launch(arrayOf("application/vnd.sqlite3", "application/octet-stream"))
                    }) {
                        Text(stringResource(R.string.open_database))
                    }
                }
            }
        )
    }

    val txtImportLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.OpenDocument(),
        onResult = { uri ->
            uri?.let {
                val fileName =
                    context.contentResolver.query(it, null, null, null, null)?.use { cursor ->
                        val nameIndex =
                            cursor.getColumnIndex(android.provider.OpenableColumns.DISPLAY_NAME)
                        cursor.moveToFirst()
                        cursor.getString(nameIndex)
                    } ?: "Imported List"
                val listName = fileName.substringBeforeLast(".")

                context.contentResolver.openInputStream(it)?.use { inputStream ->
                    val content = inputStream.readBytes().toString(Charsets.UTF_8)
                    scope.launch {
                        shoppingListsViewModel.importListFromTxt(listName, content)
                    }
                }
                setShowListDialog(false)
            }
        }
    )

    val exportLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("text/plain"),
        onResult = { uri ->
            uri?.let {
                shoppingListWithItems?.let { list ->
                    val content = shoppingListsViewModel.formatShoppingListForExport(list)
                    context.contentResolver.openOutputStream(it)?.use { outputStream ->
                        outputStream.write(content.toByteArray())
                    }
                    android.widget.Toast.makeText(
                        context,
                        R.string.data_export_successful,
                        android.widget.Toast.LENGTH_SHORT
                    ).show()
                }
            }
        }
    )

    val noteExportLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("text/plain"),
        onResult = { uri ->
            uri?.let {
                notesViewModel.noteDetails.value.toNote().let { note ->
                    val content = notesViewModel.formatNoteForExport(note)
                    context.contentResolver.openOutputStream(it)?.use { outputStream ->
                        outputStream.write(content.toByteArray())
                    }
                    android.widget.Toast.makeText(
                        context,
                        R.string.data_export_successful,
                        android.widget.Toast.LENGTH_SHORT
                    ).show()
                }
            }
        }
    )

    val recipeExportLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("text/plain"),
        onResult = { uri ->
            uri?.let {
                recipesViewModel.recipeDetails.value.toRecipe().let { recipe ->
                    val content = recipesViewModel.formatRecipeForExport(recipe)
                    context.contentResolver.openOutputStream(it)?.use { outputStream ->
                        outputStream.write(content.toByteArray())
                    }
                    android.widget.Toast.makeText(
                        context,
                        R.string.data_export_successful,
                        android.widget.Toast.LENGTH_SHORT
                    ).show()
                }
            }
        }
    )

    val noteImportLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.OpenDocument(),
        onResult = { uri ->
            uri?.let {
                context.contentResolver.openInputStream(it)?.use { inputStream ->
                    val content = inputStream.readBytes().toString(Charsets.UTF_8)
                    val fileName =
                        context.contentResolver.query(it, null, null, null, null)?.use { cursor ->
                            val nameIndex =
                                cursor.getColumnIndex(android.provider.OpenableColumns.DISPLAY_NAME)
                            cursor.moveToFirst()
                            cursor.getString(nameIndex)
                        } ?: "Imported Note"

                    scope.launch {
                        notesViewModel.resetNoteDetails()
                        notesViewModel.updateNoteDetails(
                            notesViewModel.noteDetails.value.copy(
                                title = fileName.substringBeforeLast("."),
                                content = content
                            )
                        )
                        notesViewModel.saveNote()
                    }
                }
            }
        }
    )

    val recipeImportLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.OpenDocument(),
        onResult = { uri ->
            uri?.let {
                context.contentResolver.openInputStream(it)?.use { inputStream ->
                    val content = inputStream.readBytes().toString(Charsets.UTF_8)
                    val fileName =
                        context.contentResolver.query(it, null, null, null, null)?.use { cursor ->
                            val nameIndex =
                                cursor.getColumnIndex(android.provider.OpenableColumns.DISPLAY_NAME)
                            cursor.moveToFirst()
                            cursor.getString(nameIndex)
                        } ?: "Imported Recipe"

                    scope.launch {
                        recipesViewModel.resetRecipeDetails()
                        recipesViewModel.updateRecipeDetails(
                            recipesViewModel.recipeDetails.value.copy(
                                title = fileName.substringBeforeLast("."),
                                content = content
                            )
                        )
                        recipesViewModel.saveRecipe()
                    }
                }
            }
        }
    )

    val webAppIntegrationViewModel: WebAppIntegrationViewModel = viewModel(factory = AppViewModelProvider.Factory(container))
    val onAuthenticateAndTogglePasswordVisibility: (Boolean) -> Unit = { isPassword ->
        val activity = context.findActivity() as FragmentActivity
        biometricAuthenticator.promptBiometricAuth(
            title = context.getString(R.string.authenticate_to_perform_action),
            subtitle = "",
            fragmentActivity = activity,
            onSuccess = { _ ->
                if (isPassword) {
                    webAppIntegrationViewModel.togglePasswordVisibility()
                } else {
                    webAppIntegrationViewModel.toggleDeletePasswordVisibility()
                }
            },
            onFailed = {
                android.widget.Toast.makeText(context, R.string.unlock_failed, android.widget.Toast.LENGTH_SHORT).show()
            },
            onError = { _, _ ->
                android.widget.Toast.makeText(context, R.string.unlock_failed, android.widget.Toast.LENGTH_SHORT).show()
            }
        )
    }

    ModalNavigationDrawer(
        drawerState = drawerState,
        drawerContent = {
            AppDrawer(
                drawerState = drawerState,
                currentScreen = currentScreen,
                onScreenChange = { setCurrentScreen(it) },
                shoppingListsViewModel = shoppingListsViewModel,
                notesViewModel = notesViewModel,
                shoppingListsTitle = shoppingListsTitle,
                recipesTitle = recipesTitle,
                onShowRenameDialog = { setShowRenameDialog(true) },
                onShowRecipeRenameDialog = { setShowRecipeRenameDialog(true) },
                onShowJsonDialog = { setShowJsonDialog(true) },
                databasePath = mainActivityViewModel.databasePath.collectAsState().value,
                onChooseDatabaseLocation = { setShowChooseDatabaseLocationDialog(true) },
                onSetSelectedListId = { setSelectedListId(it) },
                onSetSelectedNoteId = { setSelectedNoteId(it) }
            )
        }
    ) {
        Scaffold(
            modifier = Modifier
                .fillMaxSize()
                .windowInsetsPadding(WindowInsets.safeDrawing.only(WindowInsetsSides.Horizontal)),
            topBar = {
                AppTopBar(
                    currentScreen = currentScreen,
                    onScreenChange = { setCurrentScreen(it) },
                    drawerState = drawerState,
                    shoppingListsViewModel = shoppingListsViewModel,
                    notesViewModel = notesViewModel,
                    recipesViewModel = recipesViewModel,
                    shoppingListsTitle = shoppingListsTitle,
                    recipesTitle = recipesTitle,
                    onShowListDialog = { setShowListDialog(true) },
                    onShowNoteDialog = { setShowNoteDialog(true) },
                    onShowRecipeDialog = { setShowRecipeDialog(true) },
                    selectedListId = selectedListId,
                    onSetSelectedListId = { setSelectedListId(it) },
                    onSetSelectedNoteId = { setSelectedNoteId(it) },
                    onSetSelectedRecipeId = { setSelectedRecipeId(it) },
                    exportLauncher = exportLauncher,
                    noteExportLauncher = noteExportLauncher,
                    recipeExportLauncher = recipeExportLauncher,
                    onShowChooseLockMethodDialog = { type, id ->
                        setItemToLockType(type)
                        setItemToLockId(id)
                        setShowChooseLockMethodDialog(true)
                    }
                )
            },
            bottomBar = {
                if (currentScreen == Screen.ShoppingListDetail) {
                    BottomAppBar(
                        modifier = Modifier
                            .navigationBarsPadding()
                            .imePadding()
                            .height(56.dp),
                        containerColor = MaterialTheme.colorScheme.surface
                    ) {
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            horizontalArrangement = Arrangement.SpaceEvenly,
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            IconButton(onClick = {
                                scope.launch {
                                    shoppingListWithItems?.let { listWithItems ->
                                        shoppingListsViewModel.saveShoppingListItem(
                                            listWithItems.shoppingList.id,
                                            newItemName,
                                            shouldSplit = true
                                        )
                                    }; shoppingListsViewModel.onNewItemNameChange("")
                                }
                            }, enabled = newItemName.isNotBlank()) {
                                Icon(
                                    Icons.AutoMirrored.Filled.PlaylistAdd,
                                    contentDescription = stringResource(R.string.add_item_icon_desc),
                                    tint = MaterialTheme.colorScheme.onSurface
                                )
                            }
                            IconButton(onClick = {
                                scope.launch {
                                    shoppingListWithItems?.let { listWithItems ->
                                        shoppingListsViewModel.saveShoppingListItem(
                                            listWithItems.shoppingList.id,
                                            newItemName,
                                            shouldSplit = false
                                        )
                                    }; shoppingListsViewModel.onNewItemNameChange("")
                                 }
                            }, enabled = newItemName.isNotBlank()) {
                                Icon(
                                    Icons.Filled.ExposurePlus1,
                                    contentDescription = stringResource(R.string.add_item_simple_icon_desc),
                                    tint = MaterialTheme.colorScheme.onSurface
                                )
                            }
                            IconButton(onClick = { shoppingListsViewModel.toggleShowCompletedItems() }) {
                                Icon(
                                    imageVector = if (showCompletedItems) Icons.AutoMirrored.Filled.List else Icons.AutoMirrored.Filled.PlaylistAddCheck,
                                    contentDescription = "Toggle completed",
                                    tint = MaterialTheme.colorScheme.onSurface
                                )
                            }
                            IconButton(
                                onClick = {
                                    scope.launch {
                                        selectedListId?.let {
                                            shoppingListsViewModel.deleteCompletedItems(
                                                it
                                            )
                                        }
                                    }
                                },
                                enabled = shoppingListWithItems?.items?.any { it.isChecked } == true) {
                                Icon(
                                    Icons.Default.Delete,
                                    contentDescription = "Delete completed",
                                    tint = MaterialTheme.colorScheme.onSurface
                                )
                            }
                        }
                    }
                }
            }
        ) { innerPadding ->
            AppContent(
                innerPadding = innerPadding,
                currentScreen = currentScreen,
                onScreenChange = { setCurrentScreen(it) },
                notesViewModel = notesViewModel,
                shoppingListsViewModel = shoppingListsViewModel,
                recipesViewModel = recipesViewModel,
                selectedListId = selectedListId,
                onSetSelectedListId = { setSelectedListId(it) },
                onShowListDialog = { setShowListDialog(true) },
                selectedNoteId = selectedNoteId,
                onSetSelectedNoteId = { setSelectedNoteId(it) },
                onShowNoteDialog = { setShowNoteDialog(true) },
                selectedRecipeId = selectedRecipeId,
                onSetSelectedRecipeId = { setSelectedRecipeId(it) },
                onShowRecipeDialog = { setShowRecipeDialog(true) },
                unlockedItemIds = unlockedItemIds,
                onUnlockItem = onUnlockItem,
                shoppingListsTitle = shoppingListsTitle,
                onThemeChange = onThemeChange,
                colorTheme = colorTheme,
                isAppLockEnabled = isAppLockEnabled,
                onAppLockChange = onAppLockChange,
                biometricAuthenticator = biometricAuthenticator,
                sharedPrefs = sharedPrefs,
                onResetShoppingListsTitle = {
                    sharedPrefs.edit { remove("shopping_lists_title") }
                    setShoppingListsTitle(context.getString(R.string.menu_shopping_lists))
                },
                onResetRecipesTitle = {
                    sharedPrefs.edit { remove("recipes_title") }
                    setRecipesTitle(context.getString(R.string.menu_recipes))
                },
                startScreen = startScreen,
                onStartScreenChange = {
                    sharedPrefs.edit { putString("default_start_screen", it) }
                    setStartScreen(it)
                },
                canUseBiometrics = canUseBiometrics,
                webAppIntegrationViewModel = webAppIntegrationViewModel,
                onAuthenticateAndTogglePasswordVisibility = onAuthenticateAndTogglePasswordVisibility
            )
        }
    }

    AppDialogs(
        mainActivityViewModel = mainActivityViewModel,
        notesViewModel = notesViewModel,
        shoppingListsViewModel = shoppingListsViewModel,
        recipesViewModel = recipesViewModel,
        showListDialog = showListDialog,
        onShowListDialogChange = { setShowListDialog(it) },
        listDetails = listDetails,
        onListDetailsChange = { shoppingListsViewModel.updateListDetails(it) },

        onSetListProtection = { password ->
            selectedListId?.let { listId ->
                shoppingListsViewModel.setProtection(listId, password)
                setCurrentScreen(Screen.ShoppingLists)
            }
            setSelectedListId(null)
            setShowSetListPasswordDialog(false)
        },

        txtImportLauncher = txtImportLauncher,
        showNoteDialog = showNoteDialog,
        onShowNoteDialogChange = { setShowNoteDialog(it) },
        noteDetails = noteDetails,
        onNoteDetailsChange = { notesViewModel.updateNoteDetails(it) },

        noteImportLauncher = noteImportLauncher,
        showRecipeDialog = showRecipeDialog,
        onShowRecipeDialogChange = { setShowRecipeDialog(it) },
        recipeDetails = recipeDetails,
        onRecipeDetailsChange = { recipesViewModel.updateRecipeDetails(it) },

        recipeImportLauncher = recipeImportLauncher,
        recipesTitle = recipesTitle,
        showJsonDialog = showJsonDialog,
        onShowJsonDialogChange = { setShowJsonDialog(it) },
        shoppingListsTitle = shoppingListsTitle,
        showSetPasswordDialog = showSetPasswordDialogState,
        onShowSetPasswordDialogChange = { setShowSetPasswordDialogState(it) },
        onSetNotePassword = { password ->
            notesViewModel.setProtectionPassword(password)
            setSelectedNoteId(null)
            setShowSetPasswordDialogState(false)
            setCurrentScreen(Screen.Notes)
        },
        showSetRecipePasswordDialog = showSetRecipePasswordDialog,
        onShowSetRecipePasswordDialogChange = { setShowSetRecipePasswordDialog(it) },
        onSetRecipePassword = { password ->
            recipesViewModel.setProtectionPassword(password)
            setSelectedRecipeId(null)
            setShowSetRecipePasswordDialog(false)
            setCurrentScreen(Screen.Recipes)
        },
        showSetListPasswordDialog = showSetListPasswordDialog,
        onShowSetListPasswordDialogChange = { setShowSetListPasswordDialog(it) },
        showUnlockPasswordDialog = showUnlockPasswordDialog,
        onShowUnlockPasswordDialogChange = {
            setShowUnlockPasswordDialog(it)
            if (!it) {
                setUnlockErrorMessage(null)
                setItemToUnlockId(null)
                setItemToUnlockType(null)
            }
        },
        onUnlock = { password ->
            var isPasswordCorrect = false
            val hashedPassword = hashString(password)

            when (itemToUnlockType) {
                Screen.NoteDetail -> {
                    val note = notesViewModel.uiState.value.noteList.find { it.id == itemToUnlockId }
                    if (note != null && note.protectionHash == hashedPassword) {
                        isPasswordCorrect = true
                        setSelectedNoteId(itemToUnlockId)
                    }
                }
                Screen.RecipeDetail -> {
                    val recipe = recipesViewModel.uiState.value.recipeList.find { it.id == itemToUnlockId }
                    if (recipe != null && recipe.protectionHash == hashedPassword) {
                        isPasswordCorrect = true
                        setSelectedRecipeId(itemToUnlockId)
                    }
                }
                Screen.ShoppingListDetail -> {
                    val listWithItems = shoppingListsViewModel.uiState.value.shoppingLists.find { it.shoppingList.id == itemToUnlockId }
                    if (listWithItems != null && listWithItems.shoppingList.protectionHash == hashedPassword) {
                        isPasswordCorrect = true
                        setSelectedListId(itemToUnlockId)
                    }
                }
                else -> {} // Should not happen
            }

            if (isPasswordCorrect) {
                setUnlockErrorMessage(null)
                itemToUnlockId?.let { id -> setUnlockedItemIds(unlockedItemIds + id) }
                setCurrentScreen(itemToUnlockType!!)
                setItemToUnlockId(null)
                setItemToUnlockType(null)
                setShowUnlockPasswordDialog(false)
            } else {
                setUnlockErrorMessage(context.getString(R.string.incorrect_password))
            }
        },
        biometricAuthenticator = biometricAuthenticator,
        unlockErrorMessage = unlockErrorMessage,
        sharedPrefs = sharedPrefs,
        showRenameDialog = showRenameDialog,
        onShowRenameDialogChange = { setShowRenameDialog(it) },
        onRenameShoppingListsTitle = { newTitle ->
            sharedPrefs.edit { putString("shopping_lists_title", newTitle) }
            setShoppingListsTitle(newTitle)
        },
        showRecipeRenameDialog = showRecipeRenameDialog,
        onShowRecipeRenameDialogChange = { setShowRecipeRenameDialog(it) },
        onRenameRecipesTitle = { newTitle ->
            sharedPrefs.edit { putString("recipes_title", newTitle) }
            setRecipesTitle(newTitle)
        },
        showChooseLockMethodDialog = showChooseLockMethodDialog,
        onShowChooseLockMethodDialogChange = { setShowChooseLockMethodDialog(it) },
        onConfirmLockMethod = { lockMethod ->
            itemToLockType?.let { type ->
                itemToLockId?.let { id ->
                    when (type) {
                        LockableItemType.SHOPPING_LIST -> {
                            when (lockMethod) {
                                LockMethod.PASSWORD -> {
                                    setSelectedListId(id)
                                    setShowSetListPasswordDialog(true)
                                }
                                LockMethod.BIOMETRIC -> {
                                    shoppingListsViewModel.setProtectionBiometric(id)
                                    setCurrentScreen(Screen.ShoppingLists)
                                }
                            }
                        }
                        LockableItemType.NOTE -> {
                            when (lockMethod) {
                                LockMethod.PASSWORD -> {
                                    setSelectedNoteId(id)
                                    setShowSetPasswordDialogState(true)
                                }
                                LockMethod.BIOMETRIC -> {
                                    notesViewModel.setProtectionBiometric(id)
                                    setCurrentScreen(Screen.Notes)
                                }
                            }
                        }
                        LockableItemType.RECIPE -> {
                            when (lockMethod) {
                                LockMethod.PASSWORD -> {
                                    setSelectedRecipeId(id)
                                    setShowSetRecipePasswordDialog(true)
                                }
                                LockMethod.BIOMETRIC -> {
                                    recipesViewModel.setProtectionBiometric(id)
                                    setCurrentScreen(Screen.Recipes)
                                }
                            }
                        }
                    }
                }
            }
            setShowChooseLockMethodDialog(false)
            setItemToLockId(null)
            setItemToLockType(null)
        },
        canUseBiometrics = canUseBiometrics
    )
}
