package de.lxtools.noteshop.ui

import android.content.pm.PackageManager
import android.os.Build
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.BlendMode
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.dp
import de.lxtools.noteshop.R

@Composable
fun AboutScreen() {
    val context = LocalContext.current

    var versionName = "Unknown"
    var versionCode = 0L

    try {
        val packageInfo = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            context.packageManager.getPackageInfo(context.packageName, PackageManager.PackageInfoFlags.of(0L))
        } else {
            @Suppress("DEPRECATION")
            context.packageManager.getPackageInfo(context.packageName, 0)
        }
        versionName = packageInfo.versionName ?: "Unknown"
        versionCode = packageInfo.longVersionCode
    } catch (e: PackageManager.NameNotFoundException) {
        e.printStackTrace()
    }

    LazyColumn(
        modifier = Modifier
            .fillMaxSize()
            .padding(16.dp)
    ) {
        item {
            Image(
                painter = painterResource(id = R.drawable.noteshop_logo),
                contentDescription = stringResource(R.string.app_name),
                modifier = Modifier
                    .fillMaxWidth()
                    .height(120.dp),
                contentScale = ContentScale.Fit,
                colorFilter = ColorFilter.tint(MaterialTheme.colorScheme.onSurface, BlendMode.SrcIn)
            )
        }
        item {
            Text(
                text = stringResource(R.string.app_name),
                style = MaterialTheme.typography.headlineMedium,
                modifier = Modifier.padding(top = 16.dp, bottom = 8.dp)
            )
        }
        item {
            Text(
                text = "Version: $versionName (Build $versionCode)",
                style = MaterialTheme.typography.bodyLarge,
                modifier = Modifier.padding(bottom = 4.dp)
            )
        }
        item {
            Text(
                text = "© 2025 LXTools",
                style = MaterialTheme.typography.bodyMedium,
                modifier = Modifier.padding(bottom = 16.dp)
            )
        }

        item { Spacer(modifier = Modifier.height(16.dp)) }

        // Donation Section
        item {
            Text(
                text = stringResource(id = R.string.about_donations_title),
                style = MaterialTheme.typography.titleLarge,
                modifier = Modifier.padding(bottom = 8.dp)
            )
            val annotatedString = buildAnnotatedString {
                append(stringResource(id = R.string.about_donations_description))
                append(" ")
                val linkStyle = SpanStyle(
                    color = MaterialTheme.colorScheme.primary,
                    textDecoration = TextDecoration.Underline
                )
                pushLink(LinkAnnotation.Url("https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=polunga40@unity-mail.de"))
                withStyle(linkStyle) {
                    append(stringResource(id = R.string.about_donations_link))
                }
                pop()
                append(".")
            }
            Text(
                text = annotatedString,
                style = MaterialTheme.typography.bodyLarge
            )
        }

        item { Spacer(modifier = Modifier.height(24.dp)) }

        // License Section
        item {
            Text(
                text = stringResource(id = R.string.about_license_title),
                style = MaterialTheme.typography.titleLarge,
                modifier = Modifier.padding(bottom = 8.dp)
            )
            val licenseString = buildAnnotatedString {
                append(stringResource(id = R.string.about_license_description))
                append(" ")
                val linkStyle = SpanStyle(
                    color = MaterialTheme.colorScheme.primary,
                    textDecoration = TextDecoration.Underline
                )
                pushLink(LinkAnnotation.Url("https://www.gnu.org/licenses/gpl-3.0.html"))
                withStyle(linkStyle) {
                    append(stringResource(id = R.string.about_license_link))
                }
                pop()
                append(" ")
                append(stringResource(id = R.string.about_license_suffix))
            }
            Text(
                text = licenseString,
                style = MaterialTheme.typography.bodyLarge
            )
        }

        item { Spacer(modifier = Modifier.height(24.dp)) }

        // Privacy Policy Section
        item {
            Text(
                text = stringResource(id = R.string.about_privacy_policy_title),
                style = MaterialTheme.typography.titleLarge,
                modifier = Modifier.padding(bottom = 8.dp)
            )
            val privacyPolicyString = buildAnnotatedString {
                append(stringResource(id = R.string.about_privacy_policy_description))
                append(" ")
                val linkStyle = SpanStyle(
                    color = MaterialTheme.colorScheme.primary,
                    textDecoration = TextDecoration.Underline
                )
                pushLink(LinkAnnotation.Url("https://git.ilunix.de/punix/noteshop/src/branch/master/PRIVACY.md"))
                withStyle(linkStyle) {
                    append(stringResource(id = R.string.about_privacy_policy_link))
                }
                pop()
            }
            Text(
                text = privacyPolicyString,
                style = MaterialTheme.typography.bodyLarge
            )
        }

        item { Spacer(modifier = Modifier.height(24.dp)) }

        // Libraries Section
        item {
            Text(
                text = stringResource(id = R.string.about_libraries_title),
                style = MaterialTheme.typography.titleLarge,
                modifier = Modifier.padding(bottom = 8.dp)
            )
            Text(stringResource(id = R.string.about_libraries_description), style = MaterialTheme.typography.bodyLarge)
            Spacer(modifier = Modifier.height(8.dp))
            Text(stringResource(id = R.string.about_library_androidx), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_junit), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_reorderable), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_room), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_biometric), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_material_icons_extended), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_documentfile), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_google_material), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_compose_markdown), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_kotlinx_serialization_json), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_ktor), style = MaterialTheme.typography.bodyMedium)
            Text(stringResource(id = R.string.about_library_datastore), style = MaterialTheme.typography.bodyMedium)
        }
    }
}
