package de.lxtools.noteshop.data

import de.lxtools.noteshop.api.WebAppClient
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.firstOrNull

/**
 * Repository that provides insert, update, delete, and retrieve of data from a given data source.
 */
interface NoteshopRepository {
    /**
     * Retrieve all the notes from the the data source.
     */
    fun getAllNotesStream(): Flow<List<Note>>

    /**
     * Retrieve a specific note from the data source.
     */
    fun getNoteStream(id: Int): Flow<Note?>

    /**
     * Insert note in the data source
     */
    suspend fun insertNote(note: Note)

    /**
     * Insert or update note in the data source
     */
    suspend fun upsertNote(note: Note)

    /**
     * Update note in the data source
     */
    suspend fun updateNote(note: Note)

    /**
     * Delete note from the data source
     */
    suspend fun deleteNote(note: Note)

    /**
     * Get the number of notes.
     */
    suspend fun getNotesCount(): Int

    /**
     * Retrieve all the recipes from the the data source.
     */
    fun getAllRecipesStream(): Flow<List<Recipe>>

    /**
     * Retrieve a specific recipe from the data source.
     */
    fun getRecipeStream(id: Int): Flow<Recipe?>

    /**
     * Insert recipe in the data source
     */
    suspend fun insertRecipe(recipe: Recipe)

    /**
     * Insert or update recipe in the data source
     */
    suspend fun upsertRecipe(recipe: Recipe)

    /**
     * Update recipe in the data source
     */
    suspend fun updateRecipe(recipe: Recipe)

    /**
     * Delete recipe from the data source
     */
    suspend fun deleteRecipe(recipe: Recipe)

    /**
     * Get the number of recipes.
     */
    suspend fun getRecipesCount(): Int

    /**
     * Retrieve a specific shopping list by name from the data source.
     */
    fun getShoppingListByName(name: String): Flow<ShoppingList?>

    /**
     * Retrieve all the shopping lists with their items from the data source.
     */
    fun getAllShoppingListsWithItemsStream(): Flow<List<ShoppingListWithItems>>

    /**
     * Retrieve a specific shopping list with its items from the data source.
     */
    fun getShoppingListWithItemsStream(listId: Int): Flow<ShoppingListWithItems?>

    /**
     * Insert shopping list in the data source
     */
    suspend fun insertShoppingList(list: ShoppingList): Long

    /**
     * Insert or update shopping list in the data source
     */
    suspend fun upsertShoppingList(list: ShoppingList)

    /**
     * Update shopping list in the data source
     */
    suspend fun updateShoppingList(list: ShoppingList): Int

    /**
     * Update multiple shopping lists in the data source
     */
    suspend fun updateShoppingLists(lists: List<ShoppingList>)

    /**
     * Delete shopping list from the data source
     */
    suspend fun deleteShoppingList(list: ShoppingList)

    /**
     * Insert shopping list item in the data source
     */
    suspend fun insertShoppingListItem(item: ShoppingListItem)

    /**
     * Insert or update shopping list item in the data source
     */
    suspend fun upsertShoppingListItem(item: ShoppingListItem)

    /**
     * Insert multiple shopping list items in the data source
     */
    suspend fun insertShoppingListItems(items: List<ShoppingListItem>)

    /**
     * Update shopping list item in the data source
     */
    suspend fun updateShoppingListItem(item: ShoppingListItem)

    /**
     * Delete shopping list item from the data source
     */
    suspend fun deleteShoppingListItem(item: ShoppingListItem)

    /**
     * Delete multiple shopping list items from the data source
     */
    suspend fun deleteShoppingListItems(items: List<ShoppingListItem>)

    /**
     * Update multiple shopping list items in the data source
     */
    suspend fun updateShoppingListItems(items: List<ShoppingListItem>)

    /**
     * Get the number of shopping lists.
     */
    suspend fun getListsCount(): Int

    /**
     * Get the number of items in a specific shopping list.
     */
    suspend fun getItemsCount(listId: Int): Int

    /**
     * Get the standard list items from the assets.
     */
    fun getStandardListItems(): List<String>

    /**
     * Test the connection to the web app.
     */
    suspend fun testWebAppConnection(url: String, user: String, pass: String): Pair<Boolean, String>

    /**
     * Import items from the web app for a specific list.
     */
    suspend fun importItemsFromWebApp(listId: Int, url: String, user: String, pass: String, webAppListId: Int): Int

    /**
     * Delete marked items in the web app.
     */
    suspend fun deleteMarkedItemsInWebApp(url: String, user: String, pass: String, webAppListId: Int, deletePass: String?)

    /**
     * Mark an item in the web app.
     */
    suspend fun markItemInWebApp(url: String, user: String, pass: String, webAppListId: Int, itemName: String)

}

class OfflineNoteshopRepository(
    private val noteDao: NoteDao,
    private val shoppingListDao: ShoppingListDao,
    private val recipeDao: RecipeDao,
    private val webAppClient: WebAppClient,
    private val context: android.content.Context
) : NoteshopRepository {
    override fun getAllNotesStream(): Flow<List<Note>> = noteDao.getAllNotes()

    override fun getNoteStream(id: Int): Flow<Note?> = noteDao.getNote(id)

    override suspend fun insertNote(note: Note) = noteDao.insert(note)

    override suspend fun upsertNote(note: Note) {
        if (note.id == 0) {
            noteDao.insert(note)
        } else {
            noteDao.update(note)
        }
    }

    override suspend fun updateNote(note: Note) = noteDao.update(note)

    override suspend fun deleteNote(note: Note) = noteDao.delete(note)

    override suspend fun getNotesCount(): Int = noteDao.getNotesCount()

    override fun getAllRecipesStream(): Flow<List<Recipe>> = recipeDao.getAllRecipes()

    override fun getRecipeStream(id: Int): Flow<Recipe?> = recipeDao.getRecipe(id)

    override suspend fun insertRecipe(recipe: Recipe) = recipeDao.insert(recipe)

    override suspend fun upsertRecipe(recipe: Recipe) {
        if (recipe.id == 0) {
            recipeDao.insert(recipe)
        } else {
            recipeDao.update(recipe)
        }
    }

    override suspend fun updateRecipe(recipe: Recipe) = recipeDao.update(recipe)

    override suspend fun deleteRecipe(recipe: Recipe) = recipeDao.delete(recipe)

    override suspend fun getRecipesCount(): Int = recipeDao.getRecipesCount()

    override fun getShoppingListByName(name: String): Flow<ShoppingList?> = shoppingListDao.getShoppingListByName(name)

    override fun getAllShoppingListsWithItemsStream(): Flow<List<ShoppingListWithItems>> = shoppingListDao.getListsWithItems()

    override fun getShoppingListWithItemsStream(listId: Int): Flow<ShoppingListWithItems?> = shoppingListDao.getListWithItems(listId)

    override suspend fun insertShoppingList(list: ShoppingList): Long = shoppingListDao.insertList(list)

    override suspend fun upsertShoppingList(list: ShoppingList) {
        if (list.id == 0) {
            shoppingListDao.insertList(list)
        } else {
            shoppingListDao.updateList(list)
        }
    }

    override suspend fun updateShoppingList(list: ShoppingList): Int = shoppingListDao.updateList(list)

    override suspend fun updateShoppingLists(lists: List<ShoppingList>) = shoppingListDao.updateLists(lists)

    override suspend fun deleteShoppingList(list: ShoppingList) = shoppingListDao.deleteList(list)

    override suspend fun insertShoppingListItem(item: ShoppingListItem) = shoppingListDao.insertItem(item)

    override suspend fun upsertShoppingListItem(item: ShoppingListItem) {
        if (item.id == 0) {
            shoppingListDao.insertItem(item)
        } else {
            shoppingListDao.updateItem(item)
        }
    }

    override suspend fun insertShoppingListItems(items: List<ShoppingListItem>) = shoppingListDao.insertItems(items)

    override suspend fun updateShoppingListItem(item: ShoppingListItem) = shoppingListDao.updateItem(item)

    override suspend fun deleteShoppingListItem(item: ShoppingListItem) = shoppingListDao.deleteItem(item)

    override suspend fun deleteShoppingListItems(items: List<ShoppingListItem>) = shoppingListDao.deleteItems(items)

    override suspend fun getListsCount(): Int = shoppingListDao.getListsCount()

    override suspend fun getItemsCount(listId: Int): Int = shoppingListDao.getItemsCount(listId)

    override suspend fun updateShoppingListItems(items: List<ShoppingListItem>) = shoppingListDao.updateItems(items)

    override fun getStandardListItems(): List<String> {
        return context.resources.getStringArray(de.lxtools.noteshop.R.array.standard_list_items).toList()
    }

    override suspend fun testWebAppConnection(url: String, user: String, pass: String): Pair<Boolean, String> {
        return webAppClient.testConnection(url, user, pass)
    }

    override suspend fun importItemsFromWebApp(listId: Int, url: String, user: String, pass: String, webAppListId: Int): Int {
        val itemsFromWeb = webAppClient.fetchItems(url, user, pass, webAppListId)
        // Get existing items to prevent duplicates
        val existingItemNames = getShoppingListWithItemsStream(listId).firstOrNull()?.items?.map { it.name.lowercase() } ?: emptyList()
        val newItems = itemsFromWeb.filter { item ->
            item.name.lowercase() !in existingItemNames
        }.map { item ->
            ShoppingListItem(listId = listId, name = item.name, isChecked = (item.isMarked == 1))
        }

        if (newItems.isNotEmpty()) {
            insertShoppingListItems(newItems)
        }
        return newItems.size
    }

    override suspend fun deleteMarkedItemsInWebApp(url: String, user: String, pass: String, webAppListId: Int, deletePass: String?) {
        webAppClient.deleteMarkedItems(url, user, pass, webAppListId, deletePass)
    }

    override suspend fun markItemInWebApp(url: String, user: String, pass: String, webAppListId: Int, itemName: String) {
        webAppClient.markItems(url, user, pass, webAppListId, listOf(itemName))
    }

}
