package de.lxtools.noteshop.data

import android.content.Context
import android.net.Uri
import androidx.room.Database
import androidx.room.Room
import androidx.room.RoomDatabase
import androidx.room.migration.Migration
import androidx.sqlite.db.SupportSQLiteDatabase
import kotlinx.coroutines.withContext
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.first

@Database(
    entities = [Note::class, ShoppingList::class, ShoppingListItem::class, Recipe::class],
    version = 10, // Incremented version
    exportSchema = false
)
abstract class AppDatabase : RoomDatabase() {

    abstract fun noteDao(): NoteDao
    abstract fun shoppingListDao(): ShoppingListDao
    abstract fun recipeDao(): RecipeDao

    companion object {
        @Volatile
        private var INSTANCE: AppDatabase? = null
        @Volatile
        private var currentDbUri: Uri? = null

        fun getDatabase(context: Context, dbUri: Uri): AppDatabase {
            return INSTANCE?.takeIf { currentDbUri == dbUri } ?: synchronized(this) {
                INSTANCE?.takeIf { currentDbUri == dbUri } ?: buildDatabase(context, dbUri).also {
                    INSTANCE = it
                    currentDbUri = dbUri
                }
            }
        }

        fun closeDatabase() {
            INSTANCE?.let {
                if (it.isOpen) {
                    it.close()
                }
                INSTANCE = null
                currentDbUri = null
            }
        }

        private fun buildDatabase(context: Context, dbUri: Uri): AppDatabase {
            val internalDbName = "internal-noteshop.db"
            val internalDbFile = context.getDatabasePath(internalDbName)

            // If the internal database file doesn't exist, copy from the external URI.
            // This ensures that on first launch with an existing file, we use it.
            if (!internalDbFile.exists()) {
                try {
                    context.contentResolver.openInputStream(dbUri)?.use { inputStream ->
                        // Ensure parent directory exists
                        internalDbFile.parentFile?.mkdirs()
                        internalDbFile.outputStream().use { outputStream ->
                            inputStream.copyTo(outputStream)
                        }
                    }
                } catch (_: Exception) {
                    // This can happen if the URI points to a new, non-existent file.
                    // In this case, Room will create an empty DB, which is the desired behavior.
                }
            }

            return Room.databaseBuilder(
                context.applicationContext,
                AppDatabase::class.java,
                internalDbName
            )
            .setJournalMode(JournalMode.TRUNCATE) // Use TRUNCATE for safe file copy
            .addMigrations(MIGRATION_1_2, MIGRATION_2_3, MIGRATION_3_4, MIGRATION_4_5, MIGRATION_5_6, MIGRATION_6_7, MIGRATION_7_8, MIGRATION_8_9, MIGRATION_9_10)
            .build()
        }

        private val MIGRATION_1_2 = object : Migration(1, 2) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE shopping_list_items ADD COLUMN quantity TEXT")
            }
        }

        private val MIGRATION_2_3 = object : Migration(2, 3) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE shopping_lists ADD COLUMN displayOrder INTEGER NOT NULL DEFAULT 0")
            }
        }

        private val MIGRATION_3_4 = object : Migration(3, 4) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE shopping_list_items ADD COLUMN displayOrder INTEGER NOT NULL DEFAULT 0")
            }
        }

        private val MIGRATION_4_5 = object : Migration(4, 5) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE notes ADD COLUMN displayOrder INTEGER NOT NULL DEFAULT 0")
            }
        }

        private val MIGRATION_5_6 = object : Migration(5, 6) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE notes ADD COLUMN isLocked INTEGER NOT NULL DEFAULT 0")
                db.execSQL("ALTER TABLE shopping_lists ADD COLUMN isLocked INTEGER NOT NULL DEFAULT 0")
            }
        }

        private val MIGRATION_6_7 = object : Migration(6, 7) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("CREATE TABLE IF NOT EXISTS `recipes` (`id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL, `title` TEXT NOT NULL, `content` TEXT NOT NULL, `displayOrder` INTEGER NOT NULL, `isLocked` INTEGER NOT NULL)")
            }
        }

        private val MIGRATION_7_8 = object : Migration(7, 8) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE shopping_lists ADD COLUMN encryptedItems TEXT")
            }
        }

        private val MIGRATION_8_9 = object : Migration(8, 9) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE notes ADD COLUMN protectionHash TEXT NOT NULL DEFAULT ''")
                db.execSQL("ALTER TABLE notes ADD COLUMN protectionType INTEGER NOT NULL DEFAULT 0")
                db.execSQL("ALTER TABLE recipes ADD COLUMN protectionHash TEXT NOT NULL DEFAULT ''")
                db.execSQL("ALTER TABLE recipes ADD COLUMN protectionType INTEGER NOT NULL DEFAULT 0")
                db.execSQL("ALTER TABLE shopping_lists ADD COLUMN protectionHash TEXT NOT NULL DEFAULT ''")
                db.execSQL("ALTER TABLE shopping_lists ADD COLUMN protectionType INTEGER NOT NULL DEFAULT 0")
            }
        }

        private val MIGRATION_9_10 = object : Migration(9, 10) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE notes ADD COLUMN lockMethod INTEGER NOT NULL DEFAULT 0")
                db.execSQL("ALTER TABLE recipes ADD COLUMN lockMethod INTEGER NOT NULL DEFAULT 0")
                db.execSQL("ALTER TABLE shopping_lists ADD COLUMN lockMethod INTEGER NOT NULL DEFAULT 0")
            }
        }
    }
}
