package de.lxtools.noteshop.api

import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.engine.cio.CIO
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.request.forms.submitForm
import io.ktor.client.request.get
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.client.statement.bodyAsText
import io.ktor.http.ContentType
import io.ktor.http.Parameters
import io.ktor.http.contentType
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json

@Serializable
data class WebAppShoppingList(
    val id: Int,
    val name: String
)

@Serializable
data class TokenResponse(
    @SerialName("access_token") val accessToken: String,
    @SerialName("token_type") val tokenType: String
)

@Serializable
data class Item(
    val id: Int,
    val name: String,
    @SerialName("is_standard") val isStandard: Int,
    @SerialName("created_by_user_id") val createdByUserId: Int?,
    @SerialName("marked") val isMarked: Int
)

@Serializable
data class ItemsResponse(
    val items: List<Item>
)

@Serializable
data class DeletionRequest(
    val list_id: Int,
    val password: String? = null
)

@Serializable
data class ErrorDetail(
    val detail: String
)

class WebAppClient {

    private val client = HttpClient(CIO) {
        install(ContentNegotiation) {
            json(Json {
                ignoreUnknownKeys = true
                prettyPrint = true
                isLenient = true
            })
        }
        expectSuccess = false // Don't throw exceptions for non-2xx responses
    }

    private var token: String? = null

    suspend fun testConnection(url: String, user: String, pass: String): Pair<Boolean, String> {
        try {
            val response = client.submitForm(
                url = "$url/token",
                formParameters = Parameters.build {
                    append("username", user)
                    append("password", pass)
                }
            )

            if (response.status.value in 200..299) {
                val tokenResponse: TokenResponse = response.body()
                token = tokenResponse.accessToken
                return Pair(true, "")
            } else {
                val errorBody = response.bodyAsText()
                return try {
                    val errorDetail = Json.decodeFromString<ErrorDetail>(errorBody)
                    Pair(false, "Login failed: ${errorDetail.detail}")
                } catch (_: Exception) {
                    Pair(false, "Login failed with status: ${response.status.value}. Response: $errorBody")
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
            return Pair(false, "An unexpected error occurred: ${e.message}")
        }
    }

    suspend fun fetchWebAppShoppingLists(url: String, user: String, pass: String): List<WebAppShoppingList> {
        if (token == null) {
            val (success, _) = testConnection(url, user, pass)
            if (!success) {
                return emptyList()
            }
        }

        return try {
            val response: List<WebAppShoppingList> = client.get("$url/api/lists") {
                header("Authorization", "Bearer $token")
            }.body()
            response
        } catch (e: Exception) {
            e.printStackTrace()
            emptyList()
        }
    }

    suspend fun fetchItems(url: String, user: String, pass: String, webAppListId: Int): List<Item> {
        if (token == null) {
            val (success, _) = testConnection(url, user, pass)
            if (!success) {
                return emptyList()
            }
        }

        return try {
            val response: ItemsResponse = client.get("$url/api/items") {
                header("Authorization", "Bearer $token")
                url {
                    parameters.append("list_id", webAppListId.toString())
                }
            }.body()
            response.items
        } catch (e: Exception) {
            e.printStackTrace()
            emptyList()
        }
    }

    suspend fun markItems(url: String, user: String, pass: String, webAppListId: Int, itemsToMark: List<String>) {
        if (token == null) {
            val (success, _) = testConnection(url, user, pass)
            if (!success) {
                return
            }
        }

        try {
            // First, get all items for the specific list to find their IDs
            val allItemsResponse: ItemsResponse = client.get("$url/api/items") {
                header("Authorization", "Bearer $token")
                url {
                    parameters.append("list_id", webAppListId.toString())
                }
            }.body()

            val itemsToMarkIds = allItemsResponse.items
                .filter { it.name in itemsToMark }
                .map { it.id }

            for (itemId in itemsToMarkIds) {
                client.put("$url/api/items/$itemId/mark") {
                    header("Authorization", "Bearer $token")
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    suspend fun deleteMarkedItems(url: String, user: String, pass: String, webAppListId: Int, deletePass: String?) {
        if (token == null) {
            val (success, _) = testConnection(url, user, pass)
            if (!success) {
                return
            }
        }

        try {
            client.post("$url/api/items/delete-marked") {
                header("Authorization", "Bearer $token")
                contentType(ContentType.Application.Json)
                setBody(DeletionRequest(list_id = webAppListId, password = deletePass))
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }
}
