package de.lxtools.noteshop

import android.app.Application
import android.content.Context
import android.net.Uri
import android.util.Log
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.preferencesDataStore
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.ProcessLifecycleOwner
import de.lxtools.noteshop.data.AppContainer
import de.lxtools.noteshop.data.SettingsRepository
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch

/**
 * The application class, which holds the dependency container and manages app-level lifecycle events.
 */
class NoteshopApplication : Application(), LifecycleEventObserver {
    /**
     * AppContainer instance used by the rest of classes to obtain dependencies.
     * It's nullable because it's initialized asynchronously.
     */
    var container: AppContainer? = null

    private val applicationScope = CoroutineScope(SupervisorJob() + Dispatchers.IO)

    override fun onCreate() {
        super.onCreate()
        ProcessLifecycleOwner.get().lifecycle.addObserver(this)
    }

    override fun onStateChanged(source: LifecycleOwner, event: Lifecycle.Event) {
        if (event == Lifecycle.Event.ON_STOP) {
            writeDatabaseToExternalUri()
        }
    }

    private fun writeDatabaseToExternalUri() {
        val settingsRepository = SettingsRepository(this)
        val dbUriString = settingsRepository.databaseUri.value ?: return
        val dbUri = Uri.parse(dbUriString)
        val internalDbName = "internal-noteshop.db"
        val internalDbFile = getDatabasePath(internalDbName)

        if (!internalDbFile.exists()) {
            Log.w("NoteshopApplication", "Internal database file does not exist, skipping write-back.")
            return
        }

        applicationScope.launch {
            try {
                // The database connection is NOT closed here.
                // With TRUNCATE journal mode, the main DB file is safe to be copied.
                contentResolver.openOutputStream(dbUri, "w")?.use { outputStream ->
                    internalDbFile.inputStream().use { inputStream ->
                        inputStream.copyTo(outputStream)
                    }
                }
                Log.d("NoteshopApplication", "Successfully wrote database back to external URI.")
            } catch (e: Exception) {
                Log.e("NoteshopApplication", "Error writing database to external URI", e)
            }
        }
    }
}

private val Context.preferencesDataStore: DataStore<Preferences> by preferencesDataStore(name = "noteshop_encrypted_prefs")