package de.lxtools.noteshop

import android.app.Application
import android.content.Intent
import android.net.Uri
import android.util.Log
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.viewModelScope
import de.lxtools.noteshop.data.AppDatabase
import de.lxtools.noteshop.data.AppContainer
import de.lxtools.noteshop.data.AppDataContainer
import de.lxtools.noteshop.data.SettingsRepository
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class MainActivityViewModel(application: Application) : AndroidViewModel(application) {

    var appState by mutableStateOf(AppState.LOADING)
        private set

    var container: AppContainer? by mutableStateOf(null)
        private set

    var showDatabaseRestartHint by mutableStateOf(false)
        private set

    internal val settingsRepository = SettingsRepository(application)

    val databasePath: StateFlow<String?> = settingsRepository.databaseUri.map { uriString ->
        uriString?.let {
            val uri = Uri.parse(it)
            val context = getApplication<Application>()
            context.contentResolver.query(uri, arrayOf(android.provider.OpenableColumns.DISPLAY_NAME), null, null, null)?.use { cursor ->
                if (cursor.moveToFirst()) {
                    cursor.getString(cursor.getColumnIndexOrThrow(android.provider.OpenableColumns.DISPLAY_NAME))
                } else {
                    null
                }
            }
        }
    }.stateIn(viewModelScope, SharingStarted.WhileSubscribed(5000), null)


    fun checkDatabaseState() {
        viewModelScope.launch(Dispatchers.IO) {
            val dbUriString = settingsRepository.databaseUri.value
            if (dbUriString == null) {
                setState(AppState.SETUP_DATABASE)
            } else {
                var fileExistsAndHasPermission = false
                val dbUri = Uri.parse(dbUriString)
                try {
                    val hasPermission = getApplication<Application>().contentResolver.persistedUriPermissions.any { it.uri == dbUri && it.isReadPermission }
                    if (hasPermission) {
                        getApplication<Application>().contentResolver.openInputStream(dbUri)?.use {
                            fileExistsAndHasPermission = true
                        }
                    }
                } catch (e: Exception) {
                    Log.w("MainActivityViewModel", "Failed to verify database URI", e)
                    fileExistsAndHasPermission = false
                }

                if (fileExistsAndHasPermission) {
                    initializeContainer(dbUri)
                    setState(AppState.APP_READY)
                } else {
                    resetDatabaseConfig()
                    setState(AppState.SETUP_DATABASE)
                }
            }
        }
    }

    fun onDatabaseSelected(uri: Uri) {
        viewModelScope.launch(Dispatchers.IO) {
            setState(AppState.SWITCHING_DATABASE)

            // Close existing database
            AppDatabase.closeDatabase()

            val context = getApplication<Application>()
            
            // Persist permissions
            val takeFlags = Intent.FLAG_GRANT_READ_URI_PERMISSION or Intent.FLAG_GRANT_WRITE_URI_PERMISSION
            context.contentResolver.takePersistableUriPermission(uri, takeFlags)
            settingsRepository.setDatabaseUri(uri.toString())

            // Delete old internal DB to ensure a fresh copy
            val internalDbName = "internal-noteshop.db"
            context.getDatabasePath(internalDbName).takeIf { it.exists() }?.delete()

            // Re-initialize
            initializeContainer(uri)
            setState(AppState.APP_READY)
            showDatabaseRestartHint = true
        }
    }

    fun dismissDatabaseRestartHint() {
        showDatabaseRestartHint = false
    }

    private fun initializeContainer(dbUri: Uri) {
        val appContainer = AppDataContainer(getApplication(), dbUri)
        container = appContainer
        (getApplication() as NoteshopApplication).container = appContainer
    }

    private suspend fun resetDatabaseConfig() {
        settingsRepository.setDatabaseUri(null)
        val internalDbName = "internal-noteshop.db"
        getApplication<Application>().getDatabasePath(internalDbName).takeIf { it.exists() }?.delete()
    }

    private suspend fun setState(state: AppState) {
        withContext(Dispatchers.Main) {
            appState = state
        }
    }
}
