package de.lxtools.noteshop

import android.app.Activity
import android.content.Context
import android.content.ContextWrapper
import android.net.Uri
import android.os.Bundle
import android.os.Parcelable
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.contract.ActivityResultContracts
import androidx.activity.viewModels
import androidx.biometric.BiometricManager
import androidx.biometric.BiometricPrompt
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.core.content.ContextCompat
import androidx.core.content.edit
import androidx.documentfile.provider.DocumentFile
import androidx.fragment.app.FragmentActivity
import de.lxtools.noteshop.ui.AppShell
import de.lxtools.noteshop.ui.DatabaseSetupScreen
import de.lxtools.noteshop.ui.LockScreen
import de.lxtools.noteshop.ui.theme.ColorTheme
import de.lxtools.noteshop.ui.theme.NoteshopTheme
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Text
import androidx.compose.material3.Button
import androidx.compose.ui.res.stringResource
import kotlinx.parcelize.Parcelize

// Data class to hold the dynamic strings
data class DynamicListStrings(
    val itemName: String,
    val addItem: String,
    val editItem: String,
    val itemChecked: String
)

// Helper function to get the correct set of strings
@Composable
fun getDynamicStrings(customTitle: String): DynamicListStrings {
    val defaultShoppingListTitle = stringResource(R.string.menu_shopping_lists)
    return if (customTitle.equals(defaultShoppingListTitle, ignoreCase = true)) {
        DynamicListStrings(
            itemName = stringResource(R.string.item_name),
            addItem = stringResource(R.string.add_item),
            editItem = stringResource(R.string.edit_item),
            itemChecked = stringResource(R.string.items_checked)
        )
    } else {
        DynamicListStrings(
            itemName = stringResource(R.string.task_name),
            addItem = stringResource(R.string.add_task),
            editItem = stringResource(R.string.edit_task),
            itemChecked = stringResource(R.string.task_checked)
        )
    }
}

// Data class to hold the dynamic strings for recipes
data class DynamicRecipeStrings(
    val searchHint: String,
    val addItem: String,
    val editItem: String,
    val itemTitle: String,
    val lockItem: String,
    val unlockItem: String,
    val importButton: String
)

// Helper function to get the correct set of strings for recipes
@Composable
fun getDynamicRecipeStrings(customTitle: String): DynamicRecipeStrings {
    val defaultRecipeTitle = stringResource(R.string.menu_recipes)
    return if (customTitle.equals(defaultRecipeTitle, ignoreCase = true)) {
        DynamicRecipeStrings(
            searchHint = stringResource(R.string.search_recipes_hint),
            addItem = stringResource(R.string.add_recipe),
            editItem = stringResource(R.string.edit_recipe),
            itemTitle = stringResource(R.string.recipe_title),
            lockItem = stringResource(R.string.lock_recipe),
            unlockItem = stringResource(R.string.unlock_recipe),
            importButton = stringResource(R.string.import_from_md_txt)
        )
    } else {
        DynamicRecipeStrings(
            searchHint = stringResource(R.string.search_notes_hint),
            addItem = stringResource(R.string.add_note),
            editItem = stringResource(R.string.edit_note),
            itemTitle = stringResource(R.string.note_title),
            lockItem = stringResource(R.string.lock_note),
            unlockItem = stringResource(R.string.unlock_note),
            importButton = stringResource(R.string.import_from_md_txt)
        )
    }
}

// Helper function to find the activity from a context
fun Context.findActivity(): Activity {
    var context = this
    while (context is ContextWrapper) {
        if (context is Activity) return context
        context = context.baseContext
    }
    throw IllegalStateException("no activity")
}

class BiometricAuthenticator(private val context: Context) {

    private lateinit var promptInfo: BiometricPrompt.PromptInfo



    fun promptBiometricAuth(
        title: String,
        subtitle: String,
        fragmentActivity: FragmentActivity,
        onSuccess: (BiometricPrompt.AuthenticationResult) -> Unit,
        onFailed: () -> Unit,
        onError: (Int, CharSequence) -> Unit
    ) {
        promptInfo = BiometricPrompt.PromptInfo.Builder()
            .setTitle(title)
            .setSubtitle(subtitle)
            .setAllowedAuthenticators(BiometricManager.Authenticators.BIOMETRIC_STRONG or BiometricManager.Authenticators.DEVICE_CREDENTIAL)
            .build()

        val executor = ContextCompat.getMainExecutor(context)

        val biometricPrompt = BiometricPrompt(fragmentActivity, executor, object : BiometricPrompt.AuthenticationCallback() {
            override fun onAuthenticationSucceeded(result: BiometricPrompt.AuthenticationResult) {
                super.onAuthenticationSucceeded(result)
                onSuccess(result)
            }

            override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                super.onAuthenticationError(errorCode, errString)
                onError(errorCode, errString)
            }

            override fun onAuthenticationFailed() {
                super.onAuthenticationFailed()
                onFailed()
            }
        })
        biometricPrompt.authenticate(promptInfo)
    }

    fun promptBiometricAuth(
        title: String,
        subtitle: String,
        fragmentActivity: FragmentActivity,
        crypto: BiometricPrompt.CryptoObject,
        onSuccess: (BiometricPrompt.AuthenticationResult) -> Unit,
        onFailed: () -> Unit,
        onError: (Int, CharSequence) -> Unit
    ) {
        promptInfo = BiometricPrompt.PromptInfo.Builder()
            .setTitle(title)
            .setSubtitle(subtitle)
            .setAllowedAuthenticators(BiometricManager.Authenticators.BIOMETRIC_STRONG or BiometricManager.Authenticators.DEVICE_CREDENTIAL)
            .build()

        val executor = ContextCompat.getMainExecutor(context)

        val biometricPrompt = BiometricPrompt(fragmentActivity, executor, object : BiometricPrompt.AuthenticationCallback() {
            override fun onAuthenticationSucceeded(result: BiometricPrompt.AuthenticationResult) {
                super.onAuthenticationSucceeded(result)
                onSuccess(result)
            }

            override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                super.onAuthenticationError(errorCode, errString)
                onError(errorCode, errString)
            }

            override fun onAuthenticationFailed() {
                super.onAuthenticationFailed()
                onFailed()
            }
        })
        biometricPrompt.authenticate(promptInfo, crypto)
    }
}

// Sealed class to represent the screens in the app
@Parcelize
sealed class Screen(val route: String, val titleRes: Int) : Parcelable {
    data object ShoppingLists : Screen("shopping_lists", R.string.menu_shopping_lists)
    data object Notes : Screen("notes", R.string.menu_notes)
    data object Recipes : Screen("recipes", R.string.menu_recipes)
    data object ShoppingListDetail : Screen("shopping_list_detail", R.string.menu_shopping_list_detail)
    data object NoteDetail : Screen("note_detail", R.string.menu_note_detail)
    data object RecipeDetail : Screen("recipe_detail", R.string.menu_recipe_detail)
    data object About : Screen("about", R.string.menu_about)
    data object Settings : Screen("settings", R.string.menu_settings)
    data object WebAppIntegration : Screen("webapp_integration", R.string.webapp_integration_title)
    data object GuidedTour : Screen("guided_tour", R.string.show_guided_tour)
}

// Function to get DocumentFile for a given URI and filename
fun getDocumentFile(context: Context, uri: Uri, fileName: String, mimeType: String): DocumentFile? {
    val documentFile = DocumentFile.fromTreeUri(context, uri)
    return documentFile?.findFile(fileName) ?: documentFile?.createFile(mimeType, fileName)
}

enum class AppState { LOADING, SETUP_DATABASE, APP_READY, SWITCHING_DATABASE }

class MainActivity : FragmentActivity() {

    private val viewModel: MainActivityViewModel by viewModels()

    private val openDatabaseLauncher = registerForActivityResult(ActivityResultContracts.OpenDocument()) { uri ->
        uri?.let { viewModel.onDatabaseSelected(it) }
    }

    private val createDatabaseLauncher = registerForActivityResult(ActivityResultContracts.CreateDocument("application/x-sqlite3")) { uri ->
        uri?.let { viewModel.onDatabaseSelected(it) }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()

        setContent {
            LaunchedEffect(Unit) {
                if (viewModel.appState == AppState.LOADING) {
                    viewModel.checkDatabaseState()
                }
            }

            val context = LocalContext.current
            val sharedPrefs = remember { context.getSharedPreferences("app_prefs", MODE_PRIVATE) }
            val themeName = sharedPrefs.getString("color_theme", ColorTheme.STANDARD.name) ?: ColorTheme.STANDARD.name
            var colorTheme by remember { mutableStateOf(ColorTheme.valueOf(themeName)) }

            NoteshopTheme(colorTheme = colorTheme) {
                when (viewModel.appState) {
                    AppState.LOADING, AppState.SWITCHING_DATABASE -> {
                        Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
                            CircularProgressIndicator()
                        }
                    }
                    AppState.SETUP_DATABASE -> {
                        DatabaseSetupScreen(
                            onCreateNewDatabase = { createDatabaseLauncher.launch("noteshop.db") },
                            onOpenExistingDatabase = { openDatabaseLauncher.launch(arrayOf("application/x-sqlite3", "application/octet-stream")) }
                        )
                    }
                    AppState.APP_READY -> {
                        val currentContainer = viewModel.container // Get the current container from ViewModel
                        if (currentContainer != null) { // Ensure container is not null
                            var isAppLockEnabled by remember { mutableStateOf(sharedPrefs.getBoolean("app_lock_enabled", false)) }
                            var isUnlocked by remember { mutableStateOf(!isAppLockEnabled) }

                            val biometricAuthenticator = BiometricAuthenticator(this)
                            val biometricManager = BiometricManager.from(context)
                            val canUseBiometrics = remember {
                                biometricManager.canAuthenticate(BiometricManager.Authenticators.BIOMETRIC_STRONG) == BiometricManager.BIOMETRIC_SUCCESS
                            }

                            if (isUnlocked) {
                                AppShell(
                                    container = currentContainer,
                                    mainActivityViewModel = viewModel,
                                    colorTheme = colorTheme,
                                    onThemeChange = {
                                        sharedPrefs.edit {
                                            putString("color_theme", it.name)
                                        }
                                        colorTheme = it
                                    },
                                    isAppLockEnabled = isAppLockEnabled,
                                    onAppLockChange = { enabled ->
                                        sharedPrefs.edit { putBoolean("app_lock_enabled", enabled) }
                                        isAppLockEnabled = enabled
                                        if (!enabled) {
                                            isUnlocked = true
                                        }
                                    },
                                    sharedPrefs = sharedPrefs,
                                    biometricAuthenticator = biometricAuthenticator,
                                    canUseBiometrics = canUseBiometrics,
                                    createDatabaseLauncher = createDatabaseLauncher,
                                    openDatabaseLauncher = openDatabaseLauncher
                                )
                            } else {
                                LockScreen(onUnlockClick = {
                                    biometricAuthenticator.promptBiometricAuth(
                                        title = getString(R.string.unlock_app),
                                        subtitle = getString(R.string.confirm_to_unlock),
                                        fragmentActivity = this,
                                        onSuccess = { _ -> isUnlocked = true },
                                        onFailed = {
                                            android.widget.Toast.makeText(context, R.string.unlock_failed, android.widget.Toast.LENGTH_SHORT).show()
                                        },
                                        onError = { _, _ ->
                                            android.widget.Toast.makeText(context, R.string.unlock_failed, android.widget.Toast.LENGTH_SHORT).show()
                                        }
                                    )
                                })
                            }
                        }
                    }
                }

                if (viewModel.showDatabaseRestartHint) {
                    AlertDialog(
                        onDismissRequest = { viewModel.dismissDatabaseRestartHint() },
                        title = { Text(stringResource(R.string.database_changed_title)) },
                        text = { Text(stringResource(R.string.database_changed_message)) },
                        confirmButton = {
                            Button(onClick = { finish() }) {
                                Text(stringResource(R.string.close_app))
                            }
                        }
                    )
                }
            }
        }
    }

    // The write-back logic is now handled in NoteshopApplication based on process lifecycle
}