/*
 * Copyright (C) 2020-21 Andreas Kromke, andreas.kromke@gmail.com
 *
 * This program is free software; you can redistribute it or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

package de.kromke.andreas.audiotags;

// -> https://xiph.org/vorbis/doc/framing.html
// -> https://xiph.org/vorbis/doc/v-comment.html
// -> https://www.ietf.org/rfc/rfc3533.txt

// Note that Ogg is a container format encapsulating e.g. Vorbis-encoded audio and Vorbis comments.


import android.util.Log;
import java.io.InputStream;

@SuppressWarnings({"SameParameterValue"})
public class TagsOggVorbis extends TagsOgg
{
    private static final String LOG_TAG = "TOGV";
    int audioSampleRate = -1;


    private class Vorbis
    {
        @SuppressWarnings("FieldCanBeLocal")
        int packetType;
        int offset = 0;         // offset in packet data

        public boolean check()
        {
            packetType = data[0] & 0xff;
            Log.d(LOG_TAG, "Vorbis::check() : packet type is " + packetType);

            if ((data[1] != 'v') || (data[2] != 'o') || (data[3] != 'r') || (data[4] != 'b') || (data[5] != 'i') || (data[6] != 's'))
            {
                Log.e(LOG_TAG, "Vorbis::check() : missing vorbis signature");
                return false;
            }

            offset = 7;
            return true;
        }

        boolean getIdentificationHeader()
        {
            int version = getLittleEndianInt4(data, offset);
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : version = " + version);

            int audioChannels = data[offset + 4];
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : number of audio channels = " + audioChannels);

            audioSampleRate = getLittleEndianInt4(data, offset + 5);
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : audio sample rate = " + audioSampleRate);

            int bitrateMaximum = getLittleEndianInt4(data, offset + 9);
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : maximum bitrate = " + bitrateMaximum);
            int bitrateNominal = getLittleEndianInt4(data, offset + 13);
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : nominal bitrate = " + bitrateNominal);
            int bitrateMinimum = getLittleEndianInt4(data, offset + 17);
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : minimum bitrate = " + bitrateMinimum);

            int blocksize = data[offset + 21] & 0xff;
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : blocksize0 = " + (blocksize & 0x0f));
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : blocksize1 = " + (blocksize >> 4));

            int framing = data[offset + 22];
            Log.d(LOG_TAG, "Vorbis::readIdentificationHeader() : framing = " + framing);
            offset += 23;

            return true;
        }

        boolean getComment()
        {
            InputBuffer buf = new InputBuffer(data, offset);
            //noinspection RedundantIfStatement
            if (!readVorbisComment(buf))
            {
                return false;
            }

            return true;
        }
    }

    public TagsOggVorbis(InputStream is)
    {
        super(is);
        tagType = 11;            // Vorbis
    }


    public boolean read()
    {
        // read Ogg packet, consisting of pages, to memory, i.e. data[]

        if (!readPacket())
        {
            return false;
        }

        //
        // read first Vorbis packet inside the Ogg packet, must have type 1
        //

        Vorbis vorbis = new Vorbis();
        if (!vorbis.check())
        {
            return false;
        }
        if (vorbis.packetType == 1)
        {
            if (!vorbis.getIdentificationHeader())
            {
                return false;
            }
        }
        else
        {
            Log.e(LOG_TAG, "TagsVorbis::read() : unhandled packet type " + vorbis.packetType);
            return false;
        }

        //
        // read second Ogg packet
        //

        if (!readPacket())
        {
            return false;
        }

        //
        // read second Vorbis packet inside the Ogg packet, must have type 3
        //

        vorbis = new Vorbis();
        if (!vorbis.check())
        {
            return false;
        }
        if (vorbis.packetType == 3)
        {
            if (!vorbis.getComment())
            {
                return false;
            }
        }
        else
        {
            Log.e(LOG_TAG, "TagsVorbis::read() : unhandled packet type " + vorbis.packetType);
            return false;
        }

        //
        // The recommended method to retrieve the file duration is to seek near the end of the file,
        // scan for an Ogg page header and then read the last "absolute granule position".
        // As we cannot seek here, due to the lack of random access, we scan the whole file.
        // Note that our Ogg implementation currently cannot re-sync in case the file is corrupted.
        //

        if (doGetDuration)
        {
            // must read the whole file to get the timestamps
            int nPackets = 0;
            while (readPacket())
            {
                nPackets++;
            }
            Log.d(LOG_TAG, "TagsVorbis::read() : " + nPackets + " packet read");
            if ((absGranPos > 0) && (audioSampleRate > 0))
            {
                durationUnits = (int) ((absGranPos * 1000) / audioSampleRate);
                bitRate = audioSampleRate;
            }
        }

        return true;
    }
}
