package de.kitshn.ui.component.model.ingredient

import androidx.compose.foundation.layout.BoxWithConstraints
import androidx.compose.foundation.layout.Column
import androidx.compose.material3.ListItemColors
import androidx.compose.material3.ListItemDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.text.rememberTextMeasurer
import androidx.compose.ui.unit.coerceAtMost
import androidx.compose.ui.unit.dp
import de.kitshn.api.tandoor.model.TandoorFoodRecipe
import de.kitshn.api.tandoor.model.TandoorIngredient
import de.kitshn.ui.state.ErrorLoadingSuccessState

@Composable
fun IngredientsList(
    list: List<TandoorIngredient>,

    itemModifier: (ingredient: TandoorIngredient) -> Modifier = { Modifier },
    itemTrailingContent: @Composable (ingredient: TandoorIngredient) -> Unit = {},

    factor: Double = 1.0,
    loadingState: ErrorLoadingSuccessState = ErrorLoadingSuccessState.SUCCESS,
    colors: ListItemColors = ListItemDefaults.colors(),

    showFractionalValues: Boolean,

    onNotEnoughSpace: () -> Unit,
    onOpenRecipe: (recipe: TandoorFoodRecipe) -> Unit
) {
    val density = LocalDensity.current

    var minAmountWidth by remember { mutableStateOf(0.dp) }
    var minUnitWidth by remember { mutableStateOf(0.dp) }

    val textMeasure = rememberTextMeasurer()
    LaunchedEffect(list, list.size, factor) {
        minAmountWidth = 0.dp
        minUnitWidth = 0.dp

        list.forEach { ingredient ->
            val amount = ingredient.amount * factor

            if(!ingredient.no_amount && ingredient.amount > 0.0) {
                val widthDp =
                    with(density) {
                        textMeasure.measure(
                            ingredient.formatAmount(
                                amount,
                                showFractionalValues
                            )
                        ).size.width.toDp()
                    }
                if(widthDp > minAmountWidth) minAmountWidth = widthDp
            }

            if(!ingredient.no_amount && ingredient.unit != null) {
                val widthDp =
                    with(density) { textMeasure.measure(ingredient.getUnitLabel(amount)).size.width.toDp() }
                if(widthDp > minUnitWidth) minUnitWidth = widthDp
            }
        }

        minAmountWidth *= 1.3f
        minUnitWidth *= 1.3f

        minAmountWidth += 8.dp
        minUnitWidth += 8.dp
    }

    BoxWithConstraints {
        // try to gain more space when amount and unit fields are to wide
        if(maxWidth - ((minAmountWidth + minUnitWidth).coerceAtMost(maxWidth / 2.2f)) < 100.dp) onNotEnoughSpace()
        val maxWidth = maxWidth

        Column {
            if(loadingState == ErrorLoadingSuccessState.LOADING) {
                repeat(10) {
                    IngredientItem(
                        ingredient = null,
                        servingsFactor = factor,

                        colors = colors,
                        position = when(it) {
                            0 -> IngredientItemPosition.TOP
                            9 -> IngredientItemPosition.BOTTOM
                            else -> IngredientItemPosition.BETWEEN
                        },

                        maxWidth = maxWidth,
                        minAmountWidth = minAmountWidth,
                        minUnitWidth = minUnitWidth,

                        showFractionalValues = showFractionalValues,

                        loadingState = loadingState,

                        onOpenRecipe = onOpenRecipe
                    )
                }
            } else {
                repeat(list.size) {
                    val ingredient = list[it]
                    IngredientItem(
                        modifier = itemModifier(ingredient),
                        trailingContent = { itemTrailingContent(ingredient) },

                        ingredient = ingredient,
                        servingsFactor = factor,

                        colors = colors,
                        position = when(it) {
                            0 -> if(list.size > 1) IngredientItemPosition.TOP else IngredientItemPosition.SINGULAR
                            list.size - 1 -> IngredientItemPosition.BOTTOM
                            else -> IngredientItemPosition.BETWEEN
                        },

                        maxWidth = maxWidth,
                        minAmountWidth = minAmountWidth,
                        minUnitWidth = minUnitWidth,

                        showFractionalValues = showFractionalValues,

                        loadingState = loadingState,

                        onOpenRecipe = onOpenRecipe
                    )
                }
            }
        }
    }
}