package de.kitshn.ui.route.main.subroute.home

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.grid.GridCells
import androidx.compose.foundation.lazy.grid.GridItemSpan
import androidx.compose.foundation.lazy.grid.LazyVerticalGrid
import androidx.compose.foundation.lazy.grid.rememberLazyGridState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ListItem
import androidx.compose.material3.ListItemDefaults
import androidx.compose.material3.SearchBarScrollBehavior
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import de.kitshn.api.tandoor.TandoorRequestState
import de.kitshn.api.tandoor.TandoorRequestStateState
import de.kitshn.api.tandoor.route.TandoorRecipeQueryParameters
import de.kitshn.api.tandoor.route.TandoorRecipeQueryParametersSortOrder
import de.kitshn.handleTandoorRequestState
import de.kitshn.isScrollingUp
import de.kitshn.reachedBottom
import de.kitshn.ui.component.LoadingGradientWrapper
import de.kitshn.ui.component.alert.LoadingErrorAlertPaneWrapper
import de.kitshn.ui.component.loading.LazyGridAnimatedContainedLoadingIndicator
import de.kitshn.ui.component.model.recipe.HorizontalRecipeCardLink
import de.kitshn.ui.component.model.recipe.RecipeCard
import de.kitshn.ui.modifier.loadingPlaceHolder
import de.kitshn.ui.route.RouteParameters
import de.kitshn.ui.selectionMode.SelectionModeState
import de.kitshn.ui.state.ErrorLoadingSuccessState
import de.kitshn.ui.state.foreverRememberMutableStateList
import de.kitshn.ui.state.foreverRememberNotSavable
import de.kitshn.ui.state.rememberErrorLoadingSuccessState
import de.kitshn.ui.theme.Typography
import de.kitshn.ui.theme.playfairDisplay
import de.kitshn.ui.view.home.search.HOME_SEARCH_PAGING_SIZE
import de.kitshn.ui.view.home.search.HomeSearchState
import de.kitshn.ui.view.recipe.details.RecipeServingsAmountSaveMap
import kitshn.composeapp.generated.resources.Res
import kitshn.composeapp.generated.resources.lorem_ipsum_medium
import kitshn.composeapp.generated.resources.lorem_ipsum_title
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import org.jetbrains.compose.resources.stringResource

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun HomeTraditionalLayout(
    p: RouteParameters,

    searchBarScrollBehavior: SearchBarScrollBehavior,
    selectionModeState: SelectionModeState<Int>,
    homeSearchState: HomeSearchState,

    onIsScrollingUpChanged: (isScrollingUp: Boolean) -> Unit,

    wrap: @Composable (
        @Composable (
            pv: PaddingValues,
            supportsMultiplePages: Boolean,
            background: Color,

            onSelect: (String) -> Unit
        ) -> Unit
    ) -> Unit
) {
    val coroutineScope = rememberCoroutineScope()
    val hapticFeedback = LocalHapticFeedback.current

    var pageLoadingState by rememberErrorLoadingSuccessState()

    val listRequestState = remember { TandoorRequestState() }
    val extendedListRequestState = remember { TandoorRequestState() }

    val resultIds = foreverRememberMutableStateList<Int>(key = "RouteMainSubrouteList/resultIds")
    var currentPage by foreverRememberNotSavable(
        key = "RouteMainSubrouteList/currentPage",
        initialValue = 1
    )
    var nextPageExists by foreverRememberNotSavable(
        key = "RouteMainSubrouteList/nextPageExists",
        initialValue = true
    )

    val homeScreenSorting by p.vm.settings.getHomeScreenSorting.collectAsState(initial = "")
    var queryParameters by remember { mutableStateOf(TandoorRecipeQueryParameters(new = true)) }

    LaunchedEffect(homeScreenSorting) {
        resultIds.clear()
        pageLoadingState = ErrorLoadingSuccessState.LOADING

        queryParameters = try {
            TandoorRecipeQueryParameters(
                sortOrder = TandoorRecipeQueryParametersSortOrder.valueOf(homeScreenSorting)
            )
        } catch(e: Exception) {
            TandoorRecipeQueryParameters(
                new = true
            )
        }
    }

    LaunchedEffect(p.vm.tandoorClient, homeScreenSorting) {
        if(p.vm.tandoorClient == null) return@LaunchedEffect
        delay(100)

        if(resultIds.size > 0) {
            pageLoadingState = ErrorLoadingSuccessState.SUCCESS
            return@LaunchedEffect
        }

        currentPage = 1

        listRequestState.wrapRequest {
            p.vm.tandoorClient!!.recipe.list(
                parameters = queryParameters,
                pageSize = HOME_SEARCH_PAGING_SIZE,
                page = currentPage
            )
        }?.let {
            currentPage++

            nextPageExists = it.next != null

            resultIds.clear()
            it.results.forEach { recipe -> resultIds.add(recipe.id) }

            pageLoadingState = ErrorLoadingSuccessState.SUCCESS
        }

        if(listRequestState.state == TandoorRequestStateState.ERROR)
            pageLoadingState = ErrorLoadingSuccessState.ERROR
    }

    val gridState = rememberLazyGridState()
    val reachedBottom by remember { derivedStateOf { gridState.reachedBottom() } }

    var fetchNewItems by remember { mutableStateOf(false) }
    LaunchedEffect(reachedBottom) {
        if(reachedBottom) {
            fetchNewItems = true
        }
    }

    LaunchedEffect(fetchNewItems, nextPageExists) {
        while(fetchNewItems && nextPageExists) {
            if(listRequestState.state != TandoorRequestStateState.SUCCESS) {
                delay(500)
                continue
            }

            extendedListRequestState.wrapRequest {
                p.vm.tandoorClient!!.recipe.list(
                    parameters = queryParameters,
                    pageSize = HOME_SEARCH_PAGING_SIZE,
                    page = currentPage
                )
            }?.let {
                currentPage++

                nextPageExists = it.next != null
                it.results.forEach { recipe -> resultIds.add(recipe.id) }

                if(!reachedBottom) fetchNewItems = false
            }

            hapticFeedback.handleTandoorRequestState(extendedListRequestState)

            delay(500)
        }
    }

    val isScrollingUp = gridState.isScrollingUp()
    LaunchedEffect(isScrollingUp) { onIsScrollingUpChanged(isScrollingUp) }

    val enableMealPlanPromotion by p.vm.settings.getEnableMealPlanPromotion.collectAsState(initial = true)
    val promoteTomorrowsMealPlan by p.vm.settings.getPromoteTomorrowsMealPlan.collectAsState(initial = false)

    val enableCompactHomeScreen by p.vm.settings.getEnableCompactHomeScreen.collectAsState(initial = false)

    wrap { pv, _, background, onSelect ->
        LoadingErrorAlertPaneWrapper(loadingState = pageLoadingState) {
            LoadingGradientWrapper(
                Modifier
                    .padding(pv)
                    .nestedScroll(searchBarScrollBehavior.nestedScrollConnection),
                loadingState = pageLoadingState,
                backgroundColor = background
            ) {
                LazyVerticalGrid(
                    modifier = Modifier
                        .nestedScroll(searchBarScrollBehavior.nestedScrollConnection),

                    state = gridState,

                    contentPadding = PaddingValues(16.dp),
                    userScrollEnabled = pageLoadingState != ErrorLoadingSuccessState.LOADING,

                    columns = GridCells.Adaptive(
                        minSize =
                            when(enableCompactHomeScreen) {
                                true -> 300.dp
                                false -> 250.dp
                            }
                    ),

                    verticalArrangement = Arrangement.spacedBy(8.dp),
                    horizontalArrangement = Arrangement.spacedBy(8.dp),
                ) {
                    item(
                        span = {
                            GridItemSpan(maxCurrentLineSpan)
                        }
                    ) {
                        if(enableMealPlanPromotion) p.vm.tandoorClient?.let {
                            Spacer(Modifier.height(16.dp))

                            RouteMainSubrouteHomeMealPlanPromotionSection(
                                client = it,
                                titlePadding = PaddingValues(),
                                contentPadding = PaddingValues(top = 16.dp),
                                loadingState = pageLoadingState,
                                day = if(promoteTomorrowsMealPlan) {
                                    MealPlanPromotionSectionDay.TOMORROW
                                } else {
                                    MealPlanPromotionSectionDay.TODAY
                                }
                            ) { recipeOverview, servings ->
                                RecipeServingsAmountSaveMap[recipeOverview.id] = servings
                                onSelect(recipeOverview.id.toString())
                            }
                        }
                    }

                    if(resultIds.size == 0 && pageLoadingState != ErrorLoadingSuccessState.SUCCESS) {
                        items(20) {
                            when(enableCompactHomeScreen) {
                                true -> {
                                    Card(
                                        onClick = { }
                                    ) {
                                        ListItem(
                                            colors = ListItemDefaults.colors(
                                                containerColor = CardDefaults.cardColors().containerColor
                                            ),
                                            leadingContent = {
                                                Box(
                                                    modifier = Modifier
                                                        .size(48.dp)
                                                        .clip(RoundedCornerShape(8.dp))
                                                        .loadingPlaceHolder()
                                                )
                                            },
                                            headlineContent = {
                                                Column {
                                                    Text(
                                                        modifier = Modifier
                                                            .loadingPlaceHolder(),
                                                        text = stringResource(Res.string.lorem_ipsum_title),
                                                        style = Typography().bodyLarge.copy(
                                                            fontFamily = playfairDisplay()
                                                        ),
                                                        maxLines = 1,
                                                        overflow = TextOverflow.Ellipsis
                                                    )

                                                    Spacer(Modifier.height(2.dp))
                                                }
                                            },
                                            supportingContent = {
                                                Column {
                                                    Spacer(Modifier.height(2.dp))

                                                    Text(
                                                        modifier = Modifier
                                                            .loadingPlaceHolder(),
                                                        text = stringResource(Res.string.lorem_ipsum_medium),
                                                        maxLines = 1,
                                                        overflow = TextOverflow.Ellipsis
                                                    )
                                                }
                                            }
                                        )
                                    }
                                }

                                false -> {
                                    Box {
                                        RecipeCard(
                                            Modifier.height(300.dp)
                                                .fillMaxWidth(),
                                            alternateCoverSize = true,
                                            loadingState = pageLoadingState,
                                            onClickKeyword = { }
                                        ) { }
                                    }
                                }
                            }
                        }
                    } else {
                        items(resultIds.size) {
                            val recipeOverview =
                                p.vm.tandoorClient?.container?.recipeOverview?.get(resultIds[it])
                                    ?: return@items

                            when(enableCompactHomeScreen) {
                                true -> {
                                    HorizontalRecipeCardLink(
                                        recipeOverview = recipeOverview,
                                        selectionState = selectionModeState,
                                        defaultColors = ListItemDefaults.colors(
                                            containerColor = CardDefaults.cardColors().containerColor
                                        )
                                    ) { recipe -> onSelect(recipe.id.toString()) }
                                }

                                false -> Box {
                                    RecipeCard(
                                        Modifier.height(300.dp)
                                            .fillMaxWidth(),
                                        alternateCoverSize = true,
                                        fillChipRow = true,
                                        recipeOverview = recipeOverview,
                                        loadingState = pageLoadingState,
                                        selectionState = selectionModeState,
                                        onClickKeyword = {
                                            coroutineScope.launch {
                                                homeSearchState.openWithKeyword(
                                                    p.vm.tandoorClient!!,
                                                    it
                                                )
                                            }
                                        },
                                        onClick = { recipe -> onSelect(recipe.id.toString()) }
                                    )
                                }
                            }
                        }
                    }

                    LazyGridAnimatedContainedLoadingIndicator(
                        nextPageExists = nextPageExists,
                        extendedRequestState = extendedListRequestState
                    )
                }
            }
        }
    }
}