package de.katzenmue.birdipedia

import android.content.SharedPreferences
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.MenuAnchorType
import androidx.compose.material3.Switch
import androidx.compose.material3.SwitchDefaults
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import androidx.core.content.edit
import de.katzenmue.birdipedia.ui.theme.ThemeMode

@Composable
fun IUCNToggle(
    showIUCN: MutableState<Boolean>,
    sharedPreferences: SharedPreferences,
    onIUCNChange: (Boolean) -> Unit
) {
    Row(modifier = drawerItemSpacing, verticalAlignment = Alignment.CenterVertically) {
        if (showIUCN.value) {
            Text("IUCN enabled")
        } else {
            Text("IUCN disabled")
        }
        Spacer(modifier = Modifier.padding(start = 8.dp))
        Switch(
            checked = showIUCN.value,
            onCheckedChange = {
                showIUCN.value = it
//                        sharedPreferences.edit().putBoolean("showIUCN", it).apply()
//                        sharedPreferences.edit().putBoolean("showIUCN", it)
                sharedPreferences.edit { putBoolean("showIUCN", it) }
                onIUCNChange(showIUCN.value)
            },
            colors = SwitchDefaults.colors(
                checkedThumbColor = MaterialTheme.colorScheme.primary,
                checkedTrackColor = MaterialTheme.colorScheme.inversePrimary,
//                            checkedTrackColor = DarkGray,
                uncheckedThumbColor = MaterialTheme.colorScheme.secondary,
                uncheckedTrackColor = MaterialTheme.colorScheme.onSecondary,
            )
        )
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun RegionToggle(
    currentRegion: String,
    sharedPreferences: SharedPreferences,
    onRegionChange: (String) -> Unit
) {
    var selectedIndex by remember { mutableStateOf(0) }
    LaunchedEffect(currentRegion) {
        selectedIndex = when (currentRegion) {
            "NULL" -> 0
            "NAM" -> 1
            "SAM" -> 2
            "CAM" -> 3
            "EUR" -> 4
            "AFR" -> 5
            "AFC" -> 6
            "OCE" -> 7
            "ASI" -> 8
            "AUS" -> 9
            else -> 9
        }
    }
    val options = listOf(
        "World",
        "North America",
        "South America",
        "Central America",
        "Europe",
        "Africa",
        "Continental Africa",
        "Oceania",
        "Asia",
        "Australasia"
    )
    val optionsCode =
        listOf("NULL", "NAM", "SAM", "CAM", "EUR", "AFR", "AFC", "OCE", "ASI", "AUS") // FIXME, dupes
    var expandedRegion by remember { mutableStateOf(false) }

    Row(
//        modifier = Modifier.fillMaxWidth(), horizontalArrangement = rowArrangement, verticalAlignment = Alignment.CenterVertically
        modifier = drawerItemSpacing, verticalAlignment = Alignment.CenterVertically

    ) {
        Text("Region (by continent):")
        Spacer(modifier = Modifier.padding(start = 10.dp))
        ExposedDropdownMenuBox(
            expanded = expandedRegion,
            onExpandedChange = {
                expandedRegion = !expandedRegion
            },
            modifier = Modifier.width(200.dp)
        ) {
            TextField(
                readOnly = true,
                value = options[selectedIndex],
                onValueChange = { },
//            label = { Text("Select region (continent)") },
                trailingIcon = {
                    ExposedDropdownMenuDefaults.TrailingIcon(
                        expanded = expandedRegion
                    )
                },
                colors = ExposedDropdownMenuDefaults.textFieldColors(),
//            modifier = Modifier.clickable { expandedRegion = true }
                modifier = Modifier.menuAnchor(MenuAnchorType.PrimaryNotEditable),
            )
            ExposedDropdownMenu(
                expanded = expandedRegion,
                onDismissRequest = {
                    expandedRegion = false
                },
            ) {
                options.forEachIndexed { index, label ->
                    DropdownMenuItem(
                        onClick = {
                            selectedIndex = index
                            val newRegion = optionsCode[selectedIndex]
                            sharedPreferences.edit { putString("region", newRegion) }
                            onRegionChange(newRegion)
                            expandedRegion = false
                        },
                        text = { Text(text = label) }
                    )
                }
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ThemeToggle(
    currentThemeMode: ThemeMode,
    sharedPreferences: SharedPreferences,
    onThemeChange: (ThemeMode) -> Unit
) {
    var selectedIndex = when (currentThemeMode) {
        ThemeMode.Light -> 0
        ThemeMode.Dark -> 1
        ThemeMode.System -> 2
        ThemeMode.LightDyn -> 3
        ThemeMode.DarkDyn -> 4
        ThemeMode.SystemDyn -> 5
    }
    val options = listOf("Warbler Yellow", "Woodpecker Red", "Follow System", "Light (dynamic)", "Dark (dynamic)", "Follow System (dynamic)")
    var expandedRegion by remember { mutableStateOf(false) }

    Row(
//        modifier = Modifier.fillMaxWidth(), horizontalArrangement = rowArrangement, verticalAlignment = Alignment.CenterVertically
        modifier = drawerItemSpacing, verticalAlignment = Alignment.CenterVertically

    ) {
        Text("Theme:")
        Spacer(modifier = Modifier.padding(start = 10.dp))
        ExposedDropdownMenuBox(
            expanded = expandedRegion,
            onExpandedChange = {
                expandedRegion = !expandedRegion
            },
            modifier = Modifier.width(200.dp)
        ) {
            TextField(
                readOnly = true,
                value = options[selectedIndex],
                onValueChange = { },
//            label = { Text("Select region (continent)") },
                trailingIcon = {
                    ExposedDropdownMenuDefaults.TrailingIcon(
                        expanded = expandedRegion
                    )
                },
                colors = ExposedDropdownMenuDefaults.textFieldColors(),
//            modifier = Modifier.clickable { expandedRegion = true }
                modifier = Modifier.menuAnchor(MenuAnchorType.PrimaryNotEditable),
            )
            ExposedDropdownMenu(
                expanded = expandedRegion,
                onDismissRequest = {
                    expandedRegion = false
                },
            ) {
                options.forEachIndexed { index, label ->
                    DropdownMenuItem(
                        onClick = {
                            selectedIndex = index
//                            val newThemeMode = ThemeMode.valueOf(label)
                            val newThemeMode = ThemeMode.valueOf(ThemeMode.entries[selectedIndex].toString())
//                            Log.d("theme", "${ThemeMode.entries}")
//                            Log.d("theme", "${newThemeMode}")
                            onThemeChange(newThemeMode)
                            sharedPreferences.edit { putString("themeMode", newThemeMode.name) }
                            expandedRegion = false
                        },
                        text = { Text(text = label) }
                    )
                }
            }
        }
    }
}
