package de.katzenmue.birdipedia

import android.content.Context
import android.content.Intent
import android.content.SharedPreferences
import android.content.pm.PackageManager
import android.net.Uri
import androidx.browser.customtabs.CustomTabsIntent
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Info
import androidx.compose.material.icons.filled.Menu
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.DrawerState
import androidx.compose.material3.DrawerValue
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalDrawerSheet
import androidx.compose.material3.ModalNavigationDrawer
import androidx.compose.material3.NavigationDrawerItem
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberDrawerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.unit.dp
import androidx.core.net.toUri
import androidx.lifecycle.ViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.navigation.NavHostController
import coil3.compose.AsyncImage
import coil3.request.ImageRequest
import coil3.request.crossfade
import coil3.size.Scale
import de.katzenmue.birdipedia.ui.theme.BirdipediaTheme
import de.katzenmue.birdipedia.ui.theme.ThemeMode
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch


@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun Credits(navController: NavHostController) {
    // FIXME
    val sharedPreferences =
        LocalContext.current.getSharedPreferences("app_prefs", Context.MODE_PRIVATE)
    val storedTheme = sharedPreferences.getString("themeMode", "System") ?: "System"
    var themeMode by remember { mutableStateOf(ThemeMode.valueOf(storedTheme)) }
    val showIUCN = remember { mutableStateOf(sharedPreferences.getBoolean("showIUCN", false)) }

    BirdipediaTheme(themeMode = themeMode) {
        Scaffold(
            topBar = {
                CenterAlignedTopAppBar(
                    colors = TopAppBarDefaults.topAppBarColors(
                        containerColor = MaterialTheme.colorScheme.primaryContainer,
//                        titleContentColor = MaterialTheme.colorScheme.primary,
                    ),
                    navigationIcon = {
                        IconButton(onClick = {
                            navController.navigate(Routes.Home.route)
                        }) {
                            Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = "Back")
                        }
                    },
                    title = {
                        Text("Credits")
                    },
                )
            }, modifier = Modifier.fillMaxSize()
        ) { innerPadding ->
            Column(
                modifier = Modifier
                    .padding(innerPadding)
                    .fillMaxWidth(),
                horizontalAlignment = Alignment.CenterHorizontally,
            ) {
                val context = LocalContext.current

                Spacer(modifier = Modifier.height(12.dp))
                Image(
                    painterResource(id = R.drawable.ic_launcher_round),
                    contentDescription = null,
                    modifier = Modifier
                        .size(128.dp)
                        .clip(CircleShape),
                    contentScale = ContentScale.Crop,
                )
                Spacer(modifier = Modifier.height(12.dp))
                Text(
                    "Birdipedia", fontSize = MaterialTheme.typography.titleLarge.fontSize
                )
                Spacer(modifier = Modifier.height(12.dp))
                var versionName = "1.0.0"
                try {
                    versionName = context.packageManager
                        .getPackageInfo(context.packageName, 0).versionName.toString()
                } catch (e: PackageManager.NameNotFoundException) {
//                    e.printStackTrace()
                }
                Text(
                    "v$versionName", fontSize = MaterialTheme.typography.titleMedium.fontSize
                )
                Spacer(modifier = Modifier.height(12.dp))
                Button(
                    colors = ButtonDefaults.buttonColors(
                        containerColor = MaterialTheme.colorScheme.background,
                        contentColor = MaterialTheme.colorScheme.onBackground
                    ),
                    modifier = Modifier
                        .border(1.dp, Color.DarkGray, CircleShape)
                        .clip(CircleShape),
                    onClick = {
                        val intent = Intent(
                            Intent.ACTION_VIEW,
                            ("https://codeberg.org/JetpackJackson/Birdipedia").toUri()
                        )
                        context.startActivity(intent)
                    }) {
                    Text("View source on Codeberg")
                }
                Spacer(modifier = Modifier.height(12.dp))
                Button(
                    colors = ButtonDefaults.buttonColors(
                        containerColor = MaterialTheme.colorScheme.background,
                        contentColor = MaterialTheme.colorScheme.onBackground
                    ),
                    modifier = Modifier
                        .border(1.dp, Color.DarkGray, CircleShape)
                        .clip(CircleShape),
                    onClick = {
                        val intent = Intent(
                            Intent.ACTION_VIEW, ("https://developer.android.com/compose").toUri()
                        )
                        context.startActivity(intent)
                    }) {
                    Text("Written in Jetpack Compose")
                }
                Spacer(modifier = Modifier.height(12.dp))
                HorizontalDivider()
                Spacer(modifier = Modifier.height(12.dp))
                Button(
                    colors = ButtonDefaults.buttonColors(
                        containerColor = MaterialTheme.colorScheme.background,
                        contentColor = MaterialTheme.colorScheme.onBackground
                    ),
                    modifier = Modifier
                        .border(1.dp, Color.DarkGray, CircleShape)
                        .clip(CircleShape),
                    onClick = {
                        val intent = Intent(
                            Intent.ACTION_VIEW, ("https://avibase.bsc-eoc.org").toUri()
                        )
                        context.startActivity(intent)
                    }) {
                    Text("Region Data from Avibase")
                }
                Spacer(modifier = Modifier.height(12.dp))
                Button(
                    colors = ButtonDefaults.buttonColors(
                        containerColor = MaterialTheme.colorScheme.background,
                        contentColor = MaterialTheme.colorScheme.onBackground
                    ),
                    modifier = Modifier
                        .border(1.dp, Color.DarkGray, CircleShape)
                        .clip(CircleShape),
                    onClick = {
                        val intent = Intent(
                            Intent.ACTION_VIEW, ("https://www.avilist.org/checklist/v2025").toUri()
                        )
                        context.startActivity(intent)
                    }) {
                    Text("Species Data from AviList")
                }
                Spacer(modifier = Modifier.height(12.dp))
                Button(
                    colors = ButtonDefaults.buttonColors(
                        containerColor = MaterialTheme.colorScheme.background,
                        contentColor = MaterialTheme.colorScheme.onBackground
                    ),
                    modifier = Modifier
                        .border(1.dp, Color.DarkGray, CircleShape)
                        .clip(CircleShape),
                    onClick = {
                        val intent = Intent(
                            Intent.ACTION_VIEW, ("https://www.macaulaylibrary.org").toUri()
                        )
                        context.startActivity(intent)
                    }) {
                    Text("Preview Images from the Macaulay Library")

                }
                Spacer(modifier = Modifier.height(12.dp))
                HorizontalDivider()
                Text("Data Citations:", modifier = Modifier.padding(10.dp))
//                Spacer(modifier = Modifier.height(12.dp))
                Text(
                    "AviList Core Team. 2025. AviList: The Global Avian Checklist, v2025. https://doi.org/10.2173/avilist.v2025",
                    modifier = Modifier.padding(10.dp)
                )
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun Settings(
    navController: NavHostController,
    drawerState: DrawerState,
    viewModel: FilterBirdViewModel,
    currentFilterText: MutableState<String>
) {
    // FIXME
    val sharedPreferences =
        LocalContext.current.getSharedPreferences("app_prefs", Context.MODE_PRIVATE)
    val storedTheme = sharedPreferences.getString("themeMode", "System") ?: "System"
    var themeMode by remember { mutableStateOf(ThemeMode.valueOf(storedTheme)) }
    val storedRegion = sharedPreferences.getString("region", "NULL") ?: "NULL"
    var region by remember { mutableStateOf(storedRegion) }

    val showIUCN = remember { mutableStateOf(sharedPreferences.getBoolean("showIUCN", false)) }

    BirdipediaTheme(themeMode = themeMode) {
        Scaffold(
            topBar = {
                CenterAlignedTopAppBar(
                    colors = TopAppBarDefaults.topAppBarColors(
                        containerColor = MaterialTheme.colorScheme.primaryContainer,
//                        titleContentColor = MaterialTheme.colorScheme.onPrimary,
                    ),
                    navigationIcon = {
                        IconButton(onClick = {
                            navController.navigate(Routes.Home.route)
                        }) {
                            Icon(
                                Icons.AutoMirrored.Filled.ArrowBack, contentDescription = "Back"
                            )
                        }
                    },
                    title = {
                        Text("Settings")
                    },
                )
            }, modifier = Modifier.fillMaxSize()
        ) { innerPadding ->
            Column(modifier = Modifier.padding(innerPadding)) {
                IUCNToggle(
                    showIUCN,
                    sharedPreferences,
                    onIUCNChange = { showIUCN.value != showIUCN.value })
                RegionToggle(
                    region,
                    // currentRegion = region.value ?: "NULL",
                    sharedPreferences = sharedPreferences, onRegionChange = { newRegion ->
                        region = newRegion
                        // selectedRegion.value = newRegion
                        viewModel.setRegion(newRegion, currentFilterText)
                    })
                ThemeToggle(
                    currentThemeMode = themeMode, sharedPreferences
                ) { newMode -> themeMode = newMode }
//                RegionToggle(
//                    currentRegion = selectedRegion.toString(), sharedPreferences
//                ) {}

            }
//            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun Home(
    navController: NavHostController,
    viewModel: FilterBirdViewModel,
    eBirdList: List<String>,
    labelList: List<String>,
    sciList: List<String>,
    assetList: List<String>,
    orderList: List<String>,
    regionList: List<String>,
    IUCNList: List<String>,
    currentFilterText: MutableState<String>
) {
    val scope = rememberCoroutineScope()
    val drawerState = rememberDrawerState(initialValue = DrawerValue.Closed)

    var text = currentFilterText.value

    val sharedPreferences =
        LocalContext.current.getSharedPreferences("app_prefs", Context.MODE_PRIVATE)
    val storedTheme = sharedPreferences.getString("themeMode", "System") ?: "System"
    var themeMode by remember { mutableStateOf(ThemeMode.valueOf(storedTheme)) }
    var region = sharedPreferences.getString("region", "NULL") ?: "NULL"

    val showIUCN = remember { mutableStateOf(sharedPreferences.getBoolean("showIUCN", false)) }

//    LaunchedEffect(Unit) {
//        showIUCN.value = sharedPreferences.getBoolean("showIUCN", false)
//    }

    BirdipediaTheme(themeMode = themeMode) {
        ModalNavigationDrawer(
            drawerState = drawerState, drawerContent = {
                ModalDrawerSheet {
                    Column(
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Column(
                            modifier = Modifier.padding(horizontal = 16.dp),
                            horizontalAlignment = Alignment.Start
                        ) {
                            Image(
                                painter = painterResource(id = R.drawable.ic_launcher_round),
                                contentDescription = null,
                                modifier = Modifier
                                    .size(100.dp)
                                    .clip(CircleShape)
                                    .align(Alignment.Start),
                                contentScale = ContentScale.Crop,
                            )
                            Spacer(modifier = Modifier.height(12.dp))
                            Text(
                                "Birdipedia",
                                fontSize = MaterialTheme.typography.titleLarge.fontSize,
                                modifier = Modifier.fillMaxWidth()
                            )
                        }
                        Spacer(modifier = Modifier.height(12.dp))
                        HorizontalDivider()
                        NavigationDrawerItem(
                            label = { Text(text = "Settings") },
                            icon = { Icon(Icons.Default.Settings, contentDescription = null) },
                            selected = false,
                            onClick = { navController.navigate(Routes.Settings.route) })
                        NavigationDrawerItem(
                            label = { Text(text = "Credits") },
                            icon = { Icon(Icons.Default.Info, contentDescription = null) },
                            selected = false,
                            onClick = { navController.navigate(Routes.Credits.route) })
                    }
                }
            }) {
            Scaffold(
                topBar = {
                    CenterAlignedTopAppBar(
                        colors = TopAppBarDefaults.topAppBarColors(
//                            containerColor = MaterialTheme.colorScheme.primary,
                            containerColor = MaterialTheme.colorScheme.primaryContainer,
//                            titleContentColor = MaterialTheme.colorScheme.onPrimary,
                        ),
                        navigationIcon = {
                            IconButton(onClick = {
                                scope.launch {
                                    if (drawerState.isClosed) {
                                        drawerState.open()
                                    } else {
                                        drawerState.close()
                                    }
                                }
                            }) {
                                Icon(Icons.Default.Menu, contentDescription = "Menu")
                            }
                        },
                        title = {
                            OutlinedTextField(
                                value = text,
                                onValueChange = {
                                    text = it
                                    viewModel.filterText(text)
                                    currentFilterText.value = text
                                },
                                label = { Text("Filter by Name") },
                                textStyle = MaterialTheme.typography.bodyLarge,
                                colors = OutlinedTextFieldDefaults.colors(
                                    focusedTextColor = MaterialTheme.colorScheme.onBackground,
                                    focusedBorderColor = MaterialTheme.colorScheme.onBackground,
                                    focusedLabelColor = MaterialTheme.colorScheme.onBackground,
//                                    focusedPlaceholderColor = MaterialTheme.colorScheme.onBackground,
                                    cursorColor = MaterialTheme.colorScheme.onBackground,
                                ),
//                                colors = OutlinedTextFieldDefaults.colors().focusedLabelColor ,
                                modifier = Modifier.fillMaxWidth(),
                            )
                        },
                    )
                }, modifier = Modifier.fillMaxSize()
            ) { innerPadding ->
                Box(modifier = Modifier.fillMaxSize()) {
                    FilterBirdView(
                        viewModel,
                        labelList,
                        sciList,
                        eBirdList,
                        assetList,
                        orderList,
                        regionList,
                        IUCNList,
                        showIUCN,
                        innerPadding
                    )
                }
            }
        }
    }
}


@Composable
fun BirdBox(
    birdName: String,
    birdNameLatin: String,
    ebirdabbr: String,
    ebirdid: Int,
    showIUCN: MutableState<Boolean>,
    IUCNColor: Color,
    IUCNList: List<String>,
    index: Int,
) {
    val context = LocalContext.current
//    val IUCNColor = getIUCNColor(IUCNList[index])
//    Log.d("birdbox", "${IUCNColor}, ${showIUCN.value}")

    Surface(
        onClick = {
            val url = "https://ebird.org/species/$ebirdabbr"
            val builder = CustomTabsIntent.Builder()
            val customTabsIntent = builder.build()
            customTabsIntent.launchUrl(context, Uri.parse(url))
        }, modifier = birdBoxModifier
    ) {
        Box(modifier = Modifier.fillMaxSize()) {
            Row(
                modifier = Modifier, horizontalArrangement = rowArrangement
            ) {
//            if (showImages.value) {
                AsyncImage(
                    model = ImageRequest.Builder(LocalContext.current)
                        .data("https://cdn.download.ams.birds.cornell.edu/api/v1/asset/$ebirdid/")
                        .crossfade(true).scale(Scale.FIT)
                        // how small can we get without looking *too* pixelated?
                        .size(500, 500).build(),
                    contentDescription = "",
                    contentScale = ContentScale.Crop,
                    modifier = previewImageModifier(Modifier),
                )
                Column(
                    modifier = Modifier.align(Alignment.CenterVertically),
                    verticalArrangement = birdNameSpacing
                ) {
                    Text(
                        birdName, modifier = Modifier.fillMaxWidth()
                    )
                    Text(
                        birdNameLatin,
                        modifier = Modifier.fillMaxWidth(),
                        fontStyle = FontStyle.Italic,
                        color = Color.Gray
                    )
                }
            }
            if (showIUCN.value) {
                IUCNTriangle(IUCNList[index], IUCNColor)
            }
        }
    }
}

class FilterBirdViewModel(
    labelList: List<String>,
    eBirdList: List<String>,
    sciList: List<String>,
    assetList: List<String>,
    private val regionList: List<String>,
    IUCNList: List<String>,
    sharedPreferences: SharedPreferences,
    currentFilterText: MutableState<String>
) : ViewModel() {
    val items: List<String> = labelList

    private val _filteredItems = MutableStateFlow<List<String>>(items)
    var filteredItems: StateFlow<List<String>> = _filteredItems

    private val _filteredIndices = MutableStateFlow<List<Int>>(items.indices.toList())
    var filteredIndices: StateFlow<List<Int>> = _filteredIndices

    var selectedRegion: String

    init {
        selectedRegion = sharedPreferences.getString("region", "NULL") ?: "NULL"
        filterItems()
    }

    fun setRegion(region: String, currentFilterText: MutableState<String>) {
        selectedRegion = region
        filterItems()
        filterText(currentFilterText.value)
    }

    fun filterText(input: String) {
        val filtered = if (selectedRegion == "NULL") {
            items
        } else {
            items.filterIndexed { index, item ->
                val line = regionList[index]
                val parts = line.split("_")
                val regionCode = parts.getOrNull(2)
                val regionMatches = regionCode?.contains(selectedRegion)
                regionMatches == true
            }
        }
        if (selectedRegion != "NULL") {
            _filteredItems.value = filtered
            var input_parts = input.splitToSequence(" ").filter { it.isNotBlank() };
            _filteredIndices.value = items.indices.filter { index ->
                (regionList[index].split("_")[2].contains(selectedRegion)
//                        ) && input_parts.any { part -> items[index].contains(part, ignoreCase = true) }
                        ) && input_parts.all { part -> items[index].contains(part, ignoreCase = true) }
                        //&& items[index].contains(input, ignoreCase = true)
            }
        } else {
            _filteredItems.value = filtered
            var input_parts = input.splitToSequence(" ").filter { it.isNotBlank() };
            _filteredIndices.value = items.indices.filter { index ->
//                input_parts.any { part -> items[index].contains(part, ignoreCase = true) }
                input_parts.all { part -> items[index].contains(part, ignoreCase = true) }
//                items[index].contains(input, ignoreCase = true)
            }
        }
    }

    fun filterItems() {
        val filtered = if (selectedRegion == "NULL") {
            items
        } else {
            items.filterIndexed { index, item ->
                val line = regionList[index]
                val parts = line.split("_")
                val regionCode = parts.getOrNull(2)
                val regionMatches = regionCode?.contains(selectedRegion)
                regionMatches == true
            }
        }
        if (selectedRegion != "NULL") {
            _filteredItems.value = filtered
            _filteredIndices.value = items.indices.filter { index ->
                regionList[index].split("_")[2].contains(selectedRegion)
//                        || regionList[index].split("_").get(2) == "NULL"
                // || selectedRegion == "NULL"
            }
        } else {
            _filteredItems.value = filtered
        }
    }
}


@Composable
fun BirdFamilyHeader(initial: String, innerPadding: PaddingValues) {
    Text(
        text = initial,
        fontStyle = FontStyle.Italic,
        modifier = Modifier
            .alpha(0.8f)
            .background(MaterialTheme.colorScheme.background)
            .fillMaxWidth()
            .padding(10.dp)
            .wrapContentHeight(), // make text center vertical
    )
}


@Composable
fun FilterBirdView(
    viewModel: FilterBirdViewModel,
    labelList: List<String>,
    sciList: List<String>,
    eBirdList: List<String>,
    assetList: List<String>,
    orderList: List<String>,
    regionList: List<String>,
    IUCNList: List<String>,
    showIUCN: MutableState<Boolean>,
    innerPadding: PaddingValues,
) {
    val filteredIndices by viewModel.filteredIndices.collectAsStateWithLifecycle()
    val groupedItems = filteredIndices.groupBy {
        orderList[it]
    }
    LazyColumn(
        modifier = Modifier
            .padding(innerPadding)
            .fillMaxWidth(),
        verticalArrangement = columnArrangement
    ) {
        groupedItems.forEach { (order, indices) ->
            stickyHeader {
                BirdFamilyHeader(order.toString(), innerPadding)
            }
            items(indices.size) { index ->
                val originalIndex = indices[index]
                BirdBox(
                    // taxonList[originalIndex].split("_").get(1), // eng
                    labelList[originalIndex], // eng
                    // taxonList[originalIndex].split("_").get(0), // latin
                    sciList[originalIndex], // latin
                    eBirdList[originalIndex], // ebird abbr
                    if (assetList[originalIndex] == "NOASSET") {
                        0
                    } else {
                        assetList[originalIndex].toInt() // macaulay id
                    },
                    showIUCN,
                    getIUCNColor(IUCNList[originalIndex]),
                    IUCNList,
                    originalIndex
                )
            }
        }
    }
}
