/**
 * This file is part of Imagepipe.
 *
 * Copyright (c) 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024, 2025, 2026 Pawel Dube
 *
 * Imagepipe is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at
 * your option) any later version.
 *
 * Imagepipe is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Imagepipe. If not, see <http://www.gnu.org/licenses/>.
 */

package de.kaffeemitkoffein.imagepipe;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.os.PersistableBundle;
import android.text.Editable;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.TextWatcher;
import android.text.style.ForegroundColorSpan;
import android.util.TypedValue;
import android.view.View;
import android.widget.*;

public class ExifMenu extends Activity {

    private Context context;
    private ExifData exifData;
    private String[] allowedTags;

    @Override
    protected void onResume() {
        super.onResume();
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
    }

    @Override
    public void onSaveInstanceState(Bundle outState, PersistableBundle outPersistentState) {
        super.onSaveInstanceState(outState, outPersistentState);
    }

    @Override
    protected void onPause() {
        super.onPause();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
    }

    private boolean isAllowedTag(String tag){
        for (int i=0; i<allowedTags.length; i++){
            if (allowedTags[i].equalsIgnoreCase(tag)){
                return true;
            }
        }
        return false;
    }

    private int getTagPosition(String tag){
        for (int i=0; i<exifData.exifItems.size(); i++){
            if (exifData.exifItems.get(i).tag.equalsIgnoreCase(tag)){
                return i;
            }
        }
        return -1;
    }

    private void setTagValue(String tag, boolean state){
        int pos = getTagPosition(tag);
        if (pos>=0){
            exifData.exifItems.get(pos).isAllowed = state;
        }
    }

    private String generateAllowedString(){
        StringBuilder stringBuilder = new StringBuilder();
        for (int i=0; i<exifData.exifItems.size(); i++){
            ExifData.ExifItem exifItem = exifData.exifItems.get(i);
            if (exifItem.isAllowed){
                stringBuilder.append(exifItem.tag);
                stringBuilder.append(ImagepipePreferences.TAG_SEPERATOR);
            }
        }
        // remove last # if string not empty
        if (stringBuilder.length()>0){
            stringBuilder.setLength(stringBuilder.length()-1);
        }
        return stringBuilder.toString();
    }

    private void addCheckBox(LinearLayout parent, ExifData.ExifItem exifItem){
        CheckBox checkBox = new CheckBox(this);
        String s = exifItem.description;
        if (exifItem.present){
            int spanStart = s.length();
            s = s + " (" + exifItem.value+")";
            SpannableStringBuilder spannableStringBuilder = new SpannableStringBuilder(s);
            int color = getResources().getColor(R.color.primaryLightColor);
            spannableStringBuilder.setSpan(new ForegroundColorSpan(color),spanStart,s.length(), Spanned.SPAN_INCLUSIVE_INCLUSIVE);
            checkBox.setText(spannableStringBuilder);
        } else {
            checkBox.setText(s);
        }
        //checkBox.setText(exifItem.description);
        checkBox.setChecked(isAllowedTag(exifItem.tag));
        checkBox.setTag(exifItem);
        checkBox.setTextSize(TypedValue.COMPLEX_UNIT_SP,14);
        checkBox.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton compoundButton, boolean state) {
                ExifData.ExifItem exifItem = (ExifData.ExifItem) compoundButton.getTag();
                setTagValue(exifItem.tag,state);
                ImagepipePreferences.setPrefAllowedTags(context,generateAllowedString());
                ImagepipePreferences.setAllowedTagsModified(context);
                // disable temporary "do not show again" if new tags were added
                if (state){
                    ImagepipePreferences.setTemporarilyDisableWarning(context,false);
                }
                askForClearCache();
            }
        });
        int color = context.getResources().getColor(R.color.secondaryColor);
        if (exifItem.present){
            color = context.getResources().getColor(R.color.secondaryLightColor);
        }
        if (isAllowedTag(exifItem.tag)){
            checkBox.setChecked(true);
            exifItem.isAllowed = true;
        } else {
            checkBox.setChecked(false);
            exifItem.isAllowed = false;
        }
        checkBox.setTextColor(color);
        parent.addView(checkBox);
        /* EditText editText = new EditText(this);
        editText.setText(exifItem.value);
        editText.setTag(exifItem);
        editText.setSingleLine(true);
        editText.setTextSize(TypedValue.COMPLEX_UNIT_SP,12);
        parent.addView(editText);
        */
    }

    public void generateTagList(){
        LinearLayout linearLayout = (LinearLayout) findViewById(R.id.exifmenu_linearlayout);
        linearLayout.removeAllViews();
        for (int i=0; i<exifData.exifItems.size(); i++){
            addCheckBox(linearLayout,exifData.exifItems.get(i));
        }
    }

    public void loadAllowedTagsFromProfile(int profile){
        String[] data = ImagepipePreferences.getProfileDataArray(context,profile);
        if (data.length>0){
            allowedTags = data;
            Toast.makeText(context,getResources().getString(R.string.profile_loaded),Toast.LENGTH_SHORT).show();
            generateTagList();
        } else {
            Toast.makeText(context,getResources().getString(R.string.profile_empty),Toast.LENGTH_SHORT).show();
        }
    }

    public TextWatcher getTextWatcherForProfile(final int profile){
        return new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {
            }
            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
            }
            @Override
            public void afterTextChanged(Editable editable) {
                ImagepipePreferences.setProfileLabel(context,profile,editable.toString());
            }
        };
    }

    @Override
    protected void onCreate(Bundle bundle) {
        super.onCreate(bundle);
        context = getApplicationContext();
        setContentView(R.layout.activity_exifmenu);
        //exifData = new ExifData(context,null);
        exifData = ImagepipePreferences.readExifData(context);
        allowedTags = ImagepipePreferences.getAllowedTags(context);
        generateTagList();
        Button resetButton = (Button) findViewById(R.id.exifmenu_resetbutton);
        resetButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                ImagepipePreferences.setPrefAllowedTags(context,"");
                ImagepipePreferences.setAllowedTagsModified(context);
                recreate();
            }
        });
        // Profile 1
        Button profile1_load = (Button) findViewById(R.id.exifmenu_profile1_buttonload);
        profile1_load.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                loadAllowedTagsFromProfile(1);
            }
        });
        Button profile1_save = (Button) findViewById(R.id.exifmenu_profile1_buttonsave);
        profile1_save.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                ImagepipePreferences.setProfileData(context,1,generateAllowedString());
                Toast.makeText(context,getResources().getString(R.string.profile_saved),Toast.LENGTH_SHORT).show();
            }
        });
        EditText profile1_label = (EditText) findViewById(R.id.exifmenu_labelprofile1);
        String label1 = ImagepipePreferences.getProfileLabel(context,1);
        if (label1!=null){
            profile1_label.setText(label1);
        }
        profile1_label.addTextChangedListener(getTextWatcherForProfile(1));
        // Profile 2
        Button profile2_load = (Button) findViewById(R.id.exifmenu_profile2_buttonload);
        profile2_load.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                loadAllowedTagsFromProfile(2);
            }
        });
        Button profile2_save = (Button) findViewById(R.id.exifmenu_profile2_buttonsave);
        profile2_save.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                ImagepipePreferences.setProfileData(context,2,generateAllowedString());
                Toast.makeText(context,getResources().getString(R.string.profile_saved),Toast.LENGTH_SHORT).show();
            }
        });
        EditText profile2_label = (EditText) findViewById(R.id.exifmenu_labelprofile2);
        String label2 = ImagepipePreferences.getProfileLabel(context,2);
        if (label2!=null){
            profile2_label.setText(label2);
        }
        profile2_label.addTextChangedListener(getTextWatcherForProfile(2));
        // Profile 3
        Button profile3_load = (Button) findViewById(R.id.exifmenu_profile3_buttonload);
        profile3_load.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                loadAllowedTagsFromProfile(3);
            }
        });
        Button profile3_save = (Button) findViewById(R.id.exifmenu_profile3_buttonsave);
        profile3_save.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                ImagepipePreferences.setProfileData(context,3,generateAllowedString());
                Toast.makeText(context,getResources().getString(R.string.profile_saved),Toast.LENGTH_SHORT).show();
            }
        });
        EditText profile3_label = (EditText) findViewById(R.id.exifmenu_labelprofile3);
        String label3 = ImagepipePreferences.getProfileLabel(context,3);
        if (label3!=null){
            profile3_label.setText(label3);
        }
        profile3_label.addTextChangedListener(getTextWatcherForProfile(3));
    }

    public void askForClearCache(){

    }
}
