package de.jepfa.yapm.util

import io.ktor.util.hex


object IpConverter {

    private const val BASE_CHARS = "abcdefghijklmnopqrstuvwxyz"


    enum class PrivateNetworkClassification(val bits: Int, val rangeFrom: Long, val rangeTo: Long, val sign: Char) {
        OneClassA(24, 0x0a_00_00_00, 0x0a_ff_ff_ff, 'a'), // 8-bit mask, single class A network, 10.0.0.0 – 10.255.255.255
        ManyClassB(20, 0xac_10_00_00, 0xac_1f_ff_ff, 'e'), // 12-bit mask, 16 contiguous class B networks, 172.16.0.0 – 172.31.255.255
        ManyClassC(16, 0xc0_a8_00_00, 0xc0_a8_ff_ff, 'i'), // 16-bit block, 256 contiguous class C networks, 192.168.0.0 – 192.168.255.255
        ;

        companion object {
            fun classify(ip: Long): PrivateNetworkClassification? {
                entries.forEach {
                    if (ip >= it.rangeFrom && ip <= it.rangeTo) {
                        return it
                    }
                }
                return null
            }
        }
    }

    // only IPv4
    fun ipToLong(ip: String): Long? {
        val split = ip.split(".")
        if (split.size != 4) {
            return null
        }
        val p1 = split[0].toLongOrNull()?: return null
        val p2 = split[1].toLongOrNull()?: return null
        val p3 = split[2].toLongOrNull()?: return null
        val p4 = split[3].toLongOrNull()?: return null

        return p1.shl(24) + p2.shl(16) + p3.shl(8) + p4
    }


    fun classifyIp(ip: String): PrivateNetworkClassification? {
        val asLong = ipToLong(ip) ?: return null
        return PrivateNetworkClassification.classify(asLong)
    }

    fun getHandle(ip: String): String {
        val ipClass = classifyIp(ip) ?: return "?"
        val ipAsLong = ipToLong(ip) ?: return "?"
        val shortened = ipAsLong - ipClass.rangeFrom
        val base22 = toBasedString(shortened)
        return "@${ipClass.sign}$base22"
    }

    // my first method generated by ChatGPT :-)
    fun toBasedString(value: Long): String {
        var number = value
        val result = StringBuilder()

        do {
            val remainder = (number % BASE_CHARS.length).toInt()
            result.append(BASE_CHARS[remainder])
            number /= BASE_CHARS.length
        } while (number > 0)

        return result.reverse().toString() // Reverse the result to get the correct order
    }


}