package de.jepfa.yapm.model.secret

import android.text.Editable
import android.util.Base64
import de.jepfa.yapm.model.Clearable
import de.jepfa.yapm.util.Loop
import java.nio.ByteBuffer
import java.nio.CharBuffer
import java.nio.charset.Charset

private const val MASK_CHAR = '*'
private const val MASK_LENGTH = 16

/**
 * Represents a real password.
 *
 * Passwords are internally stored as ByteArray (encoded with system default UTF-8), not as String. This is due the VM may cache
 * all Strings internally and would them make visible in heap dumps. To at least reduce that risk
 * Strings are only created by the UI framework when displaying it (this is not in our hand unfortunately).
 * Furthermore Password instances should be cleared (#clear) if not anymore needed.
 *
 * To convert it to a readable CharSequence the ByteArray has first to be converted
 * to a CharArray. This happens without explicit Charset encoding, which means, all non-ASCII chars
 * may be displayed wrong. To get a CharArray decoded with system default Charset (UTF-8),
 * use #decodeToCharArray or #toFormattedPassword to retrieve a FormattedPassword.
 */
class Password: Secret, CharSequence {

    enum class FormattingStyle {
        IN_WORDS, IN_WORDS_MULTI_LINE, RAW;

        fun prev(): FormattingStyle {
            var prevIdx = ordinal - 1
            if (prevIdx < 0) prevIdx = values().size - 1
            return values()[prevIdx]
        }

        fun next(): FormattingStyle {
            val nextIdx = (ordinal + 1) % values().size
            return values()[nextIdx]
        }

        fun isMultiLine(): Boolean = this == IN_WORDS_MULTI_LINE

        companion object {
            val DEFAULT = IN_WORDS
            fun createFromFlags(multiLine: Boolean, formatted: Boolean) =
                if (formatted)
                    if (multiLine) IN_WORDS_MULTI_LINE else DEFAULT
                else RAW
        }
    }

    /**
     * Represents a formatted real AND encoded password to increase readability.
     */
    class FormattedPassword(): CharSequence, Clearable {
        private val wordLength = 4
        private val charList = ArrayList<Char>(32)

        private constructor(vararg sequences: CharSequence): this() {
            sequences.forEach {
                this.charList.addAll(it.toMutableList())
            }
        }

        private constructor(sequence: CharSequence): this() {
            this.charList.addAll(sequence.toMutableList())
        }

        private constructor(charList: MutableList<Char>): this() {
            this.charList.addAll(charList)
        }

        override val length: Int
            get() = charList.size

        override fun get(index: Int): Char {
            return charList[index]
        }

        override fun subSequence(startIndex: Int, endIndex: Int): CharSequence {
            return FormattedPassword(charList.subList(startIndex, endIndex))
        }


        operator fun plus(string: String): FormattedPassword {
            charList.addAll(string.toCharArray().asList())
            return this
        }

        operator fun plus(char: Char): FormattedPassword {
            charList.add(char)
            return this
        }

        override fun clear() {
            charList.clear()
        }

        /**
         * Avoid using this since it returns a String which remains in the VM heap.
         */
        override fun toString(): String {
            return String(charList.toCharArray())
        }

        companion object {
            internal fun create(
                formattingStyle: FormattingStyle,
                maskPassword: Boolean,
                useShortMask: Boolean,
                password: Password
            ): FormattedPassword {

                if (maskPassword && useShortMask) {
                    return FormattedPassword(MASK_CHAR.toString().repeat(6))
                }
                val isNumeric = password.isNumeric()
                if (!maskPassword && isNumeric && password.length < 10 && formattingStyle == FormattingStyle.IN_WORDS_MULTI_LINE) {
                    return formatNumeric(password)
                }

                val decoded = password.decodeToCharArray()

                val multiLine = formattingStyle.isMultiLine()
                val raw = formattingStyle == FormattingStyle.RAW
                val formattedPasswordLength = if (maskPassword) MASK_LENGTH else decoded.size
                val formattedPassword = FormattedPassword()
                val wordLength = if (isNumeric && !maskPassword) 2 else formattedPassword.wordLength

                for (i in 0 until formattedPasswordLength) {
                    if (!raw && i != 0 && i % wordLength == 0) {
                        if (i %  (wordLength * 2) == 0) {
                            if (multiLine) {
                                formattedPassword + System.lineSeparator()
                            } else {
                                formattedPassword + "  "
                            }
                        } else {
                            formattedPassword + " "
                        }
                    }

                    formattedPassword + (if (maskPassword) MASK_CHAR else decoded[i])
                }

                return formattedPassword
            }


            private fun formatNumeric(pin: Password): FormattedPassword {

                if (pin.length == 5) {
                    return FormattedPassword(
                        pin.subSequence(0, 2),
                        " ",
                        pin.subSequence(2, 3),
                        " ",
                        pin.subSequence(3, 5)
                    )
                }
                else if (pin.length == 6) {
                    return FormattedPassword(
                        pin.subSequence(0, 3),
                        " ",
                        pin.subSequence(3, 6)
                    )
                } else if (pin.length == 7) {
                    return FormattedPassword(
                        pin.subSequence(0, 2),
                        " ",
                        pin.subSequence(2, 5),
                        " ",
                        pin.subSequence(5, 7)
                    )
                } else if (pin.length == 8) {
                    return FormattedPassword(
                        pin.subSequence(0, 4),
                        " ",
                        pin.subSequence(4, 8)
                    )
                } else if (pin.length == 9) {
                    return FormattedPassword(
                        pin.subSequence(0, 3),
                        " ",
                        pin.subSequence(3, 6),
                        " ",
                        pin.subSequence(6, 9)
                    )
                }
                else {
                    return FormattedPassword(pin)
                }

            }
        }
    }

    /**
     * Use this constructor only for testing
     */
    constructor(passwd: String) : this(passwd.toByteArray())
    constructor(key: Key) : this(key.data)
    constructor(editable: Editable) : this(fromEditable(editable))
    constructor(chars: CharArray) : this(chars.map { it.toByte() }.toByteArray())
    constructor(bytes: ByteArray) : super(bytes)

    /**
     * Returns a CharArray 1:1 mapped from the underlying ByteArray.
     * Don't use this for non-ascii passwords since the result may look wrong! For that
     * use #decodeToCharArray instead.
     */
    fun toEncodedCharArray(): CharArray {
        return data.map { it.toChar() }.toCharArray()
    }

    fun add(other: Char) {
        val buffer = data + other.toByte()
        clear()
        data = buffer
    }

    fun replace(index: Int, other: Char) {
        data[index] = other.toByte()
    }

    fun obfuscate(key: Key) {
        Loop.loopPassword(this, key, forwards = true)
    }

    fun deobfuscate(key: Key) {
        Loop.loopPassword(this, key, forwards = false)
    }

    fun isNumeric(): Boolean {
        return data.all { it >= '0'.code.toByte() && it <= '9'.code.toByte()}
    }

    fun toFormattedPassword() =
        toFormattedPassword(FormattingStyle.DEFAULT, maskPassword = false, useShortMask = false)

    fun toFormattedPassword(
        formattingStyle: FormattingStyle,
        maskPassword: Boolean,
        useShortMask: Boolean
    ) = FormattedPassword.create(formattingStyle, maskPassword, useShortMask, this)


    fun toRawFormattedPassword() =
        toFormattedPassword(FormattingStyle.RAW, maskPassword = false, useShortMask = false)
    
    /**
     * Returns the encoded length of this password.
     * Use #toFormattedPassword to work with non-ASCII passwords
     */
    override val length: Int
        get() = data.size

    /**
     * Returns the char at position index of the encoded password.
     * Use #toFormattedPassword to work with non-ASCII passwords
     */
    override fun get(index: Int) = toEncodedCharArray()[index]

    override fun subSequence(startIndex: Int, endIndex: Int): CharSequence {
        return Password(data.copyOfRange(startIndex, endIndex))
    }

    fun toBase64String(): String = Base64.encodeToString(data, BASE64_FLAGS)

    /**
     * Avoid using this since it returns a String which remains in the VM heap.
     */
    override fun toString() = toRawFormattedPassword().toString()

    fun decodeToCharArray(): CharArray {
        val charset = Charset.defaultCharset()
        val charBuffer = charset.decode(ByteBuffer.wrap(data))
        val charArray = CharArray(charBuffer.limit())
        charBuffer.get(charArray)
        return charArray
    }

    companion object {
        private const val BASE64_FLAGS = Base64.NO_WRAP or Base64.NO_PADDING

        fun empty(): Password {
            return Password("")
        }

        fun fromBase64String(string: String): Password {
            val bytes = Base64.decode(string, BASE64_FLAGS)
            return Password(bytes)
        }

        private fun fromEditable(editable: Editable): ByteArray {
            val l = editable.length
            val charArray = CharArray(l)
            editable.getChars(0, l, charArray, 0)
            return encodeToByteArray(charArray)
        }

        private fun encodeToByteArray(charArray: CharArray): ByteArray {
            val charset = Charset.defaultCharset()
            val charBuffer = charset.encode(CharBuffer.wrap(charArray))
            val byteArray = ByteArray(charBuffer.limit())
            charBuffer.get(byteArray)
            return byteArray
        }
    }
}