part of 'internal_parts.dart';

/// Base class for each of the [Dependency] types.
/// This is an abstract class that should not  be used directly
abstract class Dependency {
  /// Loads a dependency located at [line].
  factory Dependency._loadFrom(Dependencies dependencies, LineImpl line) {
    final children = line.childrenOf(type: LineType.key);

    if (children.isEmpty) {
      // pub hosted is the default and the only type
      // that has no children
      return DependencyPubHosted._fromLine(dependencies, line);
    }

    /// So not a pub hosted dep, we use the main key
    /// from each of the dependency types to discover
    /// which type of dependeny we have.
    final depTypeLine = line.findOneOf([
      DependencyAltHosted.keyName,
      DependencyPath.keyName,
      DependencyGit.keyName,
      DependencySdk.keyName
    ]);

    // none of the children had one of the expected keys.
    if (depTypeLine == null) {
      // there may have been multiple unexpcted children but we just
      // report the first one.
      throw PubSpecException(
          line, 'Unexpected child key found ${children.first.key}');
    }

    /// We know the type of dependency so lets load the details.
    switch (depTypeLine.key) {
      case DependencyAltHosted.keyName:
        return DependencyAltHosted._fromLine(dependencies, line);
      case DependencyPath.keyName:
        return DependencyPath._fromLine(dependencies, line);
      case DependencyGit.keyName:
        return DependencyGit._fromLine(dependencies, line);
      case DependencySdk.keyName:
        return DependencySdk._fromLine(dependencies, line);
    }

    throw PubSpecException(
        depTypeLine, 'The child dependency does not appear to be valid.');
  }

  /// The name of the dependency package
  String get name;

  SectionImpl get _section;

  Comments get comments;

  /// The line number within the pubspec.yaml
  /// where this dependency is located.
  int get lineNo;

  /// Adds the [Dependency] generated by the passed [dependencyBuilder] to
  /// the pubspec's dependency section (that the this dependency is a member of)
  /// and returns the generated dependenchy.
  /// Allows a user to chain a series of dependency append.
  /// operations.
  /// In a future release we intended to sort the dependency alphabetically.
  ///
  /// ```dart
  /// pubspec.dependencies
  ///   .add(DependencyPubHostedBuilder(name: 'money')
  ///   .add(DependencyPubHostedBuilder(name: 'fixed')));
  /// ```
  Dependency add(DependencyBuilder dependencyBuilder);
}
