import 'package:simple_sparse_list/simple_sparse_list.dart';

/// An identifier of Unicode category 'Cn'.
const notAssigned = 0;

/// An identifier of Unicode category 'Cc'.
const control = 1;

/// An identifier of Unicode category 'Cf'.
const format = 2;

/// An identifier of Unicode category 'Co'.
const privateUse = 3;

/// An identifier of Unicode category 'Cs'.
const surrogate = 4;

/// An identifier of Unicode category 'Ll'.
const lowerCaseLetter = 5;

/// An identifier of Unicode category 'Lm'.
const modifierLetter = 6;

/// An identifier of Unicode category 'Lo'.
const otherLetter = 7;

/// An identifier of Unicode category 'Lt'.
const titleCaseLetter = 8;

/// An identifier of Unicode category 'Lu'.
const upperCaseLetter = 9;

/// An identifier of Unicode category 'Mc'.
const spacingMark = 10;

/// An identifier of Unicode category 'Me'.
const enclosingMark = 11;

/// An identifier of Unicode category 'Mn'.
const nonspacingMark = 12;

/// An identifier of Unicode category 'Nd'.
const decimalNumber = 13;

/// An identifier of Unicode category 'Nl'.
const letterNumber = 14;

/// An identifier of Unicode category 'No'.
const otherNumber = 15;

/// An identifier of Unicode category 'Pc'.
const connectorPunctuation = 16;

/// An identifier of Unicode category 'Pd'.
const dashPunctuation = 17;

/// An identifier of Unicode category 'Pe'.
const closePunctuation = 18;

/// An identifier of Unicode category 'Pf'.
const finalPunctuation = 19;

/// An identifier of Unicode category 'Pi'.
const initialPunctuation = 20;

/// An identifier of Unicode category 'Po'.
const otherPunctuation = 21;

/// An identifier of Unicode category 'Ps'.
const openPunctuation = 22;

/// An identifier of Unicode category 'Sc'.
const currencySymbol = 23;

/// An identifier of Unicode category 'Sk'.
const modifierSymbol = 24;

/// An identifier of Unicode category 'Sm'.
const mathSymbol = 25;

/// An identifier of Unicode category 'So'.
const otherSymbol = 26;

/// An identifier of Unicode category 'Zl'.
const lineSeparator = 27;

/// An identifier of Unicode category 'Zp'.
const paragraphSeparator = 28;

/// An identifier of Unicode category 'Zs'.
const spaceSeparator = 29;

enum UnicodeCategory {
  notAssigned,
  control,
  format,
  privateUse,
  surrogate,
  lowerCaseLetter,
  modifierLetter,
  otherLetter,
  titleCaseLetter,
  upperCaseLetter,
  spacingMark,
  enclosingMark,
  nonspacingMark,
  decimalNumber,
  letterNumber,
  otherNumber,
  connectorPunctuation,
  dashPunctuation,
  closePunctuation,
  finalPunctuation,
  initialPunctuation,
  otherPunctuation,
  openPunctuation,
  currencySymbol,
  modifierSymbol,
  mathSymbol,
  otherSymbol,
  lineSeparator,
  paragraphSeparator,
  spaceSeparator
}

/// List used to map a character to a category identifier.
final SparseList<int> generalCategories = _generalCategories;

/// Converts a [character] to lowercase.
int charToLowerCase(int character) {
  if (character < 0 || character > 1114111) {
    throw RangeError.range(character, 0, 1114111, 'character');
  }

  final delta = _lowerCaseList[character];
  return delta == null ? character : character + delta;
}

/// Converts a [character] to titlecase.
int charToTitleCase(int character) {
  if (character < 0 || character > 1114111) {
    throw RangeError.range(character, 0, 1114111, 'character');
  }

  final delta = _titleCaseList[character];
  return delta == null ? character : character + delta;
}

/// Converts a [character] to uppercase.
int charToUpperCase(int character) {
  if (character < 0 || character > 1114111) {
    throw RangeError.range(character, 0, 1114111, 'character');
  }

  final delta = _upperCaseList[character];
  return delta == null ? character : character + delta;
}

/// Returns the general Unicode category as [UnicodeCategory]
UnicodeCategory getUnicodeCategory(int character) {
  if (character < 0 || character > 1114111) {
    throw RangeError.range(character, 0, 1114111, 'character');
  }

  final index = generalCategories[character];
  final result = UnicodeCategory.values[index];
  return result;
}

/// Checks if a [character] is in the Unicode codespace 'not assigned' (Cn).
bool isNotAssigned(int character) => _getCategory(character) == notAssigned;

/// Checks if a [character] is in the Unicode codespace 'control' (Cc).
bool isControl(int character) => _getCategory(character) == control;

/// Checks if a [character] is in the Unicode codespace 'format' (Cf).
bool isFormat(int character) => _getCategory(character) == format;

/// Checks if a [character] is in the Unicode codespace 'private use' (Co).
bool isPrivateUse(int character) => _getCategory(character) == privateUse;

/// Checks if a [character] is in the Unicode codespace 'surrogate' (Cs).
bool isSurrogate(int character) => _getCategory(character) == surrogate;

/// Checks if a [character] is in the Unicode codespace 'lower case letter' (Ll).
bool isLowerCaseLetter(int character) => _getCategory(character) == lowerCaseLetter;

/// Checks if a [character] is in the Unicode codespace 'modifier letter' (Lm).
bool isModifierLetter(int character) => _getCategory(character) == modifierLetter;

/// Checks if a [character] is in the Unicode codespace 'other letter' (Lo).
bool isOtherLetter(int character) => _getCategory(character) == otherLetter;

/// Checks if a [character] is in the Unicode codespace 'title case letter' (Lt).
bool isTitleCaseLetter(int character) => _getCategory(character) == titleCaseLetter;

/// Checks if a [character] is in the Unicode codespace 'upper case letter' (Lu).
bool isUpperCaseLetter(int character) => _getCategory(character) == upperCaseLetter;

/// Checks if a [character] is in the Unicode codespace 'spacing mark' (Mc).
bool isSpacingMark(int character) => _getCategory(character) == spacingMark;

/// Checks if a [character] is in the Unicode codespace 'enclosing mark' (Me).
bool isEnclosingMark(int character) => _getCategory(character) == enclosingMark;

/// Checks if a [character] is in the Unicode codespace 'nonspacing mark' (Mn).
bool isNonspacingMark(int character) => _getCategory(character) == nonspacingMark;

/// Checks if a [character] is in the Unicode codespace 'decimal number' (Nd).
bool isDecimalNumber(int character) => _getCategory(character) == decimalNumber;

/// Checks if a [character] is in the Unicode codespace 'letter number' (Nl).
bool isLetterNumber(int character) => _getCategory(character) == letterNumber;

/// Checks if a [character] is in the Unicode codespace 'other number' (No).
bool isOtherNumber(int character) => _getCategory(character) == otherNumber;

/// Checks if a [character] is in the Unicode codespace 'connector punctuation' (Pc).
bool isConnectorPunctuation(int character) => _getCategory(character) == connectorPunctuation;

/// Checks if a [character] is in the Unicode codespace 'dash punctuation' (Pd).
bool isDashPunctuation(int character) => _getCategory(character) == dashPunctuation;

/// Checks if a [character] is in the Unicode codespace 'close punctuation' (Pe).
bool isClosePunctuation(int character) => _getCategory(character) == closePunctuation;

/// Checks if a [character] is in the Unicode codespace 'final punctuation' (Pf).
bool isFinalPunctuation(int character) => _getCategory(character) == finalPunctuation;

/// Checks if a [character] is in the Unicode codespace 'initial punctuation' (Pi).
bool isInitialPunctuation(int character) => _getCategory(character) == initialPunctuation;

/// Checks if a [character] is in the Unicode codespace 'other punctuation' (Po).
bool isOtherPunctuation(int character) => _getCategory(character) == otherPunctuation;

/// Checks if a [character] is in the Unicode codespace 'open punctuation' (Ps).
bool isOpenPunctuation(int character) => _getCategory(character) == openPunctuation;

/// Checks if a [character] is in the Unicode codespace 'currency symbol' (Sc).
bool isCurrencySymbol(int character) => _getCategory(character) == currencySymbol;

/// Checks if a [character] is in the Unicode codespace 'modifier symbol' (Sk).
bool isModifierSymbol(int character) => _getCategory(character) == modifierSymbol;

/// Checks if a [character] is in the Unicode codespace 'math symbol' (Sm).
bool isMathSymbol(int character) => _getCategory(character) == mathSymbol;

/// Checks if a [character] is in the Unicode codespace 'other symbol' (So).
bool isOtherSymbol(int character) => _getCategory(character) == otherSymbol;

/// Checks if a [character] is in the Unicode codespace 'line separator' (Zl).
bool isLineSeparator(int character) => _getCategory(character) == lineSeparator;

/// Checks if a [character] is in the Unicode codespace 'paragraph separator' (Zp).
bool isParagraphSeparator(int character) => _getCategory(character) == paragraphSeparator;

/// Checks if a [character] is in the Unicode codespace 'space separator' (Zs).
bool isSpaceSeparator(int character) => _getCategory(character) == spaceSeparator;

/// Converts a [string] to lowercase.
String toLowerCase(String string) => _toLetterCase(string, _lowerCaseList);

/// Converts a [string] to titlecase.
String toTitleCase(String string) => _toLetterCase(string, _titleCaseList);

/// Converts a [string] to uppercase.
String toUpperCase(String string) => _toLetterCase(string, _upperCaseList);

int toRune(String string) => string.isEmpty
    ? throw ArgumentError('An empty string contains no elements')
    : string.runes.first;

List<int> toRunes(String string) => string.runes.toList();

int _getCategory(int character) {
  if (character < 0 || character > 1114111) {
    throw RangeError.range(character, 0, 1114111, 'character');
  }

  return generalCategories[character];
}

String _toLetterCase(String string, SparseList<int?> letterCaseList) {
  final runes = toRunes(string);
  final length = runes.length;
  for (var i = 0; i < length; i++) {
    final rune = runes[i];
    final delta = letterCaseList[rune];
    if (delta != null) {
      runes[i] = rune + delta;
    }
  }

  return String.fromCharCodes(runes);
}

SparseList<int> _unpack(List<List<int>> data) {
  var ranges = <(int, int, int)>[];
  for (var i = 0; i < data.length; i++) {
    final category = data[i];
    var prev = 0;
    for (var j = 0; j < category.length; j += 2) {
      final start = category[j] + prev;
      prev = start;
      final end = category[j + 1] + prev;
      prev = end;
      ranges.add((start, end, i));
    }
  }

  ranges = ranges.toList();
  ranges.sort((a, b) {
    if (a.$1 != b.$1) {
      return a.$1.compareTo(b.$1);
    }

    return a.$2.compareTo(b.$2);
  });

  final list = SparseList(ranges, 0, length: 0x110000);
  return list;
}

final _lowerCaseList = SparseList([(65, 90, 32), (192, 214, 32), (216, 222, 32), (913, 929, 32), (931, 939, 32), (1040, 1071, 32), (65313, 65338, 32), (68944, 68965, 32), (71840, 71871, 32), (93760, 93791, 32), (256, 256, 1), (258, 258, 1), (260, 260, 1), (262, 262, 1), (264, 264, 1), (266, 266, 1), (268, 268, 1), (270, 270, 1), (272, 272, 1), (274, 274, 1), (276, 276, 1), (278, 278, 1), (280, 280, 1), (282, 282, 1), (284, 284, 1), (286, 286, 1), (288, 288, 1), (290, 290, 1), (292, 292, 1), (294, 294, 1), (296, 296, 1), (298, 298, 1), (300, 300, 1), (302, 302, 1), (306, 306, 1), (308, 308, 1), (310, 310, 1), (313, 313, 1), (315, 315, 1), (317, 317, 1), (319, 319, 1), (321, 321, 1), (323, 323, 1), (325, 325, 1), (327, 327, 1), (330, 330, 1), (332, 332, 1), (334, 334, 1), (336, 336, 1), (338, 338, 1), (340, 340, 1), (342, 342, 1), (344, 344, 1), (346, 346, 1), (348, 348, 1), (350, 350, 1), (352, 352, 1), (354, 354, 1), (356, 356, 1), (358, 358, 1), (360, 360, 1), (362, 362, 1), (364, 364, 1), (366, 366, 1), (368, 368, 1), (370, 370, 1), (372, 372, 1), (374, 374, 1), (377, 377, 1), (379, 379, 1), (381, 381, 1), (386, 386, 1), (388, 388, 1), (391, 391, 1), (395, 395, 1), (401, 401, 1), (408, 408, 1), (416, 416, 1), (418, 418, 1), (420, 420, 1), (423, 423, 1), (428, 428, 1), (431, 431, 1), (435, 435, 1), (437, 437, 1), (440, 440, 1), (444, 444, 1), (453, 453, 1), (456, 456, 1), (459, 459, 1), (461, 461, 1), (463, 463, 1), (465, 465, 1), (467, 467, 1), (469, 469, 1), (471, 471, 1), (473, 473, 1), (475, 475, 1), (478, 478, 1), (480, 480, 1), (482, 482, 1), (484, 484, 1), (486, 486, 1), (488, 488, 1), (490, 490, 1), (492, 492, 1), (494, 494, 1), (498, 498, 1), (500, 500, 1), (504, 504, 1), (506, 506, 1), (508, 508, 1), (510, 510, 1), (512, 512, 1), (514, 514, 1), (516, 516, 1), (518, 518, 1), (520, 520, 1), (522, 522, 1), (524, 524, 1), (526, 526, 1), (528, 528, 1), (530, 530, 1), (532, 532, 1), (534, 534, 1), (536, 536, 1), (538, 538, 1), (540, 540, 1), (542, 542, 1), (546, 546, 1), (548, 548, 1), (550, 550, 1), (552, 552, 1), (554, 554, 1), (556, 556, 1), (558, 558, 1), (560, 560, 1), (562, 562, 1), (571, 571, 1), (577, 577, 1), (582, 582, 1), (584, 584, 1), (586, 586, 1), (588, 588, 1), (590, 590, 1), (880, 880, 1), (882, 882, 1), (886, 886, 1), (984, 984, 1), (986, 986, 1), (988, 988, 1), (990, 990, 1), (992, 992, 1), (994, 994, 1), (996, 996, 1), (998, 998, 1), (1000, 1000, 1), (1002, 1002, 1), (1004, 1004, 1), (1006, 1006, 1), (1015, 1015, 1), (1018, 1018, 1), (1120, 1120, 1), (1122, 1122, 1), (1124, 1124, 1), (1126, 1126, 1), (1128, 1128, 1), (1130, 1130, 1), (1132, 1132, 1), (1134, 1134, 1), (1136, 1136, 1), (1138, 1138, 1), (1140, 1140, 1), (1142, 1142, 1), (1144, 1144, 1), (1146, 1146, 1), (1148, 1148, 1), (1150, 1150, 1), (1152, 1152, 1), (1162, 1162, 1), (1164, 1164, 1), (1166, 1166, 1), (1168, 1168, 1), (1170, 1170, 1), (1172, 1172, 1), (1174, 1174, 1), (1176, 1176, 1), (1178, 1178, 1), (1180, 1180, 1), (1182, 1182, 1), (1184, 1184, 1), (1186, 1186, 1), (1188, 1188, 1), (1190, 1190, 1), (1192, 1192, 1), (1194, 1194, 1), (1196, 1196, 1), (1198, 1198, 1), (1200, 1200, 1), (1202, 1202, 1), (1204, 1204, 1), (1206, 1206, 1), (1208, 1208, 1), (1210, 1210, 1), (1212, 1212, 1), (1214, 1214, 1), (1217, 1217, 1), (1219, 1219, 1), (1221, 1221, 1), (1223, 1223, 1), (1225, 1225, 1), (1227, 1227, 1), (1229, 1229, 1), (1232, 1232, 1), (1234, 1234, 1), (1236, 1236, 1), (1238, 1238, 1), (1240, 1240, 1), (1242, 1242, 1), (1244, 1244, 1), (1246, 1246, 1), (1248, 1248, 1), (1250, 1250, 1), (1252, 1252, 1), (1254, 1254, 1), (1256, 1256, 1), (1258, 1258, 1), (1260, 1260, 1), (1262, 1262, 1), (1264, 1264, 1), (1266, 1266, 1), (1268, 1268, 1), (1270, 1270, 1), (1272, 1272, 1), (1274, 1274, 1), (1276, 1276, 1), (1278, 1278, 1), (1280, 1280, 1), (1282, 1282, 1), (1284, 1284, 1), (1286, 1286, 1), (1288, 1288, 1), (1290, 1290, 1), (1292, 1292, 1), (1294, 1294, 1), (1296, 1296, 1), (1298, 1298, 1), (1300, 1300, 1), (1302, 1302, 1), (1304, 1304, 1), (1306, 1306, 1), (1308, 1308, 1), (1310, 1310, 1), (1312, 1312, 1), (1314, 1314, 1), (1316, 1316, 1), (1318, 1318, 1), (1320, 1320, 1), (1322, 1322, 1), (1324, 1324, 1), (1326, 1326, 1), (7305, 7305, 1), (7680, 7680, 1), (7682, 7682, 1), (7684, 7684, 1), (7686, 7686, 1), (7688, 7688, 1), (7690, 7690, 1), (7692, 7692, 1), (7694, 7694, 1), (7696, 7696, 1), (7698, 7698, 1), (7700, 7700, 1), (7702, 7702, 1), (7704, 7704, 1), (7706, 7706, 1), (7708, 7708, 1), (7710, 7710, 1), (7712, 7712, 1), (7714, 7714, 1), (7716, 7716, 1), (7718, 7718, 1), (7720, 7720, 1), (7722, 7722, 1), (7724, 7724, 1), (7726, 7726, 1), (7728, 7728, 1), (7730, 7730, 1), (7732, 7732, 1), (7734, 7734, 1), (7736, 7736, 1), (7738, 7738, 1), (7740, 7740, 1), (7742, 7742, 1), (7744, 7744, 1), (7746, 7746, 1), (7748, 7748, 1), (7750, 7750, 1), (7752, 7752, 1), (7754, 7754, 1), (7756, 7756, 1), (7758, 7758, 1), (7760, 7760, 1), (7762, 7762, 1), (7764, 7764, 1), (7766, 7766, 1), (7768, 7768, 1), (7770, 7770, 1), (7772, 7772, 1), (7774, 7774, 1), (7776, 7776, 1), (7778, 7778, 1), (7780, 7780, 1), (7782, 7782, 1), (7784, 7784, 1), (7786, 7786, 1), (7788, 7788, 1), (7790, 7790, 1), (7792, 7792, 1), (7794, 7794, 1), (7796, 7796, 1), (7798, 7798, 1), (7800, 7800, 1), (7802, 7802, 1), (7804, 7804, 1), (7806, 7806, 1), (7808, 7808, 1), (7810, 7810, 1), (7812, 7812, 1), (7814, 7814, 1), (7816, 7816, 1), (7818, 7818, 1), (7820, 7820, 1), (7822, 7822, 1), (7824, 7824, 1), (7826, 7826, 1), (7828, 7828, 1), (7840, 7840, 1), (7842, 7842, 1), (7844, 7844, 1), (7846, 7846, 1), (7848, 7848, 1), (7850, 7850, 1), (7852, 7852, 1), (7854, 7854, 1), (7856, 7856, 1), (7858, 7858, 1), (7860, 7860, 1), (7862, 7862, 1), (7864, 7864, 1), (7866, 7866, 1), (7868, 7868, 1), (7870, 7870, 1), (7872, 7872, 1), (7874, 7874, 1), (7876, 7876, 1), (7878, 7878, 1), (7880, 7880, 1), (7882, 7882, 1), (7884, 7884, 1), (7886, 7886, 1), (7888, 7888, 1), (7890, 7890, 1), (7892, 7892, 1), (7894, 7894, 1), (7896, 7896, 1), (7898, 7898, 1), (7900, 7900, 1), (7902, 7902, 1), (7904, 7904, 1), (7906, 7906, 1), (7908, 7908, 1), (7910, 7910, 1), (7912, 7912, 1), (7914, 7914, 1), (7916, 7916, 1), (7918, 7918, 1), (7920, 7920, 1), (7922, 7922, 1), (7924, 7924, 1), (7926, 7926, 1), (7928, 7928, 1), (7930, 7930, 1), (7932, 7932, 1), (7934, 7934, 1), (8579, 8579, 1), (11360, 11360, 1), (11367, 11367, 1), (11369, 11369, 1), (11371, 11371, 1), (11378, 11378, 1), (11381, 11381, 1), (11392, 11392, 1), (11394, 11394, 1), (11396, 11396, 1), (11398, 11398, 1), (11400, 11400, 1), (11402, 11402, 1), (11404, 11404, 1), (11406, 11406, 1), (11408, 11408, 1), (11410, 11410, 1), (11412, 11412, 1), (11414, 11414, 1), (11416, 11416, 1), (11418, 11418, 1), (11420, 11420, 1), (11422, 11422, 1), (11424, 11424, 1), (11426, 11426, 1), (11428, 11428, 1), (11430, 11430, 1), (11432, 11432, 1), (11434, 11434, 1), (11436, 11436, 1), (11438, 11438, 1), (11440, 11440, 1), (11442, 11442, 1), (11444, 11444, 1), (11446, 11446, 1), (11448, 11448, 1), (11450, 11450, 1), (11452, 11452, 1), (11454, 11454, 1), (11456, 11456, 1), (11458, 11458, 1), (11460, 11460, 1), (11462, 11462, 1), (11464, 11464, 1), (11466, 11466, 1), (11468, 11468, 1), (11470, 11470, 1), (11472, 11472, 1), (11474, 11474, 1), (11476, 11476, 1), (11478, 11478, 1), (11480, 11480, 1), (11482, 11482, 1), (11484, 11484, 1), (11486, 11486, 1), (11488, 11488, 1), (11490, 11490, 1), (11499, 11499, 1), (11501, 11501, 1), (11506, 11506, 1), (42560, 42560, 1), (42562, 42562, 1), (42564, 42564, 1), (42566, 42566, 1), (42568, 42568, 1), (42570, 42570, 1), (42572, 42572, 1), (42574, 42574, 1), (42576, 42576, 1), (42578, 42578, 1), (42580, 42580, 1), (42582, 42582, 1), (42584, 42584, 1), (42586, 42586, 1), (42588, 42588, 1), (42590, 42590, 1), (42592, 42592, 1), (42594, 42594, 1), (42596, 42596, 1), (42598, 42598, 1), (42600, 42600, 1), (42602, 42602, 1), (42604, 42604, 1), (42624, 42624, 1), (42626, 42626, 1), (42628, 42628, 1), (42630, 42630, 1), (42632, 42632, 1), (42634, 42634, 1), (42636, 42636, 1), (42638, 42638, 1), (42640, 42640, 1), (42642, 42642, 1), (42644, 42644, 1), (42646, 42646, 1), (42648, 42648, 1), (42650, 42650, 1), (42786, 42786, 1), (42788, 42788, 1), (42790, 42790, 1), (42792, 42792, 1), (42794, 42794, 1), (42796, 42796, 1), (42798, 42798, 1), (42802, 42802, 1), (42804, 42804, 1), (42806, 42806, 1), (42808, 42808, 1), (42810, 42810, 1), (42812, 42812, 1), (42814, 42814, 1), (42816, 42816, 1), (42818, 42818, 1), (42820, 42820, 1), (42822, 42822, 1), (42824, 42824, 1), (42826, 42826, 1), (42828, 42828, 1), (42830, 42830, 1), (42832, 42832, 1), (42834, 42834, 1), (42836, 42836, 1), (42838, 42838, 1), (42840, 42840, 1), (42842, 42842, 1), (42844, 42844, 1), (42846, 42846, 1), (42848, 42848, 1), (42850, 42850, 1), (42852, 42852, 1), (42854, 42854, 1), (42856, 42856, 1), (42858, 42858, 1), (42860, 42860, 1), (42862, 42862, 1), (42873, 42873, 1), (42875, 42875, 1), (42878, 42878, 1), (42880, 42880, 1), (42882, 42882, 1), (42884, 42884, 1), (42886, 42886, 1), (42891, 42891, 1), (42896, 42896, 1), (42898, 42898, 1), (42902, 42902, 1), (42904, 42904, 1), (42906, 42906, 1), (42908, 42908, 1), (42910, 42910, 1), (42912, 42912, 1), (42914, 42914, 1), (42916, 42916, 1), (42918, 42918, 1), (42920, 42920, 1), (42932, 42932, 1), (42934, 42934, 1), (42936, 42936, 1), (42938, 42938, 1), (42940, 42940, 1), (42942, 42942, 1), (42944, 42944, 1), (42946, 42946, 1), (42951, 42951, 1), (42953, 42953, 1), (42956, 42956, 1), (42960, 42960, 1), (42966, 42966, 1), (42968, 42968, 1), (42970, 42970, 1), (42997, 42997, 1), (304, 304, -199), (376, 376, -121), (385, 385, 210), (390, 390, 206), (393, 394, 205), (403, 403, 205), (398, 398, 79), (399, 399, 202), (400, 400, 203), (404, 404, 207), (406, 406, 211), (412, 412, 211), (407, 407, 209), (413, 413, 213), (415, 415, 214), (422, 422, 218), (425, 425, 218), (430, 430, 218), (433, 434, 217), (439, 439, 219), (452, 452, 2), (455, 455, 2), (458, 458, 2), (497, 497, 2), (502, 502, -97), (503, 503, -56), (544, 544, -130), (1021, 1023, -130), (570, 570, 10795), (573, 573, -163), (574, 574, 10792), (579, 579, -195), (580, 580, 69), (581, 581, 71), (895, 895, 116), (902, 902, 38), (904, 906, 37), (908, 908, 64), (68736, 68786, 64), (910, 911, 63), (975, 975, 8), (5104, 5109, 8), (1012, 1012, -60), (1017, 1017, -7), (8172, 8172, -7), (1024, 1039, 80), (1216, 1216, 15), (1329, 1366, 48), (11264, 11311, 48), (4256, 4293, 7264), (4295, 4295, 7264), (4301, 4301, 7264), (5024, 5103, 38864), (7312, 7354, -3008), (7357, 7359, -3008), (7838, 7838, -7615), (7944, 7951, -8), (7960, 7965, -8), (7976, 7983, -8), (7992, 7999, -8), (8008, 8013, -8), (8025, 8025, -8), (8027, 8027, -8), (8029, 8029, -8), (8031, 8031, -8), (8040, 8047, -8), (8072, 8079, -8), (8088, 8095, -8), (8104, 8111, -8), (8120, 8121, -8), (8152, 8153, -8), (8168, 8169, -8), (8122, 8123, -74), (8124, 8124, -9), (8140, 8140, -9), (8188, 8188, -9), (8136, 8139, -86), (8154, 8155, -100), (8170, 8171, -112), (8184, 8185, -128), (8186, 8187, -126), (8486, 8486, -7517), (8490, 8490, -8383), (8491, 8491, -8262), (8498, 8498, 28), (8544, 8559, 16), (9398, 9423, 26), (11362, 11362, -10743), (11363, 11363, -3814), (11364, 11364, -10727), (11373, 11373, -10780), (11374, 11374, -10749), (11375, 11375, -10783), (11376, 11376, -10782), (11390, 11391, -10815), (42877, 42877, -35332), (42893, 42893, -42280), (42922, 42922, -42308), (42926, 42926, -42308), (42923, 42923, -42319), (42924, 42924, -42315), (42925, 42925, -42305), (42928, 42928, -42258), (42929, 42929, -42282), (42930, 42930, -42261), (42931, 42931, 928), (42948, 42948, -48), (42949, 42949, -42307), (42950, 42950, -35384), (42955, 42955, -42343), (42972, 42972, -42561), (66560, 66599, 40), (66736, 66771, 40), (66928, 66938, 39), (66940, 66954, 39), (66956, 66962, 39), (66964, 66965, 39), (125184, 125217, 34)], null, length: 0x110000);

final _titleCaseList = SparseList([(97, 122, -32), (224, 246, -32), (248, 254, -32), (945, 961, -32), (963, 971, -32), (1072, 1103, -32), (65345, 65370, -32), (68976, 68997, -32), (71872, 71903, -32), (93792, 93823, -32), (181, 181, 743), (255, 255, 121), (257, 257, -1), (259, 259, -1), (261, 261, -1), (263, 263, -1), (265, 265, -1), (267, 267, -1), (269, 269, -1), (271, 271, -1), (273, 273, -1), (275, 275, -1), (277, 277, -1), (279, 279, -1), (281, 281, -1), (283, 283, -1), (285, 285, -1), (287, 287, -1), (289, 289, -1), (291, 291, -1), (293, 293, -1), (295, 295, -1), (297, 297, -1), (299, 299, -1), (301, 301, -1), (303, 303, -1), (307, 307, -1), (309, 309, -1), (311, 311, -1), (314, 314, -1), (316, 316, -1), (318, 318, -1), (320, 320, -1), (322, 322, -1), (324, 324, -1), (326, 326, -1), (328, 328, -1), (331, 331, -1), (333, 333, -1), (335, 335, -1), (337, 337, -1), (339, 339, -1), (341, 341, -1), (343, 343, -1), (345, 345, -1), (347, 347, -1), (349, 349, -1), (351, 351, -1), (353, 353, -1), (355, 355, -1), (357, 357, -1), (359, 359, -1), (361, 361, -1), (363, 363, -1), (365, 365, -1), (367, 367, -1), (369, 369, -1), (371, 371, -1), (373, 373, -1), (375, 375, -1), (378, 378, -1), (380, 380, -1), (382, 382, -1), (387, 387, -1), (389, 389, -1), (392, 392, -1), (396, 396, -1), (402, 402, -1), (409, 409, -1), (417, 417, -1), (419, 419, -1), (421, 421, -1), (424, 424, -1), (429, 429, -1), (432, 432, -1), (436, 436, -1), (438, 438, -1), (441, 441, -1), (445, 445, -1), (454, 454, -1), (457, 457, -1), (460, 460, -1), (462, 462, -1), (464, 464, -1), (466, 466, -1), (468, 468, -1), (470, 470, -1), (472, 472, -1), (474, 474, -1), (476, 476, -1), (479, 479, -1), (481, 481, -1), (483, 483, -1), (485, 485, -1), (487, 487, -1), (489, 489, -1), (491, 491, -1), (493, 493, -1), (495, 495, -1), (499, 499, -1), (501, 501, -1), (505, 505, -1), (507, 507, -1), (509, 509, -1), (511, 511, -1), (513, 513, -1), (515, 515, -1), (517, 517, -1), (519, 519, -1), (521, 521, -1), (523, 523, -1), (525, 525, -1), (527, 527, -1), (529, 529, -1), (531, 531, -1), (533, 533, -1), (535, 535, -1), (537, 537, -1), (539, 539, -1), (541, 541, -1), (543, 543, -1), (547, 547, -1), (549, 549, -1), (551, 551, -1), (553, 553, -1), (555, 555, -1), (557, 557, -1), (559, 559, -1), (561, 561, -1), (563, 563, -1), (572, 572, -1), (578, 578, -1), (583, 583, -1), (585, 585, -1), (587, 587, -1), (589, 589, -1), (591, 591, -1), (881, 881, -1), (883, 883, -1), (887, 887, -1), (985, 985, -1), (987, 987, -1), (989, 989, -1), (991, 991, -1), (993, 993, -1), (995, 995, -1), (997, 997, -1), (999, 999, -1), (1001, 1001, -1), (1003, 1003, -1), (1005, 1005, -1), (1007, 1007, -1), (1016, 1016, -1), (1019, 1019, -1), (1121, 1121, -1), (1123, 1123, -1), (1125, 1125, -1), (1127, 1127, -1), (1129, 1129, -1), (1131, 1131, -1), (1133, 1133, -1), (1135, 1135, -1), (1137, 1137, -1), (1139, 1139, -1), (1141, 1141, -1), (1143, 1143, -1), (1145, 1145, -1), (1147, 1147, -1), (1149, 1149, -1), (1151, 1151, -1), (1153, 1153, -1), (1163, 1163, -1), (1165, 1165, -1), (1167, 1167, -1), (1169, 1169, -1), (1171, 1171, -1), (1173, 1173, -1), (1175, 1175, -1), (1177, 1177, -1), (1179, 1179, -1), (1181, 1181, -1), (1183, 1183, -1), (1185, 1185, -1), (1187, 1187, -1), (1189, 1189, -1), (1191, 1191, -1), (1193, 1193, -1), (1195, 1195, -1), (1197, 1197, -1), (1199, 1199, -1), (1201, 1201, -1), (1203, 1203, -1), (1205, 1205, -1), (1207, 1207, -1), (1209, 1209, -1), (1211, 1211, -1), (1213, 1213, -1), (1215, 1215, -1), (1218, 1218, -1), (1220, 1220, -1), (1222, 1222, -1), (1224, 1224, -1), (1226, 1226, -1), (1228, 1228, -1), (1230, 1230, -1), (1233, 1233, -1), (1235, 1235, -1), (1237, 1237, -1), (1239, 1239, -1), (1241, 1241, -1), (1243, 1243, -1), (1245, 1245, -1), (1247, 1247, -1), (1249, 1249, -1), (1251, 1251, -1), (1253, 1253, -1), (1255, 1255, -1), (1257, 1257, -1), (1259, 1259, -1), (1261, 1261, -1), (1263, 1263, -1), (1265, 1265, -1), (1267, 1267, -1), (1269, 1269, -1), (1271, 1271, -1), (1273, 1273, -1), (1275, 1275, -1), (1277, 1277, -1), (1279, 1279, -1), (1281, 1281, -1), (1283, 1283, -1), (1285, 1285, -1), (1287, 1287, -1), (1289, 1289, -1), (1291, 1291, -1), (1293, 1293, -1), (1295, 1295, -1), (1297, 1297, -1), (1299, 1299, -1), (1301, 1301, -1), (1303, 1303, -1), (1305, 1305, -1), (1307, 1307, -1), (1309, 1309, -1), (1311, 1311, -1), (1313, 1313, -1), (1315, 1315, -1), (1317, 1317, -1), (1319, 1319, -1), (1321, 1321, -1), (1323, 1323, -1), (1325, 1325, -1), (1327, 1327, -1), (7306, 7306, -1), (7681, 7681, -1), (7683, 7683, -1), (7685, 7685, -1), (7687, 7687, -1), (7689, 7689, -1), (7691, 7691, -1), (7693, 7693, -1), (7695, 7695, -1), (7697, 7697, -1), (7699, 7699, -1), (7701, 7701, -1), (7703, 7703, -1), (7705, 7705, -1), (7707, 7707, -1), (7709, 7709, -1), (7711, 7711, -1), (7713, 7713, -1), (7715, 7715, -1), (7717, 7717, -1), (7719, 7719, -1), (7721, 7721, -1), (7723, 7723, -1), (7725, 7725, -1), (7727, 7727, -1), (7729, 7729, -1), (7731, 7731, -1), (7733, 7733, -1), (7735, 7735, -1), (7737, 7737, -1), (7739, 7739, -1), (7741, 7741, -1), (7743, 7743, -1), (7745, 7745, -1), (7747, 7747, -1), (7749, 7749, -1), (7751, 7751, -1), (7753, 7753, -1), (7755, 7755, -1), (7757, 7757, -1), (7759, 7759, -1), (7761, 7761, -1), (7763, 7763, -1), (7765, 7765, -1), (7767, 7767, -1), (7769, 7769, -1), (7771, 7771, -1), (7773, 7773, -1), (7775, 7775, -1), (7777, 7777, -1), (7779, 7779, -1), (7781, 7781, -1), (7783, 7783, -1), (7785, 7785, -1), (7787, 7787, -1), (7789, 7789, -1), (7791, 7791, -1), (7793, 7793, -1), (7795, 7795, -1), (7797, 7797, -1), (7799, 7799, -1), (7801, 7801, -1), (7803, 7803, -1), (7805, 7805, -1), (7807, 7807, -1), (7809, 7809, -1), (7811, 7811, -1), (7813, 7813, -1), (7815, 7815, -1), (7817, 7817, -1), (7819, 7819, -1), (7821, 7821, -1), (7823, 7823, -1), (7825, 7825, -1), (7827, 7827, -1), (7829, 7829, -1), (7841, 7841, -1), (7843, 7843, -1), (7845, 7845, -1), (7847, 7847, -1), (7849, 7849, -1), (7851, 7851, -1), (7853, 7853, -1), (7855, 7855, -1), (7857, 7857, -1), (7859, 7859, -1), (7861, 7861, -1), (7863, 7863, -1), (7865, 7865, -1), (7867, 7867, -1), (7869, 7869, -1), (7871, 7871, -1), (7873, 7873, -1), (7875, 7875, -1), (7877, 7877, -1), (7879, 7879, -1), (7881, 7881, -1), (7883, 7883, -1), (7885, 7885, -1), (7887, 7887, -1), (7889, 7889, -1), (7891, 7891, -1), (7893, 7893, -1), (7895, 7895, -1), (7897, 7897, -1), (7899, 7899, -1), (7901, 7901, -1), (7903, 7903, -1), (7905, 7905, -1), (7907, 7907, -1), (7909, 7909, -1), (7911, 7911, -1), (7913, 7913, -1), (7915, 7915, -1), (7917, 7917, -1), (7919, 7919, -1), (7921, 7921, -1), (7923, 7923, -1), (7925, 7925, -1), (7927, 7927, -1), (7929, 7929, -1), (7931, 7931, -1), (7933, 7933, -1), (7935, 7935, -1), (8580, 8580, -1), (11361, 11361, -1), (11368, 11368, -1), (11370, 11370, -1), (11372, 11372, -1), (11379, 11379, -1), (11382, 11382, -1), (11393, 11393, -1), (11395, 11395, -1), (11397, 11397, -1), (11399, 11399, -1), (11401, 11401, -1), (11403, 11403, -1), (11405, 11405, -1), (11407, 11407, -1), (11409, 11409, -1), (11411, 11411, -1), (11413, 11413, -1), (11415, 11415, -1), (11417, 11417, -1), (11419, 11419, -1), (11421, 11421, -1), (11423, 11423, -1), (11425, 11425, -1), (11427, 11427, -1), (11429, 11429, -1), (11431, 11431, -1), (11433, 11433, -1), (11435, 11435, -1), (11437, 11437, -1), (11439, 11439, -1), (11441, 11441, -1), (11443, 11443, -1), (11445, 11445, -1), (11447, 11447, -1), (11449, 11449, -1), (11451, 11451, -1), (11453, 11453, -1), (11455, 11455, -1), (11457, 11457, -1), (11459, 11459, -1), (11461, 11461, -1), (11463, 11463, -1), (11465, 11465, -1), (11467, 11467, -1), (11469, 11469, -1), (11471, 11471, -1), (11473, 11473, -1), (11475, 11475, -1), (11477, 11477, -1), (11479, 11479, -1), (11481, 11481, -1), (11483, 11483, -1), (11485, 11485, -1), (11487, 11487, -1), (11489, 11489, -1), (11491, 11491, -1), (11500, 11500, -1), (11502, 11502, -1), (11507, 11507, -1), (42561, 42561, -1), (42563, 42563, -1), (42565, 42565, -1), (42567, 42567, -1), (42569, 42569, -1), (42571, 42571, -1), (42573, 42573, -1), (42575, 42575, -1), (42577, 42577, -1), (42579, 42579, -1), (42581, 42581, -1), (42583, 42583, -1), (42585, 42585, -1), (42587, 42587, -1), (42589, 42589, -1), (42591, 42591, -1), (42593, 42593, -1), (42595, 42595, -1), (42597, 42597, -1), (42599, 42599, -1), (42601, 42601, -1), (42603, 42603, -1), (42605, 42605, -1), (42625, 42625, -1), (42627, 42627, -1), (42629, 42629, -1), (42631, 42631, -1), (42633, 42633, -1), (42635, 42635, -1), (42637, 42637, -1), (42639, 42639, -1), (42641, 42641, -1), (42643, 42643, -1), (42645, 42645, -1), (42647, 42647, -1), (42649, 42649, -1), (42651, 42651, -1), (42787, 42787, -1), (42789, 42789, -1), (42791, 42791, -1), (42793, 42793, -1), (42795, 42795, -1), (42797, 42797, -1), (42799, 42799, -1), (42803, 42803, -1), (42805, 42805, -1), (42807, 42807, -1), (42809, 42809, -1), (42811, 42811, -1), (42813, 42813, -1), (42815, 42815, -1), (42817, 42817, -1), (42819, 42819, -1), (42821, 42821, -1), (42823, 42823, -1), (42825, 42825, -1), (42827, 42827, -1), (42829, 42829, -1), (42831, 42831, -1), (42833, 42833, -1), (42835, 42835, -1), (42837, 42837, -1), (42839, 42839, -1), (42841, 42841, -1), (42843, 42843, -1), (42845, 42845, -1), (42847, 42847, -1), (42849, 42849, -1), (42851, 42851, -1), (42853, 42853, -1), (42855, 42855, -1), (42857, 42857, -1), (42859, 42859, -1), (42861, 42861, -1), (42863, 42863, -1), (42874, 42874, -1), (42876, 42876, -1), (42879, 42879, -1), (42881, 42881, -1), (42883, 42883, -1), (42885, 42885, -1), (42887, 42887, -1), (42892, 42892, -1), (42897, 42897, -1), (42899, 42899, -1), (42903, 42903, -1), (42905, 42905, -1), (42907, 42907, -1), (42909, 42909, -1), (42911, 42911, -1), (42913, 42913, -1), (42915, 42915, -1), (42917, 42917, -1), (42919, 42919, -1), (42921, 42921, -1), (42933, 42933, -1), (42935, 42935, -1), (42937, 42937, -1), (42939, 42939, -1), (42941, 42941, -1), (42943, 42943, -1), (42945, 42945, -1), (42947, 42947, -1), (42952, 42952, -1), (42954, 42954, -1), (42957, 42957, -1), (42961, 42961, -1), (42967, 42967, -1), (42969, 42969, -1), (42971, 42971, -1), (42998, 42998, -1), (305, 305, -232), (383, 383, -300), (384, 384, 195), (405, 405, 97), (410, 410, 163), (411, 411, 42561), (414, 414, 130), (891, 893, 130), (447, 447, 56), (452, 452, 1), (455, 455, 1), (458, 458, 1), (497, 497, 1), (453, 453, 0), (456, 456, 0), (459, 459, 0), (498, 498, 0), (4304, 4346, 0), (4349, 4351, 0), (477, 477, -79), (575, 576, 10815), (592, 592, 10783), (593, 593, 10780), (594, 594, 10782), (595, 595, -210), (596, 596, -206), (598, 599, -205), (608, 608, -205), (601, 601, -202), (603, 603, -203), (604, 604, 42319), (609, 609, 42315), (611, 611, -207), (612, 612, 42343), (613, 613, 42280), (614, 614, 42308), (618, 618, 42308), (616, 616, -209), (617, 617, -211), (623, 623, -211), (619, 619, 10743), (620, 620, 42305), (625, 625, 10749), (626, 626, -213), (629, 629, -214), (637, 637, 10727), (640, 640, -218), (643, 643, -218), (648, 648, -218), (642, 642, 42307), (647, 647, 42282), (649, 649, -69), (650, 651, -217), (652, 652, -71), (658, 658, -219), (669, 669, 42261), (670, 670, 42258), (837, 837, 84), (940, 940, -38), (941, 943, -37), (962, 962, -31), (972, 972, -64), (68800, 68850, -64), (973, 974, -63), (976, 976, -62), (977, 977, -57), (981, 981, -47), (982, 982, -54), (983, 983, -8), (5112, 5117, -8), (1008, 1008, -86), (1009, 1009, -80), (1104, 1119, -80), (1010, 1010, 7), (8165, 8165, 7), (1011, 1011, -116), (1013, 1013, -96), (1231, 1231, -15), (1377, 1414, -48), (11312, 11359, -48), (7296, 7296, -6254), (7297, 7297, -6253), (7298, 7298, -6244), (7299, 7300, -6242), (7301, 7301, -6243), (7302, 7302, -6236), (7303, 7303, -6181), (7304, 7304, 35266), (7545, 7545, 35332), (7549, 7549, 3814), (7566, 7566, 35384), (7835, 7835, -59), (7936, 7943, 8), (7952, 7957, 8), (7968, 7975, 8), (7984, 7991, 8), (8000, 8005, 8), (8017, 8017, 8), (8019, 8019, 8), (8021, 8021, 8), (8023, 8023, 8), (8032, 8039, 8), (8064, 8071, 8), (8080, 8087, 8), (8096, 8103, 8), (8112, 8113, 8), (8144, 8145, 8), (8160, 8161, 8), (8048, 8049, 74), (8050, 8053, 86), (8054, 8055, 100), (8056, 8057, 128), (8058, 8059, 112), (8060, 8061, 126), (8115, 8115, 9), (8131, 8131, 9), (8179, 8179, 9), (8126, 8126, -7205), (8526, 8526, -28), (8560, 8575, -16), (9424, 9449, -26), (11365, 11365, -10795), (11366, 11366, -10792), (11520, 11557, -7264), (11559, 11559, -7264), (11565, 11565, -7264), (42900, 42900, 48), (43859, 43859, -928), (43888, 43967, -38864), (66600, 66639, -40), (66776, 66811, -40), (66967, 66977, -39), (66979, 66993, -39), (66995, 67001, -39), (67003, 67004, -39), (125218, 125251, -34)], null, length: 0x110000);

final _upperCaseList = SparseList([(97, 122, -32), (224, 246, -32), (248, 254, -32), (945, 961, -32), (963, 971, -32), (1072, 1103, -32), (65345, 65370, -32), (68976, 68997, -32), (71872, 71903, -32), (93792, 93823, -32), (181, 181, 743), (255, 255, 121), (257, 257, -1), (259, 259, -1), (261, 261, -1), (263, 263, -1), (265, 265, -1), (267, 267, -1), (269, 269, -1), (271, 271, -1), (273, 273, -1), (275, 275, -1), (277, 277, -1), (279, 279, -1), (281, 281, -1), (283, 283, -1), (285, 285, -1), (287, 287, -1), (289, 289, -1), (291, 291, -1), (293, 293, -1), (295, 295, -1), (297, 297, -1), (299, 299, -1), (301, 301, -1), (303, 303, -1), (307, 307, -1), (309, 309, -1), (311, 311, -1), (314, 314, -1), (316, 316, -1), (318, 318, -1), (320, 320, -1), (322, 322, -1), (324, 324, -1), (326, 326, -1), (328, 328, -1), (331, 331, -1), (333, 333, -1), (335, 335, -1), (337, 337, -1), (339, 339, -1), (341, 341, -1), (343, 343, -1), (345, 345, -1), (347, 347, -1), (349, 349, -1), (351, 351, -1), (353, 353, -1), (355, 355, -1), (357, 357, -1), (359, 359, -1), (361, 361, -1), (363, 363, -1), (365, 365, -1), (367, 367, -1), (369, 369, -1), (371, 371, -1), (373, 373, -1), (375, 375, -1), (378, 378, -1), (380, 380, -1), (382, 382, -1), (387, 387, -1), (389, 389, -1), (392, 392, -1), (396, 396, -1), (402, 402, -1), (409, 409, -1), (417, 417, -1), (419, 419, -1), (421, 421, -1), (424, 424, -1), (429, 429, -1), (432, 432, -1), (436, 436, -1), (438, 438, -1), (441, 441, -1), (445, 445, -1), (453, 453, -1), (456, 456, -1), (459, 459, -1), (462, 462, -1), (464, 464, -1), (466, 466, -1), (468, 468, -1), (470, 470, -1), (472, 472, -1), (474, 474, -1), (476, 476, -1), (479, 479, -1), (481, 481, -1), (483, 483, -1), (485, 485, -1), (487, 487, -1), (489, 489, -1), (491, 491, -1), (493, 493, -1), (495, 495, -1), (498, 498, -1), (501, 501, -1), (505, 505, -1), (507, 507, -1), (509, 509, -1), (511, 511, -1), (513, 513, -1), (515, 515, -1), (517, 517, -1), (519, 519, -1), (521, 521, -1), (523, 523, -1), (525, 525, -1), (527, 527, -1), (529, 529, -1), (531, 531, -1), (533, 533, -1), (535, 535, -1), (537, 537, -1), (539, 539, -1), (541, 541, -1), (543, 543, -1), (547, 547, -1), (549, 549, -1), (551, 551, -1), (553, 553, -1), (555, 555, -1), (557, 557, -1), (559, 559, -1), (561, 561, -1), (563, 563, -1), (572, 572, -1), (578, 578, -1), (583, 583, -1), (585, 585, -1), (587, 587, -1), (589, 589, -1), (591, 591, -1), (881, 881, -1), (883, 883, -1), (887, 887, -1), (985, 985, -1), (987, 987, -1), (989, 989, -1), (991, 991, -1), (993, 993, -1), (995, 995, -1), (997, 997, -1), (999, 999, -1), (1001, 1001, -1), (1003, 1003, -1), (1005, 1005, -1), (1007, 1007, -1), (1016, 1016, -1), (1019, 1019, -1), (1121, 1121, -1), (1123, 1123, -1), (1125, 1125, -1), (1127, 1127, -1), (1129, 1129, -1), (1131, 1131, -1), (1133, 1133, -1), (1135, 1135, -1), (1137, 1137, -1), (1139, 1139, -1), (1141, 1141, -1), (1143, 1143, -1), (1145, 1145, -1), (1147, 1147, -1), (1149, 1149, -1), (1151, 1151, -1), (1153, 1153, -1), (1163, 1163, -1), (1165, 1165, -1), (1167, 1167, -1), (1169, 1169, -1), (1171, 1171, -1), (1173, 1173, -1), (1175, 1175, -1), (1177, 1177, -1), (1179, 1179, -1), (1181, 1181, -1), (1183, 1183, -1), (1185, 1185, -1), (1187, 1187, -1), (1189, 1189, -1), (1191, 1191, -1), (1193, 1193, -1), (1195, 1195, -1), (1197, 1197, -1), (1199, 1199, -1), (1201, 1201, -1), (1203, 1203, -1), (1205, 1205, -1), (1207, 1207, -1), (1209, 1209, -1), (1211, 1211, -1), (1213, 1213, -1), (1215, 1215, -1), (1218, 1218, -1), (1220, 1220, -1), (1222, 1222, -1), (1224, 1224, -1), (1226, 1226, -1), (1228, 1228, -1), (1230, 1230, -1), (1233, 1233, -1), (1235, 1235, -1), (1237, 1237, -1), (1239, 1239, -1), (1241, 1241, -1), (1243, 1243, -1), (1245, 1245, -1), (1247, 1247, -1), (1249, 1249, -1), (1251, 1251, -1), (1253, 1253, -1), (1255, 1255, -1), (1257, 1257, -1), (1259, 1259, -1), (1261, 1261, -1), (1263, 1263, -1), (1265, 1265, -1), (1267, 1267, -1), (1269, 1269, -1), (1271, 1271, -1), (1273, 1273, -1), (1275, 1275, -1), (1277, 1277, -1), (1279, 1279, -1), (1281, 1281, -1), (1283, 1283, -1), (1285, 1285, -1), (1287, 1287, -1), (1289, 1289, -1), (1291, 1291, -1), (1293, 1293, -1), (1295, 1295, -1), (1297, 1297, -1), (1299, 1299, -1), (1301, 1301, -1), (1303, 1303, -1), (1305, 1305, -1), (1307, 1307, -1), (1309, 1309, -1), (1311, 1311, -1), (1313, 1313, -1), (1315, 1315, -1), (1317, 1317, -1), (1319, 1319, -1), (1321, 1321, -1), (1323, 1323, -1), (1325, 1325, -1), (1327, 1327, -1), (7306, 7306, -1), (7681, 7681, -1), (7683, 7683, -1), (7685, 7685, -1), (7687, 7687, -1), (7689, 7689, -1), (7691, 7691, -1), (7693, 7693, -1), (7695, 7695, -1), (7697, 7697, -1), (7699, 7699, -1), (7701, 7701, -1), (7703, 7703, -1), (7705, 7705, -1), (7707, 7707, -1), (7709, 7709, -1), (7711, 7711, -1), (7713, 7713, -1), (7715, 7715, -1), (7717, 7717, -1), (7719, 7719, -1), (7721, 7721, -1), (7723, 7723, -1), (7725, 7725, -1), (7727, 7727, -1), (7729, 7729, -1), (7731, 7731, -1), (7733, 7733, -1), (7735, 7735, -1), (7737, 7737, -1), (7739, 7739, -1), (7741, 7741, -1), (7743, 7743, -1), (7745, 7745, -1), (7747, 7747, -1), (7749, 7749, -1), (7751, 7751, -1), (7753, 7753, -1), (7755, 7755, -1), (7757, 7757, -1), (7759, 7759, -1), (7761, 7761, -1), (7763, 7763, -1), (7765, 7765, -1), (7767, 7767, -1), (7769, 7769, -1), (7771, 7771, -1), (7773, 7773, -1), (7775, 7775, -1), (7777, 7777, -1), (7779, 7779, -1), (7781, 7781, -1), (7783, 7783, -1), (7785, 7785, -1), (7787, 7787, -1), (7789, 7789, -1), (7791, 7791, -1), (7793, 7793, -1), (7795, 7795, -1), (7797, 7797, -1), (7799, 7799, -1), (7801, 7801, -1), (7803, 7803, -1), (7805, 7805, -1), (7807, 7807, -1), (7809, 7809, -1), (7811, 7811, -1), (7813, 7813, -1), (7815, 7815, -1), (7817, 7817, -1), (7819, 7819, -1), (7821, 7821, -1), (7823, 7823, -1), (7825, 7825, -1), (7827, 7827, -1), (7829, 7829, -1), (7841, 7841, -1), (7843, 7843, -1), (7845, 7845, -1), (7847, 7847, -1), (7849, 7849, -1), (7851, 7851, -1), (7853, 7853, -1), (7855, 7855, -1), (7857, 7857, -1), (7859, 7859, -1), (7861, 7861, -1), (7863, 7863, -1), (7865, 7865, -1), (7867, 7867, -1), (7869, 7869, -1), (7871, 7871, -1), (7873, 7873, -1), (7875, 7875, -1), (7877, 7877, -1), (7879, 7879, -1), (7881, 7881, -1), (7883, 7883, -1), (7885, 7885, -1), (7887, 7887, -1), (7889, 7889, -1), (7891, 7891, -1), (7893, 7893, -1), (7895, 7895, -1), (7897, 7897, -1), (7899, 7899, -1), (7901, 7901, -1), (7903, 7903, -1), (7905, 7905, -1), (7907, 7907, -1), (7909, 7909, -1), (7911, 7911, -1), (7913, 7913, -1), (7915, 7915, -1), (7917, 7917, -1), (7919, 7919, -1), (7921, 7921, -1), (7923, 7923, -1), (7925, 7925, -1), (7927, 7927, -1), (7929, 7929, -1), (7931, 7931, -1), (7933, 7933, -1), (7935, 7935, -1), (8580, 8580, -1), (11361, 11361, -1), (11368, 11368, -1), (11370, 11370, -1), (11372, 11372, -1), (11379, 11379, -1), (11382, 11382, -1), (11393, 11393, -1), (11395, 11395, -1), (11397, 11397, -1), (11399, 11399, -1), (11401, 11401, -1), (11403, 11403, -1), (11405, 11405, -1), (11407, 11407, -1), (11409, 11409, -1), (11411, 11411, -1), (11413, 11413, -1), (11415, 11415, -1), (11417, 11417, -1), (11419, 11419, -1), (11421, 11421, -1), (11423, 11423, -1), (11425, 11425, -1), (11427, 11427, -1), (11429, 11429, -1), (11431, 11431, -1), (11433, 11433, -1), (11435, 11435, -1), (11437, 11437, -1), (11439, 11439, -1), (11441, 11441, -1), (11443, 11443, -1), (11445, 11445, -1), (11447, 11447, -1), (11449, 11449, -1), (11451, 11451, -1), (11453, 11453, -1), (11455, 11455, -1), (11457, 11457, -1), (11459, 11459, -1), (11461, 11461, -1), (11463, 11463, -1), (11465, 11465, -1), (11467, 11467, -1), (11469, 11469, -1), (11471, 11471, -1), (11473, 11473, -1), (11475, 11475, -1), (11477, 11477, -1), (11479, 11479, -1), (11481, 11481, -1), (11483, 11483, -1), (11485, 11485, -1), (11487, 11487, -1), (11489, 11489, -1), (11491, 11491, -1), (11500, 11500, -1), (11502, 11502, -1), (11507, 11507, -1), (42561, 42561, -1), (42563, 42563, -1), (42565, 42565, -1), (42567, 42567, -1), (42569, 42569, -1), (42571, 42571, -1), (42573, 42573, -1), (42575, 42575, -1), (42577, 42577, -1), (42579, 42579, -1), (42581, 42581, -1), (42583, 42583, -1), (42585, 42585, -1), (42587, 42587, -1), (42589, 42589, -1), (42591, 42591, -1), (42593, 42593, -1), (42595, 42595, -1), (42597, 42597, -1), (42599, 42599, -1), (42601, 42601, -1), (42603, 42603, -1), (42605, 42605, -1), (42625, 42625, -1), (42627, 42627, -1), (42629, 42629, -1), (42631, 42631, -1), (42633, 42633, -1), (42635, 42635, -1), (42637, 42637, -1), (42639, 42639, -1), (42641, 42641, -1), (42643, 42643, -1), (42645, 42645, -1), (42647, 42647, -1), (42649, 42649, -1), (42651, 42651, -1), (42787, 42787, -1), (42789, 42789, -1), (42791, 42791, -1), (42793, 42793, -1), (42795, 42795, -1), (42797, 42797, -1), (42799, 42799, -1), (42803, 42803, -1), (42805, 42805, -1), (42807, 42807, -1), (42809, 42809, -1), (42811, 42811, -1), (42813, 42813, -1), (42815, 42815, -1), (42817, 42817, -1), (42819, 42819, -1), (42821, 42821, -1), (42823, 42823, -1), (42825, 42825, -1), (42827, 42827, -1), (42829, 42829, -1), (42831, 42831, -1), (42833, 42833, -1), (42835, 42835, -1), (42837, 42837, -1), (42839, 42839, -1), (42841, 42841, -1), (42843, 42843, -1), (42845, 42845, -1), (42847, 42847, -1), (42849, 42849, -1), (42851, 42851, -1), (42853, 42853, -1), (42855, 42855, -1), (42857, 42857, -1), (42859, 42859, -1), (42861, 42861, -1), (42863, 42863, -1), (42874, 42874, -1), (42876, 42876, -1), (42879, 42879, -1), (42881, 42881, -1), (42883, 42883, -1), (42885, 42885, -1), (42887, 42887, -1), (42892, 42892, -1), (42897, 42897, -1), (42899, 42899, -1), (42903, 42903, -1), (42905, 42905, -1), (42907, 42907, -1), (42909, 42909, -1), (42911, 42911, -1), (42913, 42913, -1), (42915, 42915, -1), (42917, 42917, -1), (42919, 42919, -1), (42921, 42921, -1), (42933, 42933, -1), (42935, 42935, -1), (42937, 42937, -1), (42939, 42939, -1), (42941, 42941, -1), (42943, 42943, -1), (42945, 42945, -1), (42947, 42947, -1), (42952, 42952, -1), (42954, 42954, -1), (42957, 42957, -1), (42961, 42961, -1), (42967, 42967, -1), (42969, 42969, -1), (42971, 42971, -1), (42998, 42998, -1), (305, 305, -232), (383, 383, -300), (384, 384, 195), (405, 405, 97), (410, 410, 163), (411, 411, 42561), (414, 414, 130), (891, 893, 130), (447, 447, 56), (454, 454, -2), (457, 457, -2), (460, 460, -2), (499, 499, -2), (477, 477, -79), (575, 576, 10815), (592, 592, 10783), (593, 593, 10780), (594, 594, 10782), (595, 595, -210), (596, 596, -206), (598, 599, -205), (608, 608, -205), (601, 601, -202), (603, 603, -203), (604, 604, 42319), (609, 609, 42315), (611, 611, -207), (612, 612, 42343), (613, 613, 42280), (614, 614, 42308), (618, 618, 42308), (616, 616, -209), (617, 617, -211), (623, 623, -211), (619, 619, 10743), (620, 620, 42305), (625, 625, 10749), (626, 626, -213), (629, 629, -214), (637, 637, 10727), (640, 640, -218), (643, 643, -218), (648, 648, -218), (642, 642, 42307), (647, 647, 42282), (649, 649, -69), (650, 651, -217), (652, 652, -71), (658, 658, -219), (669, 669, 42261), (670, 670, 42258), (837, 837, 84), (940, 940, -38), (941, 943, -37), (962, 962, -31), (972, 972, -64), (68800, 68850, -64), (973, 974, -63), (976, 976, -62), (977, 977, -57), (981, 981, -47), (982, 982, -54), (983, 983, -8), (5112, 5117, -8), (1008, 1008, -86), (1009, 1009, -80), (1104, 1119, -80), (1010, 1010, 7), (8165, 8165, 7), (1011, 1011, -116), (1013, 1013, -96), (1231, 1231, -15), (1377, 1414, -48), (11312, 11359, -48), (4304, 4346, 3008), (4349, 4351, 3008), (7296, 7296, -6254), (7297, 7297, -6253), (7298, 7298, -6244), (7299, 7300, -6242), (7301, 7301, -6243), (7302, 7302, -6236), (7303, 7303, -6181), (7304, 7304, 35266), (7545, 7545, 35332), (7549, 7549, 3814), (7566, 7566, 35384), (7835, 7835, -59), (7936, 7943, 8), (7952, 7957, 8), (7968, 7975, 8), (7984, 7991, 8), (8000, 8005, 8), (8017, 8017, 8), (8019, 8019, 8), (8021, 8021, 8), (8023, 8023, 8), (8032, 8039, 8), (8064, 8071, 8), (8080, 8087, 8), (8096, 8103, 8), (8112, 8113, 8), (8144, 8145, 8), (8160, 8161, 8), (8048, 8049, 74), (8050, 8053, 86), (8054, 8055, 100), (8056, 8057, 128), (8058, 8059, 112), (8060, 8061, 126), (8115, 8115, 9), (8131, 8131, 9), (8179, 8179, 9), (8126, 8126, -7205), (8526, 8526, -28), (8560, 8575, -16), (9424, 9449, -26), (11365, 11365, -10795), (11366, 11366, -10792), (11520, 11557, -7264), (11559, 11559, -7264), (11565, 11565, -7264), (42900, 42900, 48), (43859, 43859, -928), (43888, 43967, -38864), (66600, 66639, -40), (66776, 66811, -40), (66967, 66977, -39), (66979, 66993, -39), (66995, 67001, -39), (67003, 67004, -39), (125218, 125251, -34)], null, length: 0x110000);

final SparseList<int> _generalCategories = _unpack([[], [0, 31, 96, 32], [173, 0, 1363, 5, 23, 0, 193, 0, 50, 0, 385, 1, 81, 0, 3884, 0, 2045, 4, 27, 4, 50, 4, 2, 9, 56976, 0, 250, 2, 4290, 0, 16, 0, 9059, 15, 34913, 3, 5328, 7, 798343, 0, 31, 95], [57344, 6399, 919297, 65533, 3, 65533], [55296, 2047], [97, 25, 59, 0, 42, 23, 2, 7, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 3, 0, 2, 0, 2, 2, 3, 0, 2, 0, 3, 0, 4, 1, 5, 0, 3, 0, 4, 2, 3, 0, 3, 0, 2, 0, 2, 0, 3, 0, 2, 1, 2, 0, 3, 0, 4, 0, 2, 0, 3, 1, 3, 2, 7, 0, 3, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 3, 0, 2, 0, 4, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 6, 3, 0, 3, 1, 2, 0, 5, 0, 2, 0, 2, 0, 2, 0, 2, 68, 2, 26, 194, 0, 2, 0, 4, 0, 4, 2, 19, 0, 28, 34, 2, 1, 4, 2, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 4, 2, 0, 3, 0, 3, 1, 52, 47, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 10, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 49, 40, 2888, 42, 3, 2, 761, 5, 2179, 8, 2, 0, 118, 43, 64, 12, 2, 33, 103, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 8, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 8, 9, 5, 11, 7, 9, 7, 9, 5, 11, 7, 9, 7, 9, 13, 3, 7, 9, 7, 9, 7, 9, 4, 2, 1, 7, 0, 4, 2, 2, 1, 9, 3, 3, 1, 9, 7, 11, 2, 2, 1, 275, 0, 4, 1, 4, 0, 28, 0, 5, 0, 5, 0, 3, 1, 9, 3, 5, 0, 54, 0, 2732, 47, 2, 0, 4, 1, 2, 0, 2, 0, 2, 0, 5, 0, 2, 1, 2, 5, 6, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 8, 0, 2, 0, 5, 0, 13, 37, 2, 0, 6, 0, 30996, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 20, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 136, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 2, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 7, 2, 0, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 5, 0, 2, 0, 3, 0, 2, 2, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 6, 0, 6, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 5, 0, 2, 0, 3, 0, 4, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 27, 0, 4, 0, 822, 42, 6, 8, 8, 79, 20289, 6, 13, 4, 1066, 25, 1230, 39, 137, 35, 156, 10, 2, 14, 2, 6, 2, 1, 1796, 50, 126, 21, 2875, 31, 21889, 31, 26011, 25, 27, 6, 2, 17, 27, 25, 27, 3, 2, 0, 2, 6, 2, 10, 27, 25, 27, 25, 27, 25, 27, 25, 27, 25, 27, 25, 27, 25, 27, 25, 27, 27, 29, 24, 2, 5, 27, 24, 2, 5, 27, 24, 2, 5, 27, 24, 2, 5, 27, 24, 2, 5, 2, 0, 1845, 9, 2, 19, 7, 5, 2552, 33], [688, 17, 5, 11, 15, 4, 8, 0, 2, 0, 134, 0, 6, 0, 479, 0, 231, 0, 165, 1, 270, 1, 5, 0, 32, 0, 10, 0, 4, 0, 161, 0, 168, 0, 1237, 0, 128, 0, 566, 0, 1755, 0, 108, 0, 612, 0, 465, 5, 175, 62, 14, 0, 35, 36, 690, 0, 14, 0, 17, 12, 3040, 1, 242, 0, 192, 0, 470, 0, 44, 4, 6, 0, 98, 1, 94, 2, 28439, 0, 1251, 5, 271, 0, 115, 0, 29, 1, 122, 8, 81, 0, 24, 0, 106, 2, 4, 1, 470, 0, 23, 0, 138, 0, 109, 0, 22, 1, 104, 3, 10, 0, 21511, 0, 46, 1, 2017, 5, 2, 41, 2, 8, 1428, 0, 33, 0, 24017, 3, 509, 2, 41, 1, 551, 12, 65, 1, 2, 0, 16397, 3, 2, 6, 2, 1, 12338, 61, 202, 6, 942, 0, 1120, 0], [170, 0, 16, 0, 257, 0, 5, 3, 209, 0, 828, 26, 5, 3, 46, 31, 2, 9, 36, 1, 2, 98, 2, 0, 25, 1, 11, 2, 3, 0, 17, 0, 2, 29, 30, 88, 12, 0, 25, 32, 22, 21, 43, 24, 8, 10, 6, 23, 2, 5, 18, 40, 60, 53, 4, 0, 19, 0, 8, 9, 17, 14, 5, 7, 3, 1, 3, 21, 2, 6, 2, 0, 4, 3, 4, 0, 17, 0, 14, 1, 2, 2, 15, 1, 11, 0, 9, 5, 5, 1, 3, 21, 2, 6, 2, 1, 2, 1, 2, 1, 32, 3, 2, 0, 20, 2, 17, 8, 2, 2, 2, 21, 2, 6, 2, 1, 2, 4, 4, 0, 19, 0, 16, 1, 24, 0, 12, 7, 3, 1, 3, 21, 2, 6, 2, 1, 2, 4, 4, 0, 31, 1, 2, 2, 16, 0, 18, 0, 2, 5, 4, 2, 2, 3, 4, 1, 2, 0, 2, 1, 4, 1, 4, 2, 4, 11, 23, 0, 53, 7, 2, 2, 2, 22, 2, 15, 4, 0, 27, 2, 3, 0, 3, 1, 31, 0, 5, 7, 2, 2, 2, 22, 2, 9, 2, 4, 4, 0, 32, 1, 2, 1, 16, 1, 18, 8, 2, 2, 2, 40, 3, 0, 17, 0, 6, 2, 9, 2, 25, 5, 6, 17, 4, 23, 2, 8, 2, 0, 3, 6, 59, 47, 2, 1, 13, 5, 60, 1, 2, 0, 2, 4, 2, 23, 2, 0, 2, 9, 2, 1, 10, 0, 3, 4, 24, 3, 33, 0, 64, 7, 2, 35, 28, 4, 116, 42, 21, 0, 17, 5, 5, 3, 4, 0, 4, 1, 8, 2, 5, 12, 13, 0, 114, 328, 2, 3, 3, 6, 2, 0, 2, 3, 3, 40, 2, 3, 3, 32, 2, 3, 3, 6, 2, 0, 2, 3, 3, 14, 2, 56, 2, 3, 3, 66, 38, 15, 114, 619, 3, 16, 2, 25, 6, 74, 7, 7, 8, 17, 14, 18, 15, 17, 15, 12, 2, 2, 16, 51, 41, 0, 68, 34, 2, 52, 8, 4, 3, 33, 2, 0, 6, 69, 11, 30, 50, 29, 3, 4, 12, 43, 5, 25, 55, 22, 10, 52, 177, 46, 18, 7, 55, 29, 14, 1, 11, 43, 27, 35, 42, 2, 11, 29, 114, 3, 2, 5, 2, 1, 4, 0, 1083, 3, 3064, 55, 25, 22, 10, 6, 2, 6, 2, 6, 2, 6, 2, 6, 2, 6, 2, 6, 2, 6, 552, 0, 54, 0, 5, 85, 9, 0, 2, 89, 5, 0, 6, 42, 2, 93, 18, 31, 49, 15, 513, 6591, 65, 21012, 2, 1142, 68, 39, 9, 267, 5, 15, 11, 1, 67, 0, 50, 69, 170, 0, 104, 0, 4, 6, 2, 2, 2, 3, 2, 22, 30, 51, 15, 49, 63, 5, 4, 0, 2, 1, 12, 27, 11, 22, 26, 28, 8, 46, 46, 4, 3, 8, 11, 4, 2, 40, 24, 2, 2, 7, 21, 15, 2, 5, 4, 0, 4, 49, 2, 0, 4, 1, 3, 4, 3, 0, 2, 0, 25, 1, 4, 10, 8, 0, 15, 5, 3, 5, 3, 5, 10, 6, 2, 6, 146, 34, 30, 11171, 13, 22, 5, 48, 8453, 365, 3, 105, 68, 0, 2, 9, 2, 12, 2, 4, 2, 0, 2, 1, 2, 1, 2, 107, 34, 362, 19, 63, 3, 53, 41, 11, 117, 4, 2, 134, 106, 9, 2, 44, 3, 30, 4, 5, 3, 5, 3, 5, 3, 2, 36, 11, 2, 25, 2, 18, 2, 1, 2, 14, 3, 13, 35, 122, 390, 28, 4, 48, 48, 31, 14, 19, 2, 7, 7, 37, 11, 29, 3, 35, 5, 7, 129, 77, 99, 39, 9, 51, 93, 51, 13, 310, 10, 21, 11, 7, 153, 5, 3, 0, 2, 43, 2, 1, 4, 0, 3, 22, 11, 22, 10, 30, 66, 18, 2, 1, 11, 21, 11, 25, 71, 55, 7, 1, 65, 0, 16, 3, 2, 2, 2, 28, 43, 28, 4, 28, 36, 7, 2, 27, 28, 53, 11, 21, 11, 18, 14, 17, 111, 72, 184, 35, 39, 3, 2, 0, 305, 41, 7, 1, 17, 2, 60, 28, 11, 0, 9, 21, 43, 17, 47, 20, 28, 22, 13, 52, 58, 1, 3, 0, 14, 44, 33, 24, 27, 35, 30, 0, 3, 0, 9, 34, 4, 0, 13, 47, 15, 3, 22, 0, 2, 0, 36, 17, 2, 24, 20, 1, 64, 6, 2, 0, 2, 3, 2, 14, 2, 9, 8, 46, 39, 7, 3, 1, 3, 21, 2, 6, 2, 1, 2, 4, 4, 0, 19, 0, 13, 4, 31, 9, 2, 0, 3, 0, 2, 37, 2, 0, 26, 0, 2, 0, 45, 52, 19, 3, 21, 2, 31, 47, 21, 1, 2, 0, 185, 46, 42, 3, 37, 47, 21, 0, 60, 42, 14, 0, 72, 26, 38, 6, 186, 43, 212, 7, 3, 0, 3, 7, 2, 1, 2, 23, 16, 0, 2, 0, 95, 7, 3, 38, 17, 0, 2, 0, 29, 0, 11, 39, 8, 0, 22, 0, 12, 45, 20, 0, 19, 72, 200, 32, 32, 8, 2, 36, 18, 0, 50, 29, 113, 6, 2, 1, 2, 37, 22, 0, 26, 5, 2, 1, 2, 31, 15, 0, 328, 18, 16, 0, 2, 12, 2, 33, 125, 0, 80, 921, 231, 195, 2637, 96, 16, 1071, 18, 5, 26, 3994, 6, 582, 6842, 29, 1763, 568, 8, 30, 18, 78, 18, 29, 19, 47, 52, 20, 6, 18, 436, 39, 406, 74, 6, 0, 176, 6135, 9, 1237, 42, 9, 8952, 290, 16, 0, 30, 2, 3, 0, 15, 3, 9, 395, 2309, 106, 6, 12, 4, 8, 8, 9, 8817, 0, 502, 44, 34, 0, 322, 29, 19, 43, 485, 26, 230, 29, 3, 0, 496, 6, 2, 3, 2, 1, 2, 14, 2, 196, 1340, 3, 2, 26, 2, 1, 2, 0, 3, 0, 2, 9, 2, 3, 2, 0, 2, 0, 7, 0, 5, 0, 2, 0, 2, 0, 2, 2, 2, 1, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 3, 3, 2, 6, 2, 3, 2, 3, 2, 0, 2, 9, 2, 16, 6, 2, 2, 4, 2, 16, 4421, 42719, 33, 4153, 7, 221, 3, 5761, 15, 7472, 16, 621, 2467, 541, 1507, 4938, 6, 4191], [453, 0, 3, 0, 3, 0, 39, 0, 7574, 7, 9, 7, 9, 7, 13, 0, 16, 0, 48, 0], [65, 25, 102, 22, 2, 6, 34, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 2, 0, 4, 1, 2, 0, 2, 1, 2, 2, 3, 3, 2, 1, 2, 2, 4, 1, 2, 1, 2, 0, 2, 0, 2, 1, 2, 0, 3, 0, 2, 1, 2, 2, 2, 0, 2, 1, 4, 0, 8, 0, 3, 0, 3, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 3, 0, 3, 0, 2, 2, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 8, 1, 2, 1, 3, 0, 2, 3, 2, 0, 2, 0, 2, 0, 2, 0, 290, 0, 2, 0, 4, 0, 9, 0, 7, 0, 2, 2, 2, 0, 2, 1, 2, 16, 2, 8, 36, 0, 3, 2, 4, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 6, 0, 3, 0, 2, 1, 3, 50, 49, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 10, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 3, 37, 2890, 37, 2, 0, 6, 0, 723, 85, 2196, 0, 7, 42, 3, 2, 321, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 10, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 10, 7, 9, 5, 11, 7, 9, 7, 9, 5, 12, 0, 2, 0, 2, 0, 2, 0, 9, 7, 73, 3, 13, 3, 13, 3, 13, 4, 12, 3, 263, 0, 5, 0, 4, 2, 3, 2, 3, 0, 4, 4, 7, 0, 2, 0, 2, 0, 2, 3, 3, 3, 11, 1, 6, 0, 62, 0, 2685, 47, 49, 0, 2, 2, 3, 0, 2, 0, 2, 0, 2, 3, 2, 0, 3, 0, 9, 2, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 9, 0, 2, 0, 5, 0, 31054, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 20, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 136, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 4, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 11, 0, 2, 0, 2, 1, 2, 0, 2, 0, 2, 0, 2, 0, 5, 0, 2, 0, 3, 0, 2, 0, 4, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 4, 2, 4, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 3, 2, 0, 2, 1, 4, 0, 6, 0, 2, 0, 2, 0, 2, 0, 25, 0, 22316, 25, 1222, 39, 137, 35, 157, 10, 2, 14, 2, 6, 2, 1, 1771, 50, 158, 21, 2875, 31, 21889, 31, 26017, 25, 27, 25, 27, 25, 27, 0, 2, 1, 3, 0, 3, 1, 3, 3, 2, 7, 27, 25, 27, 1, 2, 3, 3, 7, 2, 6, 28, 1, 2, 3, 2, 4, 2, 0, 4, 6, 28, 25, 27, 25, 27, 25, 27, 25, 27, 25, 27, 25, 31, 24, 34, 24, 34, 24, 34, 24, 34, 24, 34, 0, 4406, 33], [2307, 0, 56, 0, 3, 2, 9, 3, 2, 1, 51, 1, 59, 2, 7, 1, 3, 1, 11, 0, 44, 0, 59, 2, 67, 0, 59, 2, 9, 0, 2, 1, 54, 1, 59, 0, 2, 0, 7, 1, 3, 1, 11, 0, 103, 1, 2, 1, 4, 2, 2, 2, 11, 0, 42, 2, 62, 3, 62, 1, 59, 0, 2, 4, 3, 1, 2, 1, 10, 1, 29, 0, 15, 1, 59, 2, 6, 2, 2, 2, 11, 0, 43, 1, 76, 2, 7, 7, 19, 1, 331, 1, 64, 0, 172, 1, 5, 0, 7, 0, 3, 1, 26, 1, 11, 2, 3, 6, 22, 1, 3, 5, 3, 0, 11, 2, 1657, 0, 31, 0, 130, 0, 8, 7, 2, 1, 347, 3, 3, 2, 5, 1, 2, 5, 225, 1, 59, 0, 2, 0, 10, 0, 2, 1, 9, 5, 146, 0, 49, 0, 6, 0, 2, 4, 2, 1, 62, 0, 31, 0, 5, 1, 3, 0, 61, 0, 3, 2, 2, 0, 4, 1, 49, 7, 9, 1, 172, 0, 22, 0, 4919, 1, 30708, 1, 3, 0, 89, 1, 51, 15, 143, 1, 48, 0, 49, 1, 5, 1, 3, 2, 111, 1, 3, 1, 25, 0, 46, 0, 2, 0, 110, 0, 3, 1, 6, 0, 238, 1, 2, 1, 2, 1, 2, 0, 25620, 0, 2, 0, 128, 0, 46, 2, 5, 1, 116, 0, 25, 1, 60, 0, 49, 2, 10, 1, 14, 0, 94, 2, 4, 1, 2, 0, 171, 2, 32, 1, 59, 1, 2, 3, 3, 1, 3, 2, 10, 0, 11, 1, 85, 2, 8, 0, 3, 0, 2, 3, 2, 1, 2, 0, 102, 2, 9, 1, 4, 0, 107, 2, 7, 0, 2, 3, 3, 0, 238, 2, 7, 3, 3, 0, 114, 2, 9, 1, 2, 0, 110, 0, 2, 1, 7, 0, 104, 0, 2, 1, 5, 0, 262, 2, 10, 0, 248, 5, 2, 1, 5, 0, 3, 0, 2, 0, 143, 2, 9, 3, 5, 0, 85, 0, 30, 1, 63, 0, 408, 0, 15, 0, 107, 0, 8, 0, 3, 0, 214, 4, 5, 1, 2, 0, 351, 1, 13, 0, 49, 1, 9, 1, 2, 0, 16873, 2, 3621, 54, 105, 1, 24948, 1, 7, 5], [1160, 1, 5685, 0, 1567, 3, 2, 2, 34188, 2], [768, 111, 276, 4, 266, 44, 2, 0, 2, 1, 2, 1, 2, 0, 73, 10, 49, 20, 17, 0, 102, 6, 3, 5, 3, 1, 2, 3, 36, 0, 31, 26, 92, 10, 59, 8, 10, 0, 25, 3, 2, 8, 2, 2, 2, 4, 44, 2, 60, 8, 43, 23, 2, 31, 56, 0, 2, 0, 5, 7, 5, 0, 4, 6, 11, 1, 30, 0, 59, 0, 5, 3, 9, 0, 21, 1, 27, 0, 3, 1, 58, 0, 5, 1, 5, 1, 3, 2, 4, 0, 31, 1, 4, 0, 12, 1, 58, 0, 5, 4, 2, 1, 5, 0, 21, 1, 23, 5, 2, 0, 59, 0, 3, 0, 2, 3, 9, 0, 8, 1, 12, 1, 31, 0, 62, 0, 13, 0, 51, 0, 4, 0, 56, 0, 2, 2, 6, 2, 2, 3, 8, 1, 12, 1, 30, 0, 59, 0, 3, 0, 7, 0, 6, 1, 21, 1, 29, 1, 58, 1, 5, 3, 9, 0, 21, 1, 30, 0, 73, 0, 8, 2, 2, 0, 91, 0, 3, 6, 13, 7, 99, 0, 3, 8, 12, 6, 74, 1, 28, 0, 2, 0, 2, 0, 56, 13, 2, 4, 2, 1, 6, 10, 2, 35, 10, 0, 103, 3, 2, 5, 2, 1, 3, 1, 26, 1, 5, 2, 17, 3, 14, 0, 3, 1, 7, 0, 16, 0, 704, 2, 947, 2, 30, 1, 31, 1, 31, 1, 65, 1, 2, 6, 9, 0, 3, 10, 10, 0, 46, 2, 2, 0, 118, 1, 35, 0, 119, 2, 5, 1, 10, 0, 7, 2, 220, 1, 3, 0, 59, 0, 2, 6, 2, 0, 2, 0, 3, 7, 7, 9, 3, 0, 49, 13, 2, 15, 50, 3, 49, 0, 2, 4, 2, 0, 6, 0, 41, 8, 13, 1, 33, 3, 3, 1, 2, 2, 57, 0, 2, 1, 4, 0, 2, 2, 59, 7, 3, 1, 153, 2, 2, 12, 2, 6, 5, 0, 7, 0, 4, 1, 199, 63, 721, 12, 5, 0, 4, 11, 3071, 2, 142, 0, 97, 31, 555, 3, 108, 1, 30165, 0, 5, 9, 33, 1, 81, 1, 273, 0, 4, 0, 5, 0, 26, 1, 6, 0, 152, 1, 27, 17, 14, 0, 39, 7, 26, 10, 47, 2, 49, 0, 3, 3, 3, 1, 40, 0, 68, 5, 3, 1, 3, 1, 13, 0, 9, 0, 48, 0, 52, 0, 2, 2, 3, 1, 6, 1, 2, 0, 43, 1, 9, 0, 239, 0, 3, 0, 5, 0, 20273, 0, 738, 15, 17, 15, 974, 0, 227, 0, 150, 4, 1671, 2, 2, 1, 6, 3, 41, 2, 5, 0, 166, 1, 574, 3, 66, 4, 318, 1, 80, 3, 71, 10, 50, 3, 124, 0, 55, 14, 42, 0, 3, 1, 11, 2, 50, 3, 3, 1, 8, 0, 62, 2, 37, 4, 2, 7, 63, 0, 13, 1, 53, 8, 11, 3, 3, 0, 96, 2, 3, 0, 2, 1, 7, 0, 3, 0, 158, 0, 4, 7, 22, 1, 58, 1, 4, 0, 38, 6, 4, 4, 71, 5, 14, 0, 2, 0, 2, 0, 15, 1, 86, 7, 3, 2, 2, 0, 24, 0, 85, 5, 2, 0, 5, 1, 2, 1, 239, 3, 7, 1, 2, 1, 28, 1, 86, 7, 3, 0, 2, 1, 107, 0, 2, 0, 3, 5, 2, 0, 102, 0, 2, 0, 3, 3, 2, 4, 260, 8, 2, 1, 257, 1, 2, 0, 5, 0, 145, 3, 3, 1, 5, 0, 33, 9, 41, 5, 3, 3, 9, 0, 10, 5, 3, 2, 47, 12, 2, 1, 407, 6, 2, 5, 2, 0, 83, 21, 3, 6, 2, 1, 2, 1, 123, 5, 4, 0, 2, 1, 2, 6, 2, 0, 73, 1, 4, 0, 2, 0, 348, 1, 12, 1, 53, 4, 6, 0, 2, 0, 24, 0, 5350, 0, 7, 14, 11465, 11, 4, 2, 2497, 4, 60, 6, 1049, 0, 64, 3, 82, 0, 19641, 1, 4706, 45, 3, 22, 545, 2, 18, 7, 3, 6, 31, 3, 149, 2, 1980, 54, 5, 49, 9, 0, 15, 0, 23, 4, 2, 14, 1361, 6, 2, 16, 3, 6, 2, 1, 2, 4, 101, 0, 161, 6, 376, 0, 62, 3, 509, 3, 255, 1, 737, 6, 110, 6, 792502, 239], [48, 9, 1575, 9, 135, 9, 199, 9, 413, 9, 119, 9, 119, 9, 119, 9, 119, 9, 119, 9, 119, 9, 119, 9, 119, 9, 119, 9, 97, 9, 119, 9, 71, 9, 279, 9, 71, 9, 1863, 9, 39, 9, 301, 9, 129, 9, 167, 9, 7, 9, 183, 9, 87, 9, 135, 9, 7, 9, 35271, 9, 679, 9, 39, 9, 199, 9, 23, 9, 87, 9, 407, 9, 21271, 9, 1415, 9, 2183, 9, 7, 9, 797, 9, 129, 9, 61, 9, 145, 9, 279, 9, 343, 9, 119, 9, 375, 9, 103, 9, 7, 19, 77, 9, 423, 9, 103, 9, 663, 9, 87, 9, 247, 9, 71, 9, 423, 9, 16855, 9, 2343, 9, 87, 9, 135, 9, 535, 9, 24439, 9, 2773, 49, 2369, 9, 423, 9, 503, 9, 248, 9, 854, 9, 4759, 9], [5870, 2, 2672, 34, 3, 3, 3711, 0, 26, 8, 15, 2, 30380, 9, 23121, 52, 461, 0, 9, 0, 135, 4, 8235, 110], [178, 1, 6, 0, 3, 2, 2358, 5, 377, 5, 121, 2, 134, 6, 218, 6, 18, 8, 434, 9, 1078, 19, 1140, 9, 481, 0, 1686, 0, 4, 5, 7, 9, 199, 15, 42, 0, 727, 59, 79, 21, 631, 29, 1386, 0, 1173, 3, 139, 9, 31, 7, 2, 14, 33, 9, 40, 14, 30065, 5, 22738, 44, 66, 3, 18, 1, 342, 26, 37, 3, 1333, 7, 26, 6, 40, 8, 76, 4, 23, 5, 161, 1, 3, 15, 3, 45, 65, 8, 53, 1, 31, 2, 76, 4, 105, 7, 25, 7, 42, 6, 331, 5, 353, 30, 159, 9, 43, 3, 113, 6, 135, 19, 380, 19, 1350, 1, 431, 8, 872, 18, 852, 20, 19335, 6, 799, 22, 25642, 19, 13, 19, 109, 24, 5455, 8, 930, 58, 2, 2, 2, 3, 77, 44, 2, 14, 963, 12], [95, 0, 8160, 1, 20, 0, 56799, 1, 25, 2, 240, 0], [45, 0, 1373, 0, 52, 0, 3650, 0, 1030, 0, 2058, 5, 3586, 0, 3, 0, 32, 1, 5, 0, 29, 0, 447, 0, 20, 0, 112, 0, 52625, 1, 38, 0, 11, 0, 170, 0, 3681, 0, 319, 0], [41, 0, 52, 0, 32, 0, 3774, 0, 2, 0, 1887, 0, 2474, 0, 56, 0, 16, 0, 635, 0, 2, 0, 31, 0, 1087, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 81, 0, 33, 0, 2, 0, 2, 0, 2, 0, 2, 0, 405, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 65, 0, 2, 0, 34, 0, 1062, 0, 2, 0, 2, 0, 2, 0, 45, 0, 2, 0, 2, 0, 2, 0, 429, 0, 2, 0, 2, 0, 2, 0, 2, 0, 4, 0, 2, 0, 2, 0, 2, 0, 3, 1, 52511, 0, 218, 0, 30, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 4, 0, 18, 0, 2, 0, 2, 0, 171, 0, 52, 0, 32, 0, 3, 0, 3, 0], [187, 0, 8030, 0, 4, 0, 29, 0, 3529, 0, 2, 0, 5, 0, 3, 0, 16, 0, 4, 0], [171, 0, 8045, 0, 3, 1, 3, 0, 26, 0, 3529, 0, 2, 0, 5, 0, 3, 0, 16, 0, 4, 0], [33, 2, 2, 2, 3, 0, 2, 0, 2, 1, 11, 1, 4, 1, 28, 0, 69, 0, 6, 0, 15, 1, 8, 0, 703, 0, 9, 0, 467, 5, 42, 0, 55, 0, 3, 0, 3, 0, 45, 1, 21, 1, 2, 1, 14, 0, 2, 2, 75, 3, 103, 0, 44, 13, 234, 2, 55, 14, 32, 0, 262, 1, 11, 0, 141, 0, 121, 0, 122, 0, 391, 0, 13, 0, 368, 0, 91, 0, 11, 1, 169, 14, 2, 0, 113, 0, 75, 4, 5, 1, 112, 5, 172, 0, 613, 8, 774, 0, 125, 2, 72, 1, 158, 2, 2, 2, 38, 5, 2, 3, 314, 1, 217, 1, 129, 6, 2, 5, 161, 1, 11, 6, 29, 2, 125, 3, 60, 4, 63, 1, 65, 7, 12, 0, 835, 1, 9, 7, 9, 8, 3, 3, 3, 2, 4, 10, 2, 0, 2, 9, 3227, 3, 2, 1, 113, 0, 144, 1, 5, 2, 3, 0, 3, 8, 2, 1, 2, 0, 3, 1, 11, 4, 2, 9, 3, 3, 2, 0, 2, 12, 3, 2, 429, 2, 58, 0, 190, 0, 29699, 1, 270, 2, 100, 0, 11, 0, 116, 5, 381, 3, 87, 1, 41, 2, 2, 0, 50, 1, 48, 0, 98, 12, 17, 1, 125, 3, 127, 1, 17, 1, 250, 0, 21029, 6, 3, 0, 23, 0, 21, 1, 3, 3, 4, 2, 2, 3, 8, 2, 7, 0, 2, 1, 150, 2, 2, 2, 3, 0, 2, 0, 2, 1, 11, 1, 4, 1, 28, 0, 37, 0, 3, 1, 411, 2, 669, 0, 49, 0, 415, 0, 744, 0, 200, 0, 32, 0, 273, 8, 39, 0, 113, 6, 67, 6, 90, 3, 953, 4, 45, 3, 190, 6, 110, 1, 2, 3, 127, 3, 49, 1, 80, 3, 5, 0, 14, 0, 2, 2, 89, 5, 108, 0, 299, 1, 2, 1, 115, 4, 11, 1, 2, 0, 105, 0, 251, 22, 106, 2, 29, 12, 77, 0, 131, 2, 253, 0, 265, 2, 156, 0, 93, 7, 84, 2, 2, 4, 94, 9, 216, 0, 96, 4, 43, 1, 646, 1, 75, 12, 176, 0, 1137, 4, 2941, 1, 14972, 1, 134, 0, 66, 4, 9, 0, 553, 2, 296, 3, 328, 0, 19645, 0, 7656, 4, 2932, 0, 863, 1], [40, 0, 51, 0, 32, 0, 3775, 0, 2, 0, 1887, 0, 2431, 0, 4, 0, 39, 0, 56, 0, 16, 0, 635, 0, 2, 0, 31, 0, 1087, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 81, 0, 33, 0, 2, 0, 2, 0, 2, 0, 2, 0, 405, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 65, 0, 2, 0, 34, 0, 1062, 0, 2, 0, 2, 0, 2, 0, 26, 0, 19, 0, 2, 0, 2, 0, 2, 0, 429, 0, 2, 0, 2, 0, 2, 0, 2, 0, 4, 0, 2, 0, 2, 0, 2, 0, 3, 0, 52514, 0, 216, 0, 30, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 4, 0, 18, 0, 2, 0, 2, 0, 171, 0, 51, 0, 32, 0, 4, 0, 3, 0], [36, 0, 126, 3, 1258, 0, 124, 0, 499, 1, 499, 1, 8, 0, 246, 0, 264, 0, 582, 0, 2460, 0, 2245, 32, 34680, 0, 21956, 0, 109, 0, 155, 0, 220, 1, 4, 1, 8183, 3, 49951, 0, 2481, 0], [94, 0, 2, 0, 72, 0, 7, 0, 5, 0, 4, 0, 522, 3, 13, 13, 6, 6, 2, 0, 2, 16, 118, 0, 15, 1, 1283, 0, 5941, 0, 2, 2, 12, 2, 14, 2, 14, 2, 14, 1, 4253, 1, 30308, 22, 10, 1, 104, 1, 977, 0, 15, 1, 20551, 16, 892, 0, 2, 0, 163, 0, 62488, 4], [43, 0, 17, 2, 62, 0, 2, 0, 46, 0, 5, 0, 38, 0, 32, 0, 767, 0, 528, 2, 6716, 0, 14, 0, 40, 2, 14, 2, 140, 0, 40, 4, 7, 0, 69, 4, 6, 1, 5, 0, 3, 0, 3, 0, 8, 0, 32, 1, 3, 0, 2, 0, 32, 267, 33, 1, 91, 0, 31, 24, 41, 5, 470, 0, 10, 0, 55, 7, 112, 0, 337, 4, 3, 30, 11, 15, 257, 130, 23, 62, 5, 31, 3, 257, 49, 20, 3, 5, 53213, 0, 825, 0, 2, 2, 165, 0, 17, 2, 62, 0, 2, 0, 132, 0, 7, 3, 3490, 1, 51506, 0, 26, 0, 32, 0, 26, 0, 32, 0, 26, 0, 32, 0, 26, 0, 32, 0, 26, 0, 5933, 1], [166, 0, 3, 0, 5, 0, 2, 0, 978, 0, 267, 1, 128, 1, 207, 0, 11, 0, 20, 1, 248, 0, 516, 0, 374, 0, 131, 5, 2, 0, 133, 0, 208, 0, 42, 0, 392, 2, 16, 0, 2, 2, 3, 5, 21, 0, 2, 0, 2, 0, 134, 7, 2, 5, 2, 1, 6, 3, 198, 1, 753, 9, 724, 0, 723, 0, 158, 33, 354, 9, 10, 8, 1412, 1, 2, 3, 2, 1, 11, 0, 2, 1, 7, 5, 2, 0, 2, 0, 2, 0, 5, 0, 12, 1, 15, 0, 2, 1, 2, 0, 59, 1, 10, 4, 3, 3, 2, 1, 2, 1, 2, 6, 2, 30, 3, 1, 2, 0, 2, 30, 269, 7, 5, 19, 3, 6, 3, 80, 2, 29, 26, 39, 7, 71, 23, 10, 82, 77, 23, 182, 2, 8, 2, 53, 9, 110, 2, 247, 45, 43, 65, 255, 513, 47, 22, 1, 7, 38, 3, 31, 2, 104, 230, 5, 358, 1, 47, 25, 2, 88, 13, 213, 27, 15, 5, 0, 14, 1, 13, 0, 22, 1, 7, 1, 337, 1, 5, 9, 33, 37, 10, 0, 17, 30, 12, 29, 9, 0, 16, 31, 11, 38, 16, 319, 6593, 63, 22161, 54, 866, 3, 11, 1, 2, 0, 574, 2, 21191, 15, 128, 0, 46, 2, 485, 0, 4, 0, 5, 1, 14, 1, 314, 8, 58, 16, 3, 2, 2, 12, 4, 0, 48, 44, 1659, 1, 592, 0, 3191, 0, 2198, 7, 5, 16, 19275, 3, 6, 0, 20823, 0, 3940, 239, 17, 435, 157, 115, 61, 245, 11, 38, 3, 59, 6, 2, 23, 1, 8, 29, 5, 60, 22, 65, 4, 0, 187, 86, 1194, 511, 56, 3, 51, 7, 2, 13, 2, 1, 1737, 0, 2909, 0, 130, 0, 722, 43, 5, 99, 13, 14, 3, 14, 2, 14, 2, 36, 24, 160, 57, 28, 14, 43, 5, 8, 8, 1, 15, 5, 155, 250, 6, 727, 5, 16, 4, 12, 4, 118, 5, 94, 7, 11, 5, 0, 16, 11, 5, 55, 9, 9, 7, 39, 9, 29, 3, 11, 5, 1, 63, 339, 13, 13, 3, 12, 4, 9, 6, 55, 8, 14, 3, 10, 7, 8, 8, 146, 2, 91], [8232, 0], [8233, 0], [32, 0, 128, 0, 5600, 0, 2432, 10, 37, 0, 48, 0, 4001, 0]]);
