package de.hbch.traewelling.util

import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyListScope
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.rememberUpdatedState
import androidx.compose.runtime.snapshotFlow
import androidx.compose.runtime.snapshots.SnapshotStateList
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.asAndroidBitmap
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.core.content.FileProvider.getUriForFile
import androidx.navigation.NavHostController
import com.jcloquell.androidsecurestorage.SecureStorage
import de.hbch.traewelling.BuildConfig
import de.hbch.traewelling.R
import de.hbch.traewelling.api.TraewellingApi
import de.hbch.traewelling.api.models.lineIcons.LineIcon
import de.hbch.traewelling.api.models.status.Status
import de.hbch.traewelling.logging.Logger
import de.hbch.traewelling.navigation.Destination
import de.hbch.traewelling.shared.LoggedInUserViewModel
import de.hbch.traewelling.shared.SharedValues
import de.hbch.traewelling.theme.LocalFont
import de.hbch.traewelling.ui.include.status.CheckInCard
import de.hbch.traewelling.ui.include.status.CheckInCardViewModel
import kotlinx.coroutines.CoroutineScope
import net.openid.appauth.AppAuthConfiguration
import net.openid.appauth.AuthorizationService
import net.openid.appauth.GrantTypeValues
import net.openid.appauth.TokenRequest
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.io.File
import java.lang.Exception
import java.time.LocalDate
import java.time.ZonedDateTime
import java.util.UUID
import androidx.core.graphics.toColorInt
import androidx.core.net.toUri

fun NavHostController.popBackStackAndNavigate(
    destination: Destination,
    launchSingleTop: Boolean = true,
    popUpToInclusive: Boolean = true
) {
    navigate(destination) {
        popUpTo(graph.id) {
            inclusive = popUpToInclusive
        }
        this.launchSingleTop = launchSingleTop
    }
}

fun LazyListScope.checkInList(
    checkIns: SnapshotStateList<Status>,
    checkInCardViewModel: CheckInCardViewModel,
    loggedInUserViewModel: LoggedInUserViewModel,
    joinConnection: (Status) -> Unit,
    stationSelectedAction: (Int, ZonedDateTime?) -> Unit = { _, _ -> },
    statusSelectedAction: (Int) -> Unit = { },
    statusEditAction: (Status) -> Unit = { },
    statusDeletedAction: () -> Unit = { },
    userSelectedAction: (String, Boolean, Boolean) -> Unit = { _, _, _ -> },
    showDailyStatisticsLink: Boolean = false,
    dailyStatisticsSelectedAction: (LocalDate) -> Unit = { },
    showDate: Boolean = true
) {
    itemsIndexed(
        items = checkIns,
        key = { _, status -> status.id }
    ) { index, status ->
        val previousStatus = checkIns.getOrNull(index - 1)
        if (
            showDate &&
            (
                previousStatus == null ||
                !isSameDay(
                    previousStatus.journey.origin.departurePlanned.toLocalDate(),
                    status.journey.origin.departurePlanned.toLocalDate()
                )
            )
        ) {
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(bottom = 16.dp),
                horizontalArrangement = Arrangement.spacedBy(8.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = getLongLocalDateString(status.journey.origin.departurePlanned),
                    modifier = Modifier
                        .weight(1f),
                    maxLines = 2,
                    overflow = TextOverflow.Ellipsis,
                    style = LocalFont.current.titleLarge
                )
                if (showDailyStatisticsLink) {
                    Icon(
                        painter = painterResource(id = R.drawable.ic_score),
                        contentDescription = null,
                        modifier = Modifier.clickable(onClick = {
                            dailyStatisticsSelectedAction(status.journey.origin.departurePlanned.toLocalDate())
                        })
                    )
                }
            }
        }
        CheckInCard(
            checkInCardViewModel = checkInCardViewModel,
            status = status,
            joinConnection = joinConnection,
            loggedInUserViewModel = loggedInUserViewModel,
            stationSelected = stationSelectedAction,
            statusSelected = statusSelectedAction,
            handleEditClicked = statusEditAction,
            onDeleted = { statusValue ->
                checkIns.removeIf { it.id == statusValue.id }
                statusDeletedAction()
            },
            userSelected = userSelectedAction
        )
        if (checkIns.size == (index + 1)) {
            Box(Modifier.height(16.dp))
        }
    }
}


@Composable
fun LazyListState.OnBottomReached(
    loadMore : () -> Unit
){
    val shouldLoadMore = remember {
        derivedStateOf {
            val lastVisibleItem = layoutInfo.visibleItemsInfo.lastOrNull()
                ?: return@derivedStateOf true

            lastVisibleItem.index == layoutInfo.totalItemsCount - 1
        }
    }

    LaunchedEffect(shouldLoadMore){
        snapshotFlow { shouldLoadMore.value }
            .collect {
                if (it) loadMore()
            }
    }
}

fun Context.shareStatus(
    status: Status,
    imageBitmap: ImageBitmap? = null
) {
    var shareText =
        if (status.getStatusText().isBlank())
            getString(R.string.share_text, status.journey.line, status.journey.destination.name)
        else
            getString(R.string.share_text_with_body, status.getStatusText(), status.journey.line, status.journey.destination.name)

    val shareUri = Uri.Builder()
        .scheme("https")
        .authority("traewelling.de")
        .appendPath("status")
        .appendPath(status.id.toString())
        .build()

    shareText = shareText.plus(" #NowTräwelling\n\n$shareUri")

    val sendIntent: Intent = Intent().apply {
        action = Intent.ACTION_SEND
        type = "text/plain"
    }

    if (imageBitmap != null) {
        try {
            val imagePath = File(cacheDir, "sharePics")
            if (!imagePath.exists())
                imagePath.mkdirs()

            val imageFile = File.createTempFile(status.id.toString(), ".png", imagePath)
            val outputStream = imageFile.outputStream()
            imageBitmap.asAndroidBitmap().compress(Bitmap.CompressFormat.PNG, 95, outputStream)
            outputStream.close()
            val contentUri = getUriForFile(this, "de.traewelldroid.fileprovider", imageFile)
            sendIntent.putExtra(Intent.EXTRA_STREAM, contentUri)
            sendIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
            sendIntent.type = "image/png"
            shareText = shareUri.toString()
        } catch (e: Exception) {
            Logger.captureException(e)
        }
    }

    sendIntent.putExtra(Intent.EXTRA_TEXT, shareText)

    val shareIntent = Intent.createChooser(
        sendIntent,
        getString(R.string.title_share)
    )
    startActivity(shareIntent)
}

fun Context.shareImage(
    shareText: String,
    imageBitmap: ImageBitmap
) {
    try {
        val imagePath = File(cacheDir, "sharePics")
        if (!imagePath.exists())
            imagePath.mkdirs()

        val sendIntent: Intent = Intent().apply {
            action = Intent.ACTION_SEND
            type = "text/plain"
        }

        val imageFile = File.createTempFile(UUID.randomUUID().toString(), ".png", imagePath)
        val outputStream = imageFile.outputStream()
        imageBitmap.asAndroidBitmap().compress(Bitmap.CompressFormat.PNG, 95, outputStream)
        outputStream.close()
        val contentUri = getUriForFile(this, "de.traewelldroid.fileprovider", imageFile)
        sendIntent.putExtra(Intent.EXTRA_STREAM, contentUri)
        sendIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
        sendIntent.type = "image/png"
        if (shareText.isNotEmpty()) {
            sendIntent.putExtra(Intent.EXTRA_TEXT, shareText)
        }

        val shareIntent = Intent.createChooser(
            sendIntent,
            getString(R.string.title_share)
        )
        startActivity(shareIntent)
    } catch (e: Exception) {
        Logger.captureException(e)
    }
}

fun TraewelldroidUriBuilder(): Uri.Builder {
    return Uri.Builder()
        .scheme("traewelldroid")
        .authority("app.traewelldroid.de")
}

fun colorFromHex(color: String)
    = try {
        Color(color.toColorInt())
    } catch (_: Exception) {
        null
    }



fun Context.refreshJwt(onTokenReceived: (String) -> Unit = { }, onError: () -> Unit = { }) {
    val authorizationService = AuthorizationService(
        this,
        AppAuthConfiguration.Builder().build()
    )
    val secureStorage = SecureStorage(this)
    val refreshToken = secureStorage.getObject(SharedValues.SS_REFRESH_TOKEN, String::class.java)
    val tokenRequest = TokenRequest.Builder(SharedValues.AUTH_SERVICE_CONFIG, BuildConfig.OAUTH_CLIENT_ID)
        .setGrantType(GrantTypeValues.REFRESH_TOKEN)
        .setRefreshToken(refreshToken)
        .build()

    authorizationService.performTokenRequest(tokenRequest) { response, error ->
        if (error != null) {
            onError()
            return@performTokenRequest
        }
        if (response?.accessToken != null && response.refreshToken != null) {
            secureStorage.storeObject(SharedValues.SS_JWT, response.accessToken!!)
            secureStorage.storeObject(SharedValues.SS_REFRESH_TOKEN, response.refreshToken!!)
            TraewellingApi.jwt = response.accessToken!!
            onTokenReceived(response.accessToken!!)
        } else {
            onError()
        }
    }
}

fun Context.openLink(url: String) {
    val intent = Intent(Intent.ACTION_VIEW, url.toUri())
    try {
        startActivity(intent)
    } catch (_: Exception) { }
}

@Composable
fun <T> T.useDebounce(
        delayMillis: Long = 300L,
        coroutineScope: CoroutineScope = rememberCoroutineScope(),
        onChange: suspend (T) -> Unit
    ): T{
        val state by rememberUpdatedState(this)

    DisposableEffect(state){
        val job = coroutineScope.launch {
            delay(delayMillis)
            onChange(state)
        }
        onDispose {
            job.cancel()
        }
    }

    return state
}

fun String.extractUsernames() = "@(\\S*\\w)".toRegex().findAll(this).toList()
fun String.checkAnyUsernames() = "@(\\S*|$)".toRegex().findAll(this).toList()
