package de.georgelsas.tuttocounter;

import static android.view.View.VISIBLE;

import android.app.Activity;
import android.graphics.Color;
import android.view.View;

import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.widget.ActionMenuView;
import androidx.appcompat.widget.Toolbar;
import androidx.recyclerview.widget.RecyclerView;

import uk.co.deanwild.materialshowcaseview.MaterialShowcaseSequence;
import uk.co.deanwild.materialshowcaseview.MaterialShowcaseView;

/**
 * A helper class to display a tutorial for the main features of the app using MaterialShowcaseView.
 */
public class ShowcaseHelper {

    /**
     * Displays a full tutorial showcasing the main UI elements.
     *
     * @param activity The hosting activity.
     * @param recyclerView The RecyclerView containing the player cards.
     * @param toolbar The toolbar of the activity.
     * @param resetRecycler A flag indicating whether to reset the RecyclerView after the tutorial.
     * @param deleteAllCards A Runnable to be executed to delete all cards.
     * @param hideWinningButton A Runnable to be executed to hide the winning button.
     */
    public static void showFullTutorial(Activity activity,
                                        RecyclerView recyclerView,
                                        Toolbar toolbar,
                                        Boolean resetRecycler,
                                        Runnable deleteAllCards,
                                        Runnable hideWinningButton)
    {
        // Create a new showcase sequence.
        MaterialShowcaseSequence sequence = new MaterialShowcaseSequence(activity);

        // Showcase for adding a new player.
        View add_player = activity.findViewById(R.id.action_add_player);
        sequence.addSequenceItem(
                new MaterialShowcaseView.Builder(activity)
                        .setTarget(add_player)
                        .setTitleText(activity.getString(R.string.add_player))
                        .setContentText(activity.getString(R.string.here_you_can_add_new_players))
                        .setDismissText(activity.getString(R.string.ok))
                        .setShapePadding(30)
                        .build()
        );

        // Showcase for the action history.
        View action_history = activity.findViewById(R.id.action_history);
        sequence.addSequenceItem(
                new MaterialShowcaseView.Builder(activity)
                        .setTarget(action_history)
                        .setTitleText(activity.getString(R.string.action_history))
                        .setContentText(activity.getString(R.string.tap_this_to_inspect_recent_changes))
                        .setDismissText(activity.getString(R.string.ok))
                        .setShapePadding(30)
                        .build()
        );

        // Showcase for the overflow menu in the toolbar.
        toolbar.post(() -> {
            View overflowButton = findOverflowMenuButton(activity, toolbar);
            if (overflowButton != null) {
                sequence.addSequenceItem(
                        new MaterialShowcaseView.Builder(activity)
                                .setTarget(overflowButton)
                                .setTitleText(activity.getString(R.string.options))
                                .setContentText(activity.getString(R.string.here_you_can_start_a_new_game))
                                .setDismissText(activity.getString(R.string.ok))
                                .build()
                );
            }
        });

        // Showcase for the player cards in the RecyclerView.
        View firstCardOrEmpty = recyclerView.getLayoutManager().getChildAt(0);
        if (firstCardOrEmpty == null) {
            recyclerView.post(() -> {
                View firstCard = recyclerView.getLayoutManager().getChildAt(0);
                if (firstCard != null) {
                    sequence.addSequenceItem(
                            new MaterialShowcaseView.Builder(activity)
                                    .setTarget(firstCard)
                                    .setTitleText(activity.getString(R.string.players))
                                    .setContentText(activity.getString(R.string.these_are_the_players))
                                    .setDismissText(activity.getString(R.string.ok))
                                    .setShapePadding(30)
                                    .build()
                    );
                }
                View editButton = firstCard.findViewById(R.id.card_edit_icon);
                sequence.addSequenceItem(
                        new MaterialShowcaseView.Builder(activity)
                                .setTarget(editButton)
                                .setTitleText(activity.getString(R.string.edit_player))
                                .setContentText(activity.getString(R.string.here_you_can_edit))
                                .setDismissText(activity.getString(R.string.ok))
                                .setShapePadding(50)
                                .build()
                );
            });
        } else {
            sequence.addSequenceItem(
                    new MaterialShowcaseView.Builder(activity)
                            .setTarget(firstCardOrEmpty)
                            .setTitleText(activity.getString(R.string.players))
                            .setContentText(activity.getString(R.string.these_are_the_players))
                            .setDismissText(activity.getString(R.string.ok))
                            .setShapePadding(30)
                            .build()
            );
            View editButton = activity.findViewById(R.id.card_edit_icon);
            sequence.addSequenceItem(
                    new MaterialShowcaseView.Builder(activity)
                            .setTarget(editButton)
                            .setTitleText(activity.getString(R.string.edit_player))
                            .setContentText(activity.getString(R.string.here_you_can_edit))
                            .setDismissText(activity.getString(R.string.ok))
                            .setShapePadding(50)
                            .build()
            );
        }

        // Showcase for the win button.
        View winnerButton = activity.findViewById(R.id.winningButton);
        winnerButton.setVisibility(VISIBLE);
        winnerButton.setClickable(true);
        winnerButton.setFocusable(true);
        winnerButton.post(() -> {
            MaterialShowcaseView lastStep = new MaterialShowcaseView.Builder(activity)
                    .setTarget(winnerButton)
                    .setTitleText(activity.getString(R.string.win_button))
                    .setContentText(activity.getString(R.string.if_you_have_a_winner))
                    .setDismissText(activity.getString(R.string.ok))
                    .setShapePadding(50)
                    .build();
            sequence.addSequenceItem(lastStep);
            sequence.setOnItemDismissedListener((itemView, position) -> {
                if (itemView == lastStep) {
                    hideWinningButton.run();
                    if (resetRecycler) deleteAllCards.run();
                }
            });
        });
        // Start the tutorial sequence.
        sequence.start();

    }

    /**
     * Finds the overflow menu button within the toolbar.
     *
     * @param activity The hosting activity.
     * @param toolbar The toolbar to search in.
     * @return The overflow menu button view, or null if not found.
     */
    private static View findOverflowMenuButton(Activity activity, Toolbar toolbar) {
        if (toolbar == null) return null;

        String overflowDesc = activity.getString(R.string.overflow_menu);

        // First, search for the ActionMenuView in the toolbar.
        for (int i = 0; i < toolbar.getChildCount(); i++) {
            View child = toolbar.getChildAt(i);
            if (child instanceof ActionMenuView) {
                ActionMenuView menuView =
                        (ActionMenuView) child;

                // Then, search the children of the ActionMenuView for the overflow button.
                for (int j = 0; j < menuView.getChildCount(); j++) {
                    View menuChild = menuView.getChildAt(j);
                    CharSequence cd = menuChild.getContentDescription();
                    if (cd != null && overflowDesc.contentEquals(cd)) {
                        return menuChild; // Match found: this is the overflow button.
                    }
                }
            }
        }
        return null;
    }

    private static void showArcDialogTutorial(Activity activity, AlertDialog dialog) {

        View arcSeekBar = dialog.findViewById(R.id.arcSeekBar);
        View modeSlider = dialog.findViewById(R.id.mode_slider);
        View confirmButton = dialog.findViewById(R.id.confirm_button);

        MaterialShowcaseSequence dialogSequence =
                new MaterialShowcaseSequence(activity);

        dialogSequence.addSequenceItem(
                new MaterialShowcaseView.Builder(activity)
                        .setTarget(arcSeekBar)
                        .setTitleText(activity.getString(R.string.free_input))
                        .setContentText(activity.getString(R.string.tutorial_arc_seekbar_description))
                        .setDismissText(activity.getString(R.string.ok))
                        .setShapePadding(40)
                        .build()
        );

        dialogSequence.addSequenceItem(
                new MaterialShowcaseView.Builder(activity)
                        .setTarget(modeSlider)
                        .setTitleText(activity.getString(R.string.slider))
                        .setContentText(activity.getString(R.string.tutorial_mode_slider_description))
                        .setDismissText(activity.getString(R.string.ok))
                        .setShapePadding(40)
                        .build()
        );

        dialogSequence.addSequenceItem(
                new MaterialShowcaseView.Builder(activity)
                        .setTarget(confirmButton)
                        .setTitleText(activity.getString(R.string.confirm))
                        .setContentText(activity.getString(R.string.save))
                        .setDismissText(activity.getString(R.string.ok))
                        .setShapePadding(40)
                        .build()
        );

        dialogSequence.start();
    }



}
