package de.georgelsas.tuttocounter;

import static android.view.View.VISIBLE;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.ValueAnimator;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.TextView;

import androidx.annotation.NonNull;
import
        androidx.appcompat.widget.Toolbar;

import androidx.activity.EdgeToEdge;
import androidx.appcompat.app.AppCompatActivity;
import androidx.cardview.widget.CardView;
import androidx.core.graphics.Insets;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.recyclerview.widget.ItemTouchHelper;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.google.android.material.floatingactionbutton.FloatingActionButton;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import nl.dionsegijn.konfetti.core.Party;
import nl.dionsegijn.konfetti.core.Position;
import nl.dionsegijn.konfetti.core.Rotation;
import nl.dionsegijn.konfetti.core.emitter.Emitter;
import nl.dionsegijn.konfetti.core.models.Shape;
import nl.dionsegijn.konfetti.core.models.Size;
import nl.dionsegijn.konfetti.xml.KonfettiView;

import java.util.concurrent.TimeUnit;

// TODO: Maybe must be mentioned https://github.com/DanielMartinus/Konfetti?tab=ISC-1-ov-file

/**
 * The main activity of the application, which displays the list of player cards.
 */
public class MainActivity extends AppCompatActivity {
    // Manages the colors for the player cards.
    private ColorManager colorManager;
    // Manages the names for the players.
    private NameManager nameManager;
    // Manages the player cards, including their data and state.
    private CardManager cardManager;

    // The RecyclerView that displays the list of cards.
    private RecyclerView recyclerView;
    // The adapter for the RecyclerView.
    private CardAdapter adapter;
    // The toolbar at the top of the activity.
    private Toolbar toolbar;
    // Timestamp of the last win button click to prevent spamming.
    private long lastWinClickTime = 0;

    // Manages the settings for the app.
    private TuttoSettings tuttoSettings;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setContentView(R.layout.activity_main);

        // Set up the toolbar.
        this.toolbar = findViewById(R.id.main_toolbar);
        setSupportActionBar(toolbar);


        // Initialize managers.
        colorManager = new ColorManager();
        tuttoSettings = new TuttoSettings(this);
        nameManager = new NameManager(this);
        cardManager = new CardManager(this, tuttoSettings);
        // Set up the RecyclerView.
        recyclerView = findViewById(R.id.card_recycler_view);
        recyclerView.setLayoutManager(new LinearLayoutManager(this));
        adapter = new CardAdapter(this, cardManager);
        recyclerView.setAdapter(adapter);

        // Enable drag and drop functionality for the RecyclerView.
        enableDragAndDrop();

        // Enable edge-to-edge display.
        EdgeToEdge.enable(this);

        // Prepare the win button with its confetti animation.
        prepareWinButton();
        // Apply window insets to handle system bars for edge-to-edge display.
        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.main), (v, insets) -> {
            Insets systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars());
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom);
            return insets;
        });
    }

    /**
     * Sets up the winning button, including its click listener and confetti animation.
     */
    private void prepareWinButton() {
        FloatingActionButton winButton = findViewById(R.id.winningButton);
        KonfettiView konfettiView = findViewById(R.id.konfettiView);
        winButton.setOnClickListener(view -> {

            long currentTime = System.currentTimeMillis();

            // Block execution if it has been fired in the last second to prevent spamming.
            if (currentTime - lastWinClickTime < 1000) {
                return; // ignore
            }

            lastWinClickTime = currentTime; // Update the timestamp.
            // Configure the confetti party.
            Party party = new Party(
                    0, // angle
                    360, // spread
                    0f, // speed (minSpeed)
                    30f, // maxSpeed
                    0.9f, // damping
                    Arrays.asList(new Size(12, 5f, 1f)), // sizes
                    Arrays.asList(0xfce18a, 0xff726d, 0xf4306d, 0xb48def), // colors
                    Arrays.asList(Shape.Circle.INSTANCE, Shape.Square.INSTANCE), // shapes
                    1500L, // timeToLive in ms
                    true, // fadeOutEnabled
                    new Position.Relative(0.5, 0.5), // position (center)
                    0, // delay in ms
                    new Rotation(true, 180f, 180f, 1f, 1f),// rotation
                    new Emitter(5, TimeUnit.SECONDS).perSecond(200) // emitter
            );


            konfettiView.start(party);
        });
        // Make the win button visible if there is a winner.
        if (this.cardManager.isWinnerAvailable()) {
            winButton.setFocusable(true);
            winButton.setClickable(true);
            winButton.setVisibility(VISIBLE);
        }

    }

    /**
     * Enables drag-and-drop functionality for reordering items in the RecyclerView.
     */
    private void enableDragAndDrop() {
        ItemTouchHelper helper = new ItemTouchHelper(new ItemTouchHelper.SimpleCallback(
                ItemTouchHelper.UP | ItemTouchHelper.DOWN,
                0) {
            @Override
            public boolean onMove(@NonNull RecyclerView recyclerView,
                                  @NonNull RecyclerView.ViewHolder viewHolder,
                                  @NonNull RecyclerView.ViewHolder target) {
                int from = viewHolder.getAdapterPosition();
                int to = target.getAdapterPosition();
                adapter.moveItem(from, to);
                return true;
            }

            @Override
            public void onSwiped(@NonNull RecyclerView.ViewHolder viewHolder, int direction) {
                // Swipe behavior is not needed.
            }
        });

        helper.attachToRecyclerView(recyclerView);
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.main_menu, menu);

        return true;
    }

    /**
     * Ensures that icons are visible in the overflow menu.
     */
    @Override
    public boolean onMenuOpened(int featureId, Menu menu) {
        if (menu != null) {
            if (menu.getClass().getSimpleName().equals("MenuBuilder")) {
                try {
                    Method m = menu.getClass().getDeclaredMethod("setOptionalIconsVisible", Boolean.TYPE);
                    m.setAccessible(true);
                    m.invoke(menu, true);
                } catch (Exception exc) {
                    Log.e("TuttoError", Objects.requireNonNull(exc.getLocalizedMessage()));
                }
            }
        }
        return super.onMenuOpened(featureId, menu);
    }

    /**
     * Handles selections of items in the options menu.
     * @param item The menu item that was selected.
     * @return True if the event was handled, false otherwise.
     */
    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == R.id.action_add_player) {
            addNewCard();
            return true;
        }
        if (item.getItemId() == R.id.delete_playernames) {
            deleteAllCards();
            return true;
        }
        if (item.getItemId() == R.id.quit_app) {
            finishAndRemoveTask();
            return true;
        }
        if (item.getItemId() == R.id.undo) {
            undo();
            return true;
        }
        if (item.getItemId() == R.id.new_game) {
            new_game();
            return true;
        }
        if (item.getItemId() == R.id.show_game_progress) {
            showProgressChartDialog();
            return true;
        }
        if (item.getItemId() == R.id.action_history) {
            showHistoryChartDialog();
            return true;
        }
        if (item.getItemId() == R.id.settings) {
            showSettingsDialog();
            return true;
        }
        if (item.getItemId() == R.id.help) {
            startTutorial();
            return true;
        }
        if (item.getItemId() == R.id.action_share_scores) {
            new ScreenshotSharer(this).shareScreenshot();
            return true;
        }
        return super.onOptionsItemSelected(item);
    }

    /**
     * Starts the interactive tutorial for the app.
     */
    private void startTutorial() {
        boolean resetRecycler = false;
        // If there are no cards, add a temporary one for the tutorial.
        if (adapter.getItemCount() == 0) {
            addNewCard();
            resetRecycler = true;
        }

        ShowcaseHelper.showFullTutorial(
                this,
                recyclerView,
                toolbar,
                resetRecycler,
                () -> {
                    // Clean up after the tutorial is finished or skipped.
                    cardManager.deleteAllCards();
                    refreshCards();
                },
                () -> {
                    // Adapt the winner button state after the tutorial.
                    cardManager.adaptWinnerButton();
                }
        );
    }

    /**
     * Shows the settings dialog.
     */
    private void showSettingsDialog() {
        SettingsDialog.show(this, R.drawable.dialog_background);
    }

    /**
     * Shows the dialog with the history of the last changes.
     */
    private void showHistoryChartDialog() {
        LastChangesDialog.show(this, R.drawable.dialog_background);
    }

    /**
     * Shows the dialog with the game progress chart.
     */
    private void showProgressChartDialog() {
        ProgressChartDialog.show(this, R.drawable.dialog_background);
    }


    /**
     * Starts a new game by resetting all scores to zero.
     */
    private void new_game() {
        ConfirmationDialogHelper dialogHelper = new ConfirmationDialogHelper(this);

        // Show a confirmation dialog before resetting the game.
        dialogHelper.show(
                getString(R.string.reallyReset),
                () -> {
                    List<CardItem> cardList = cardManager.getCardItems();
                    for (CardItem card : cardList) {
                        card.setNumber(0);
                    }
                    cardManager.deleteHistory();
                    cardManager.saveCards();
                    refreshCards();
                },
                R.string.confirm,
                R.string.abortButton,
                R.color.my_positive_color,
                R.color.my_negative_color,
                R.drawable.dialog_background
        );
    }

    /**
     * Refreshes the adapter and adapts the winner button.
     */
    private void refreshCards() {
        refreshAdapter();
        cardManager.adaptWinnerButton();
    }

    /**
     * Notifies the adapter that the data set has changed.
     */
    private void refreshAdapter() {
        adapter.notifyDataSetChanged();
    }


    /**
     * Undoes the last action performed on the cards.
     */
    private void undo() {
        cardManager.undoLast();
        refreshCards();
    }


    /**
     * Restores the cards from saved state.
     */
    private void restoreCards() {
        refreshCards();
    }

    /**
     * Animates the score change on a TextView.
     *
     * @param textView The TextView to animate.
     * @param endValue The final score to display.
     */
    private void animateScoreChange(TextView textView, int endValue, Runnable onAnimationEnd) {
        int startValue = 0;
        try {
            startValue = Integer.parseInt(textView.getText().toString());
        } catch (NumberFormatException e) {
            // If parsing fails, just snap to the end value without animation.
            textView.setText(String.valueOf(endValue));
            if (onAnimationEnd != null) {
                onAnimationEnd.run();
            }
            return;
        }

        ValueAnimator animator = ValueAnimator.ofInt(startValue, endValue);
        animator.setDuration(500); // Animation duration in milliseconds

        animator.addUpdateListener(animation -> {
            textView.setText(String.valueOf(animation.getAnimatedValue()));
        });

        if (onAnimationEnd != null) {
            animator.addListener(new AnimatorListenerAdapter() {
                @Override
                public void onAnimationEnd(Animator animation) {
                    super.onAnimationEnd(animation);
                    onAnimationEnd.run();
                }
            });
        }

        animator.start();
    }

    private void animateAllVisibleChanges() {
        LinearLayoutManager layoutManager = (LinearLayoutManager) recyclerView.getLayoutManager();
        if (layoutManager == null) {
            refreshCards();
            return;
        }
        int firstVisible = layoutManager.findFirstVisibleItemPosition();
        int lastVisible = layoutManager.findLastVisibleItemPosition();

        if (firstVisible == -1) {
            refreshCards(); // Fallback if no items are visible
            return;
        }

        for (int i = firstVisible; i <= lastVisible; i++) {
            CardAdapter.CardViewHolder holder = (CardAdapter.CardViewHolder) recyclerView.findViewHolderForAdapterPosition(i);
            if (holder != null) {
                CardItem card = cardManager.getCardItems().get(i);
                int newValue = card.getNumber();
                int oldValue;
                try {
                    oldValue = Integer.parseInt(holder.numberView.getText().toString());
                } catch (NumberFormatException e) {
                    holder.numberView.setText(String.valueOf(newValue));
                    continue;
                }

                if (oldValue != newValue) {
                    animateScoreChange(holder.numberView, newValue, null);
                }
            }
        }

        new Handler(Looper.getMainLooper()).postDelayed(this::refreshCards, 600);
    }


    /**
     * Shows a dialog to edit the points of a player card.
     * @param cardItem The card item to be edited.
     * @param numberTextView The TextView displaying the number of points.
     */
    public void showEditDialog(CardItem cardItem, TextView numberTextView) {
        AddPointsDialog.showEditDialog(
                this,
                cardItem,
                numberTextView,
                cardManager,
                this::animateAllVisibleChanges
        );
    }

    /**
     * Shows a dialog to edit the name and color of a player card.
     * @param card The card item to be edited.
     * @param nameTextView The TextView displaying the name.
     * @param cardView The CardView of the item.
     */
    public void showNameAndColorEditDialog(CardItem card, TextView nameTextView, CardView cardView) {
        NameAndColorDialog.showNameAndColorEditDialog(
                this,
                cardManager,
                card,
                nameTextView,
                cardView,
                this::refreshCards,
                R.drawable.dialog_background
        );
    }

    /**
     * Adds a new player card to the list.
     */
    private void addNewCard() {
        String name = nameManager.getNextName();
        int initialValue = 0;
        int cardColor = colorManager.getNextColor();
        int index = cardManager.getNextIndex();

        CardItem newCardItem = new CardItem(name, initialValue, cardColor, index);

        cardManager.addNewCard(newCardItem);
        adapter.notifyItemInserted(cardManager.getCardItems().size() - 1);
    }


    /**
     * Deletes all player cards after a confirmation.
     */
    private void deleteAllCards() {
        ConfirmationDialogHelper dialogHelper = new ConfirmationDialogHelper(this);

        // Show a confirmation dialog before deleting all cards.
        dialogHelper.show(
                getString(R.string.reallydelete),
                () -> {
                    cardManager.deleteAllCards();
                    refreshCards();
                },
                R.string.confirm,
                R.string.abortButton,
                R.color.my_positive_color,
                R.color.my_negative_color,
                R.drawable.dialog_background
        );
    }

    /**
     * Opens a dialog to set the winning value for the game.
     */
    public void setWinningValue() {
        SetWinningValueDialog winningValueDialog = new SetWinningValueDialog(this);

        winningValueDialog.show(getString(R.string.set_value_for_winning_the_game),
                () -> this.cardManager.adaptWinnerButton(),
                R.string.confirm,
                R.string.abortButton,
                R.color.my_positive_color,
                R.color.my_negative_color,
                R.drawable.dialog_background,
                tuttoSettings
        );
    }

    /**
     * Opens a dialog to set the list of possible player names.
     */
    public void setPossibleNames() {
        SetPossibleNamesDialog possibleNamesDialog = new SetPossibleNamesDialog(this);

        possibleNamesDialog.show(getString(R.string.setPossibleNames),
                () -> {
                    // Re-initialize the name manager with the new names.
                    this.nameManager = new NameManager(this);
                },
                R.string.confirm,
                R.string.abortButton,
                R.color.my_positive_color,
                R.color.my_negative_color,
                R.drawable.dialog_background,
                tuttoSettings
        );
    }
    /**
     * Returns the app's settings manager.
     * @return The TuttoSettings instance.
     */
    public TuttoSettings getTuttoSettings() {
        return tuttoSettings;
    }

}
