package de.georgelsas.tuttocounter;

import android.content.Context;
import android.content.SharedPreferences;
import android.graphics.Color;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import androidx.appcompat.app.AlertDialog;
import androidx.core.content.ContextCompat;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * A dialog that displays the recent changes in player scores.
 */
public class LastChangesDialog {

    /**
     * Shows the dialog with the list of last changes.
     *
     * @param context The context.
     * @param backgroundDrawableResId The resource ID of the background drawable for the dialog.
     */
    public static void show(Context context, int backgroundDrawableResId) {
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setTitle(R.string.last_changes);
        View view = LayoutInflater.from(context).inflate(R.layout.dialog_card_changes, null);
        LinearLayout container = view.findViewById(R.id.changeListContainer);

        SharedPreferences sharedPreferences = context.getSharedPreferences("CardData", Context.MODE_PRIVATE);
        try {
            String historyJson = sharedPreferences.getString("cardHistory", "[]");
            JSONArray historyArray = new JSONArray(historyJson);

            List<CardChange> changes = new ArrayList<>();

            // Iterate in reverse: from old to new
            // 1. Current cards from "cards"
            String cardsJson = sharedPreferences.getString("cards", "[]");
            JSONArray cardsArray = new JSONArray(cardsJson);
            Map<String, Integer> currentCardValues = new HashMap<>();
            Map<String, Integer> cardColors = new HashMap<>();

            for (int i = 0; i < cardsArray.length(); i++) {
                JSONObject card = cardsArray.getJSONObject(i);
                String name = card.getString("name");
                int number = card.getInt("number");
                int color = card.getInt("color");
                currentCardValues.put(name, number);
                cardColors.put(name, color);
            }

            // 2. Last history entry
            if (historyArray.length() > 0) {
                JSONObject lastSnapshot = historyArray.getJSONObject(historyArray.length() - 1);
                JSONArray lastItems = lastSnapshot.getJSONArray("cardItems");

                for (int j = 0; j < lastItems.length(); j++) {
                    JSONObject card = lastItems.getJSONObject(j);
                    String name = card.getString("name");
                    int oldValue = card.getInt("number");
                    int color = card.getInt("color");
                    int currentValue = currentCardValues.getOrDefault(name, oldValue);
                    int diff = currentValue - oldValue;
                    if (diff != 0) {
                        changes.add(new CardChange(name, diff, color));
                    }
                }
            }

            // 3. Then evaluate the history backwards
            Map<String, Integer> lastKnownValues = new HashMap<>();
            for (int i = 0; i < historyArray.length(); i++) {
                JSONObject snapshot = historyArray.getJSONObject(i);
                JSONArray items = snapshot.getJSONArray("cardItems");

                for (int j = 0; j < items.length(); j++) {
                    JSONObject card = items.getJSONObject(j);
                    String name = card.getString("name");
                    int number = card.getInt("number");
                    int color = card.getInt("color");

                    if (lastKnownValues.containsKey(name)) {
                        int diff = number - lastKnownValues.get(name);
                        if (diff != 0) {
                            changes.add(new CardChange(name, diff, color));
                        }
                    }
                    lastKnownValues.put(name, number);
                }
            }

            // Reverse, so that the newest changes are at the top
            Collections.reverse(changes);

            for (CardChange change : changes) {
                View item = LayoutInflater.from(context).inflate(R.layout.card_change_item, container, false);

                View colorStrip = item.findViewById(R.id.colorStrip);
                TextView cardName = item.findViewById(R.id.cardName);
                TextView changeAmount = item.findViewById(R.id.changeAmount);

                colorStrip.setBackgroundColor(change.color);
                cardName.setText(change.name);

                String sign = change.amount > 0 ? "+" : "";
                changeAmount.setText(sign + change.amount);
                changeAmount.setTextColor(change.amount >= 0 ? Color.GREEN : Color.RED);

                container.addView(item);
            }

            builder.setView(view);
            builder.setPositiveButton(context.getText(R.string.close), null);
            AlertDialog dialog = builder.create();
            dialog.show();
            Window window = dialog.getWindow();
            if (window != null) {
                window.setBackgroundDrawable(ContextCompat.getDrawable(context, backgroundDrawableResId));
            }

        } catch (JSONException e) {
            Log.e( "TuttoError", Objects.requireNonNull(e.getLocalizedMessage()));
            Toast.makeText(context, R.string.error_loading_changes, Toast.LENGTH_SHORT).show();
        }
    }

    /**
     * A private inner class to represent a change in a card's value.
     */
    private static class CardChange {
        String name;
        int amount;
        int color;

        CardChange(String name, int amount, int color) {
            this.name = name;
            this.amount = amount;
            this.color = color;
        }
    }
}
