package de.georgelsas.tuttocounter;

import static android.view.View.GONE;
import static android.view.View.VISIBLE;

import android.content.Context;
import android.content.SharedPreferences;

import androidx.appcompat.app.ActionBar;

import com.google.android.material.floatingactionbutton.FloatingActionButton;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Manages the player cards, including their data, history, and persistence.
 */
public class CardManager {

    private final SharedPreferences sharedPreferences;
    private List<CardItem> cardItems;
    private List<CardHistoryItem> cardHistory;

    private TuttoSettings tuttoSettings;
    private FloatingActionButton winnerButton;
    private ActionBar toolbar;



    public CardManager(Context context, TuttoSettings tuttoSettings) {
        MainActivity activity = (MainActivity) context;
        this.winnerButton = activity.findViewById(R.id.winningButton);
        this.toolbar = activity.getSupportActionBar();

        this.sharedPreferences = context.getSharedPreferences("CardData", Context.MODE_PRIVATE);
        this.cardItems = new ArrayList<>();
        this.cardHistory = new ArrayList<>();
        this.tuttoSettings = tuttoSettings;

        // Load cards and history
        loadCards();
        loadHistory();
    }


    // Save cards
    public void saveCards() {
        Gson gson = new Gson();
        String json = gson.toJson(cardItems);

        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putString("cards", json);
        editor.apply();

    }

    // Save history
    private void saveHistory() {
        Gson gson = new Gson();
        String json = gson.toJson(cardHistory);

        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putString("cardHistory", json);
        editor.apply();
    }

    // Load cards
    public void loadCards() {
        Gson gson = new Gson();
        String json = sharedPreferences.getString("cards", null);

        if (json != null) {
            Type type = new TypeToken<List<CardItem>>(){}.getType();
            cardItems = gson.fromJson(json, type);
        }
    }

    // Load history
    private void loadHistory() {
        Gson gson = new Gson();
        String json = sharedPreferences.getString("cardHistory", null);

        if (json != null) {
            Type type = new TypeToken<List<CardHistoryItem>>(){}.getType();
            cardHistory = gson.fromJson(json, type);
        }
    }


    public void addNewCard(CardItem newCardItem) {
        cardItems.add(newCardItem);

        // Add a history entry
        addHistorySnapshot();

        // Save
        saveCards();
    }

    // Update card number
    public void updateCardNumber(int index, int newNumber) {
        if (index >= 0) {
            for (CardItem card : cardItems)
                {
                if (card.getIndex() == index) {
                    card.setNumber(newNumber);
                }
            }
            adaptWinnerButton();

            adaptLeaderNameOnToolbar();
        }
    }

    private void adaptLeaderNameOnToolbar() {
        int  maxValue = Integer.MIN_VALUE;
        String leader = "";
        for (CardItem card : cardItems) {
            if (maxValue < card.getNumber()){
                maxValue = card.getNumber();
            }
        }
        for (CardItem card : cardItems) {
            if (maxValue == card.getNumber()){
                leader = leader.concat(card.getName());
                // add a little whitespace
                leader = leader.concat(" ");
            }
        }
        this.toolbar.setTitle(leader);
    }

    public void adaptWinnerButton() {
        if (isWinnerAvailable()) {
            winnerButton.setVisibility(VISIBLE);
            winnerButton.setClickable(true);
            winnerButton.setFocusable(true);
        }
        else {
            winnerButton.setVisibility(GONE);
            winnerButton.setClickable(false);
            winnerButton.setFocusable(false);
        }
    }

    // Add history snapshot
    public void addHistorySnapshot() {
        List<CardItem> cardItemsCopy = new ArrayList<>();
        for (CardItem card : cardItems) {
            CardItem cardCopy = new CardItem(card.getName(), card.getNumber(),
                                            card.getColor(), card.getIndex());
            cardItemsCopy.add(cardCopy);
        }
        CardHistoryItem historyItem = new CardHistoryItem(cardItemsCopy, System.currentTimeMillis());
        cardHistory.add(historyItem);
        saveHistory();
    }

    // Get card history
    public List<CardHistoryItem> getCardHistory() {
        return cardHistory;
    }

    // Delete all cards
    public void deleteAllCards() {
        cardItems.clear();
        deleteHistory();
        saveCards();
    }

    // Get card items
    public List<CardItem> getCardItems() {
        return cardItems;
    }

    // Revert to a specific history point
    public void revertToHistoryVersion(int index) {
        if (index >= 0 && index < cardHistory.size()) {
            CardHistoryItem historyItem = cardHistory.get(index);
            cardItems = new ArrayList<>(historyItem.getCardItems());
        }
    }
    public void undoLast() {
        if (cardHistory.size() < 2) {
            return;
        }
        revertToHistoryVersion(cardHistory.size() - 2);
        cardHistory.remove(cardHistory.size() - 1);
        saveHistory();
        saveCards();
        adaptWinnerButton();
        adaptLeaderNameOnToolbar();
    }

    public int getNextIndex() {
        return cardItems.size();
    }

    public void deleteHistory() {
        this.cardHistory = new ArrayList<>();
        addHistorySnapshot();
    }

    public void swapCards(int fromPosition, int toPosition) {
        if (fromPosition < 0 || toPosition < 0 ||
                fromPosition >= cardItems.size() || toPosition >= cardItems.size()) return;

        // Swap
        CardItem fromCard = cardItems.get(fromPosition);
        cardItems.set(fromPosition, cardItems.get(toPosition));
        cardItems.set(toPosition, fromCard);

        // Optional: Update indices
        for (int i = 0; i < cardItems.size(); i++) {
            cardItems.get(i).setIndex(i);
        }

        // Save history
        addHistorySnapshot();

        // Save
        saveCards();
    }
    public void removeCard(CardItem card) {
        Iterator<CardItem> iterator = cardItems.iterator();
        while (iterator.hasNext()) {
            CardItem current = iterator.next();
            if (current.getIndex() == card.getIndex()) {
                iterator.remove();
                break;
            }
        }
        addHistorySnapshot();
        saveCards();
    }

    public boolean isWinnerAvailable() {
        if (!cardItems.isEmpty()) {
            for (CardItem card : cardItems) {
                if (card.getNumber() >= this.tuttoSettings.getWinningValue()){
                    return true;
                }
            }
        }
        return false;
    }


}
