package de.ferreum.pto.page.media

import android.content.Intent
import android.net.Uri
import androidx.core.content.IntentCompat
import androidx.core.net.toUri
import de.ferreum.pto.page.media.MarkdownMediaIntentHandler.MIMETYPE_TEXT_PLAIN
import io.mockk.every
import io.mockk.mockk
import io.mockk.mockkStatic
import io.mockk.unmockkAll
import org.junit.After
import org.junit.Assert.assertEquals
import org.junit.Before
import org.junit.Test

internal class MarkdownMediaIntentHandlerTest {

    @Before
    fun setUp() {
        mockkStatic(Uri::parse)
        every { Uri.parse(any()) } answers {
            val s: String = firstArg()
            mockk {
                every { this@mockk.toString() } returns s
                every { this@mockk.scheme } returns s.substringBefore(":")
                every { this@mockk == any() } answers {
                    val other: Any = firstArg()
                    other is Uri
                    other.toString() == s
                }
            }
        }
        mockkStatic(IntentCompat::class)
    }

    @After
    fun tearDown() {
        unmockkAll()
    }

    @Test
    fun `plaintext containing link without title is returned as link`() {
        val intent = mkIntent(
            type = MIMETYPE_TEXT_PLAIN,
            text = "https://example.com/12",
        )

        val result = MarkdownMediaIntentHandler.getShareIntentLinks(intent)

        assertEquals(listOf(MarkdownMediaIntentHandler.Link(
            null,
            "https://example.com/12".toUri(),
            "<https://example.com/12>",
        )), result)
    }

    @Test
    fun `plaintext containing link with title is returned as link`() {
        val intent = mkIntent(
            type = MIMETYPE_TEXT_PLAIN,
            text = "https://example.com/12",
            title = "title12",
        )

        val result = MarkdownMediaIntentHandler.getShareIntentLinks(intent)

        assertEquals(listOf(MarkdownMediaIntentHandler.Link(
            "title12",
            "https://example.com/12".toUri(),
            "[title12](https://example.com/12)",
        )), result)
    }

    @Test
    fun `plaintext without text is returned as is`() {
        val intent = mkIntent(
            type = MIMETYPE_TEXT_PLAIN,
            text = "test text",
        )

        val result = MarkdownMediaIntentHandler.getShareIntentLinks(intent)

        assertEquals(listOf(MarkdownMediaIntentHandler.Link(
            "test text",
            null,
            "test text",
        )), result)
    }

    @Test
    fun `plaintext multiline links are returned as is`() {
        val intent = mkIntent(
            type = MIMETYPE_TEXT_PLAIN,
            text = "https://example.com/12\nhttps://example.com/23",
            title = "title12",
        )

        val result = MarkdownMediaIntentHandler.getShareIntentLinks(intent)

        assertEquals(listOf(MarkdownMediaIntentHandler.Link(
            "https://example.com/12\nhttps://example.com/23",
            null,
            "https://example.com/12\nhttps://example.com/23",
        )), result)
    }

    @Test
    fun `plaintext title with stream is returned as link`() {
        val intent = mkIntent(
            type = MIMETYPE_TEXT_PLAIN,
            title = "title12",
            stream = "https://example.com/12".toUri(),
        )

        val result = MarkdownMediaIntentHandler.getShareIntentLinks(intent)

        assertEquals(listOf(MarkdownMediaIntentHandler.Link(
            "title12",
            "https://example.com/12".toUri(),
            "[title12](https://example.com/12)",
        )), result)
    }

    @Test
    fun `action send is returned as link`() {
        val intent = mkIntent(
            action = Intent.ACTION_SEND,
            stream = "https://example.com/12".toUri(),
        )

        val result = MarkdownMediaIntentHandler.getShareIntentLinks(intent)

        assertEquals(listOf(MarkdownMediaIntentHandler.Link(
            null,
            "https://example.com/12".toUri(),
            "<https://example.com/12>",
        )), result)
    }

    @Test
    fun `action send prefers http url from text`() {
        val intent = mkIntent(
            action = Intent.ACTION_SEND,
            text = "https://example.com/234",
            stream = "https://example.com/12".toUri(),
        )

        val result = MarkdownMediaIntentHandler.getShareIntentLinks(intent)

        assertEquals(listOf(MarkdownMediaIntentHandler.Link(
            null,
            "https://example.com/234".toUri(),
            "<https://example.com/234>",
        )), result)
    }

    @Test
    fun `action send multiple returns list of links`() {
        val intent = mkIntent(
            action = Intent.ACTION_SEND_MULTIPLE,
            streams = listOf(
                "https://example.com/1".toUri(),
                "https://example.com/2".toUri(),
                "https://example.com/3".toUri(),
            ),
        )

        val result = MarkdownMediaIntentHandler.getShareIntentLinks(intent)

        assertEquals(listOf(
            MarkdownMediaIntentHandler.Link(null, "https://example.com/1".toUri(), "<https://example.com/1>"),
            MarkdownMediaIntentHandler.Link(null, "https://example.com/2".toUri(), "<https://example.com/2>"),
            MarkdownMediaIntentHandler.Link(null, "https://example.com/3".toUri(), "<https://example.com/3>"),
        ), result)
    }

    @Suppress("DEPRECATION")
    private fun mkIntent(
        action: String? = null,
        type: String? = null,
        text: String? = null,
        title: String? = null,
        subject: String? = null,
        data: Uri? = null,
        stream: Uri? = null,
        streams: List<Uri>? = null,
    ) = mockk<Intent> {
        every { this@mockk.action } returns action
        every { this@mockk.type } returns type
        every { this@mockk.data } returns data
        every { this@mockk.getStringExtra(Intent.EXTRA_TEXT) } returns text
        every { this@mockk.getStringExtra(Intent.EXTRA_TITLE) } returns title
        every { this@mockk.getStringExtra(Intent.EXTRA_SUBJECT) } returns subject
        every { this@mockk.getParcelableExtra<Uri>(Intent.EXTRA_STREAM) } returns stream
        every { this@mockk.getParcelableExtra<Uri>(Intent.EXTRA_STREAM, Uri::class.java) } returns stream
        every { this@mockk.getParcelableArrayListExtra<Uri>(Intent.EXTRA_STREAM) } returns streams?.let { ArrayList(it) }
        every { this@mockk.getParcelableArrayListExtra<Uri>(Intent.EXTRA_STREAM, Uri::class.java) } returns streams?.let { ArrayList(it) }
    }
}
