import 'package:flutter_bloc/flutter_bloc.dart';

import 'package:calibre_web_companion/features/shelf_view.dart/bloc/shelf_view_bloc.dart';
import 'package:calibre_web_companion/features/shelf_view.dart/bloc/shelf_view_event.dart';
import 'package:calibre_web_companion/features/shelf_details/bloc/shelf_details_event.dart';
import 'package:calibre_web_companion/features/shelf_details/bloc/shelf_details_state.dart';

import 'package:calibre_web_companion/features/shelf_details/data/repositories/shelf_details_repository.dart';

class ShelfDetailsBloc extends Bloc<ShelfDetailsEvent, ShelfDetailsState> {
  final ShelfDetailsRepository repository;
  final ShelfViewBloc shelfViewBloc;

  ShelfDetailsBloc({required this.repository, required this.shelfViewBloc})
    : super(const ShelfDetailsState()) {
    on<LoadShelfDetails>(_onLoadShelfDetails);
    on<RemoveFromShelf>(_onRemoveFromShelf);
    on<EditShelf>(_onEditShelf);
    on<DeleteShelf>(_onDeleteShelf);
  }

  Future<void> _onLoadShelfDetails(
    LoadShelfDetails event,
    Emitter<ShelfDetailsState> emit,
  ) async {
    emit(state.copyWith(status: ShelfDetailsStatus.loading));

    try {
      final result = await repository.getShelfDetails(event.shelfId);

      final mergedResult = result.copyWith(
        name: event.shelfTitle,
        isPublic: result.isPublic || event.isPublic,
      );

      emit(
        state.copyWith(
          status: ShelfDetailsStatus.loaded,
          currentShelfDetail: mergedResult,
          errorMessage: null,
        ),
      );
    } catch (e) {
      emit(
        state.copyWith(
          status: ShelfDetailsStatus.error,
          errorMessage: e.toString(),
        ),
      );
    }
  }

  Future<void> _onRemoveFromShelf(
    RemoveFromShelf event,
    Emitter<ShelfDetailsState> emit,
  ) async {
    emit(state.copyWith(actionDetailsStatus: ShelfDetailsActionStatus.loading));

    try {
      final success = await repository.removeFromShelf(
        event.shelfId,
        event.bookId,
      );

      if (success) {
        emit(
          state.copyWith(
            actionDetailsStatus: ShelfDetailsActionStatus.success,
            actionMessage: 'Book removed from shelf successfully',
          ),
        );

        shelfViewBloc.add(RemoveShelfFromState(event.shelfId));
      } else {
        emit(
          state.copyWith(
            actionDetailsStatus: ShelfDetailsActionStatus.error,
            actionMessage: 'Failed to remove book from shelf',
          ),
        );
      }
    } catch (e) {
      emit(
        state.copyWith(
          actionDetailsStatus: ShelfDetailsActionStatus.error,
          actionMessage: e.toString(),
        ),
      );
      return;
    }
  }

  Future<void> _onEditShelf(
    EditShelf event,
    Emitter<ShelfDetailsState> emit,
  ) async {
    emit(
      state.copyWith(
        actionDetailsStatus: ShelfDetailsActionStatus.loading,
        actionMessage: null,
      ),
    );

    try {
      final success = await repository.editShelf(
        event.shelfId,
        event.newShelfName,
        isPublic: event.isPublic,
      );
      if (success) {
        emit(
          state.copyWith(
            currentShelfDetail: state.currentShelfDetail!.copyWith(
              name: event.newShelfName,
              isPublic: event.isPublic,
            ),
            actionDetailsStatus: ShelfDetailsActionStatus.success,
            actionMessage: 'Shelf edited successfully',
          ),
        );

        // Hier isPublic mit übergeben
        shelfViewBloc.add(
          EditShelfState(
            event.shelfId,
            event.newShelfName,
            isPublic: event.isPublic,
          ),
        );
      } else {
        emit(
          state.copyWith(
            actionDetailsStatus: ShelfDetailsActionStatus.error,
            actionMessage: 'Failed to edit shelf',
          ),
        );
      }
    } catch (e) {
      emit(
        state.copyWith(
          actionDetailsStatus: ShelfDetailsActionStatus.error,
          actionMessage: e.toString(),
        ),
      );
      return;
    }
  }

  Future<void> _onDeleteShelf(
    DeleteShelf event,
    Emitter<ShelfDetailsState> emit,
  ) async {
    emit(
      state.copyWith(
        actionDetailsStatus: ShelfDetailsActionStatus.loading,
        actionMessage: null,
      ),
    );

    try {
      final success = await repository.deleteShelf(event.shelfId);

      if (success) {
        emit(
          state.copyWith(
            actionDetailsStatus: ShelfDetailsActionStatus.success,
            actionMessage: 'Shelf deleted successfully',
          ),
        );
      } else {
        emit(
          state.copyWith(
            actionDetailsStatus: ShelfDetailsActionStatus.error,
            actionMessage: 'Failed to delete shelf',
          ),
        );
      }
    } catch (e) {
      emit(
        state.copyWith(
          actionDetailsStatus: ShelfDetailsActionStatus.error,
          actionMessage: e.toString(),
        ),
      );
      return;
    }
  }
}
