package de.ciluvien.mensen.ui.screens

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Scaffold
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.pulltorefresh.PullToRefreshBox
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalContext
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavController
import de.ciluvien.mensen.MensenApp
import de.ciluvien.mensen.R
import de.ciluvien.mensen.ui.components.CanteenList
import de.ciluvien.mensen.ui.components.core.BottomNavigationBar
import de.ciluvien.mensen.ui.components.core.SettingsTopAppBar
import de.ciluvien.mensen.ui.components.core.NothingFoundRefresh
import de.ciluvien.mensen.ui.viewModelFactory
import de.ciluvien.mensen.ui.viewmodels.CanteenListViewModel
import sh.calvin.reorderable.rememberReorderableLazyListState
import java.time.LocalDate

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun CanteensScreen(
    navController: NavController,
    onNavigateSettings: () -> Unit,
    onNavigateMenu: (Int, LocalDate) -> Unit
) {
    val context = LocalContext.current
    val canteenViewModel = viewModel<CanteenListViewModel>(
        factory = viewModelFactory {
            CanteenListViewModel(MensenApp.appModule)
        }
    )

    LaunchedEffect(key1 = true) {
        canteenViewModel.refreshState()
    }

    val lazyListState = rememberLazyListState()
    val reorderableLazyListState = rememberReorderableLazyListState(lazyListState = lazyListState) { from, to ->
        canteenViewModel.changePosition(from.index, to.index)
    }

    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior()
    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        topBar = {
            SettingsTopAppBar(
                onNavigateSettings = onNavigateSettings,
                title = context.resources.getText(R.string.canteens_title).toString(),
                scrollBehavior = scrollBehavior
            )
        },
        bottomBar = { BottomNavigationBar(navController = navController)}
    ) {padding ->
        Box(modifier = Modifier.padding(padding)){
            if (canteenViewModel.state.value.canteensAndPositions.isEmpty()) {
                NothingFoundRefresh(text = context.resources.getText(R.string.canteens_no_canteen).toString()) {
                    canteenViewModel.forceRemoteRefresh()
                }
            } else {
                PullToRefreshBox(
                    isRefreshing = canteenViewModel.state.value.isRefreshing,
                    onRefresh = { canteenViewModel.forceRemoteRefresh() }
                ) {
                    CanteenList(
                        onNavigateMenu = onNavigateMenu,
                        scrollBehavior = scrollBehavior,
                        listState = lazyListState,
                        reorderableLazyListState = reorderableLazyListState,
                        canteenList = canteenViewModel.state.value.canteensAndPositions,
                        bookmarkedCanteens = canteenViewModel.state.value.bookmarkedCanteens
                    )
                }
            }
        }
    }
}
