package de.ciluvien.mensen.ui.components.settings

import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.selection.selectable
import androidx.compose.foundation.selection.selectableGroup
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.RadioButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.unit.dp
import de.ciluvien.mensen.R
import de.ciluvien.mensen.ui.viewmodels.SettingsViewModel

@Composable
fun Filters(settingsViewModel: SettingsViewModel) {
    val context = LocalContext.current

    Card(
        colors = CardDefaults.cardColors(
            containerColor = MaterialTheme.colorScheme.surfaceContainer,
        ),
        border = BorderStroke(2.dp, MaterialTheme.colorScheme.surfaceContainer),
        modifier = Modifier
            .fillMaxWidth()
            .padding(2.dp)
    ) {
        Column(
            modifier = Modifier
                .fillMaxWidth()
                .padding(10.dp),
        ) {
            Text(
                text = context.resources.getText(R.string.settings_filters).toString(),
                style = MaterialTheme.typography.headlineSmall,
                modifier = Modifier.align(Alignment.CenterHorizontally)
            )

            Spacer(modifier = Modifier.size(10.dp))

            Column(
                horizontalAlignment = Alignment.Start,
                verticalArrangement = Arrangement.spacedBy(5.dp),
                modifier = Modifier.fillMaxWidth().selectableGroup()
            ) {
                val isVegan = settingsViewModel.isVegan.collectAsState()
                val isVegetarian = settingsViewModel.isVegetarian.collectAsState()
                val isPescetarian = settingsViewModel.isPescetarian.collectAsState()
                val isEverything = !isVegan.value && !isVegetarian.value && !isPescetarian.value

                val veganColor = if (isVegan.value || isVegetarian.value || isPescetarian.value || isEverything) {
                    MaterialTheme.colorScheme.onSurface
                } else {
                    MaterialTheme.colorScheme.onSurfaceVariant
                }

                val vegetarianColor = if (isVegetarian.value || isPescetarian.value || isEverything) {
                    MaterialTheme.colorScheme.onSurface
                } else {
                    MaterialTheme.colorScheme.onSurfaceVariant
                }

                val pescetarianColor = if (isPescetarian.value || isEverything) {
                    MaterialTheme.colorScheme.onSurface
                } else {
                    MaterialTheme.colorScheme.onSurfaceVariant
                }

                val everythingColor = if (isEverything) {
                    MaterialTheme.colorScheme.onSurface
                } else {
                    MaterialTheme.colorScheme.onSurfaceVariant
                }


                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.Start,
                    modifier = Modifier.selectable(
                        selected = isVegan.value,
                        onClick = { settingsViewModel.setFilter(SettingsViewModel.Filter.VEGAN.ordinal) },
                        role = Role.RadioButton
                    )
                ) {
                    RadioButton(
                        selected = isVegan.value,
                        onClick = null
                    )
                    Spacer(Modifier.size(10.dp))
                    Text(
                        text = context.resources.getText(R.string.vegan).toString(),
                        style = MaterialTheme.typography.titleLarge,
                        color = veganColor
                    )
                    Spacer(Modifier.size(5.dp))
                    Icon(
                        painter = painterResource(id = R.drawable.leaf_24),
                        contentDescription = context.resources.getText(R.string.vegan).toString()
                    )
                }

                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.Start,
                    modifier = Modifier.selectable(
                        selected = isVegetarian.value,
                        onClick = { settingsViewModel.setFilter(SettingsViewModel.Filter.VEGETARIAN.ordinal) },
                        role = Role.RadioButton
                    )
                ) {
                    RadioButton(
                        selected = isVegetarian.value,
                        onClick = null
                    )
                    Spacer(Modifier.size(10.dp))
                    Text(
                        text = context.resources.getText(R.string.vegetarian).toString(),
                        style = MaterialTheme.typography.titleLarge,
                        color = vegetarianColor
                    )
                    Spacer(Modifier.size(5.dp))
                    Icon(
                        painter = painterResource(id = R.drawable.carrot_24),
                        contentDescription = context.resources.getText(R.string.vegetarian).toString()
                    )
                }

                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.Start,
                    modifier = Modifier.selectable(
                        selected = isPescetarian.value,
                        onClick = { settingsViewModel.setFilter(SettingsViewModel.Filter.PESCETARIAN.ordinal) },
                        role = Role.RadioButton
                    )
                ) {
                    RadioButton(
                        selected = isPescetarian.value,
                        onClick = null
                    )
                    Spacer(Modifier.size(10.dp))
                    Text(
                        text = context.resources.getText(R.string.pescetarian).toString(),
                        style = MaterialTheme.typography.titleLarge,
                        color = pescetarianColor
                    )
                    Spacer(Modifier.size(5.dp))
                    Icon(
                        painter = painterResource(id = R.drawable.fish_24),
                        contentDescription = context.resources.getText(R.string.pescetarian).toString()
                    )
                }
                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.Start,
                    modifier = Modifier.selectable(
                        selected = isEverything,
                        onClick = { settingsViewModel.setFilter(SettingsViewModel.Filter.EVERYTHING.ordinal) },
                        role = Role.RadioButton
                    )
                ) {
                    RadioButton(
                        selected = isEverything,
                        onClick = null
                    )
                    Spacer(Modifier.size(10.dp))
                    Text(
                        text = context.resources.getText(R.string.settings_filter_everything).toString(),
                        style = MaterialTheme.typography.titleLarge,
                        color = everythingColor
                    )
                }
            }
        }
    }
}
