package de.ciluvien.mensen.ui.components

import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.viewinterop.AndroidView
import de.ciluvien.mensen.R
import de.ciluvien.mensen.data.local.relations.CanteenAndPosition
import org.osmdroid.tileprovider.tilesource.TileSourceFactory
import org.osmdroid.util.GeoPoint
import org.osmdroid.views.CustomZoomButtonsController
import org.osmdroid.views.MapView
import org.osmdroid.views.overlay.CopyrightOverlay
import org.osmdroid.views.overlay.Marker
import java.time.LocalDate

@Composable
fun MapViewAdapter(
    canteensAndPositions: List<CanteenAndPosition>,
    onNavigateCanteen: (Int, LocalDate) -> Unit
) {
    val resourceContext = LocalContext.current
    val mapPinDrawable = remember {
        resourceContext.resources.getDrawable(
            R.drawable.baseline_location_pin_48,
            null
        )
    }
    mapPinDrawable.setTint(MaterialTheme.colorScheme.primary.toArgb())
    mapPinDrawable.colorFilter = PorterDuffColorFilter(Color.DKGRAY, PorterDuff.Mode.MULTIPLY)

    AndroidView(
        modifier = Modifier.fillMaxSize(),
        factory = { context ->
            MapView(context).apply {
                setTileSource(TileSourceFactory.DEFAULT_TILE_SOURCE)
                overlays.add(CopyrightOverlay(context))
                clipToOutline = true
                controller.setZoom(15.0)
                setExpectedCenter(GeoPoint(51.05080, 13.73832))
                setMultiTouchControls(true)
                maxZoomLevel = 20.0
                minZoomLevel = 9.0
                zoomController.setVisibility(CustomZoomButtonsController.Visibility.NEVER)
            }
        }
    ) { mapView ->
        canteensAndPositions.forEach { canteenAndPosition ->
            val canteen = canteenAndPosition.canteen
            val position = canteenAndPosition.position.position
            val onClick = { onNavigateCanteen(position, LocalDate.now()) }

            val marker = Marker(mapView)
            marker.position = GeoPoint(canteen.coordinates[0], canteen.coordinates[1])
            marker.title = canteen.name
            marker.infoWindow = CustomInfoWindow(mapView, resourceContext, onClick)
            marker.icon = mapPinDrawable
            marker.subDescription = canteen.address
            mapView.overlays.add(marker)
        }
    }
}