package de.ciluvien.mensen.ui.components

import android.content.Intent
import android.net.Uri
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import de.ciluvien.mensen.R
import de.ciluvien.mensen.data.local.Canteen
import de.ciluvien.mensen.ui.components.core.IconRow

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun CanteenDetails(
    canteen: Canteen,
    scrollBehavior: TopAppBarScrollBehavior = TopAppBarDefaults.pinnedScrollBehavior()
) {
    val context = LocalContext.current

    Column(
        Modifier
            .verticalScroll(rememberScrollState())
            .nestedScroll(scrollBehavior.nestedScrollConnection)
    ) {
        AsyncImage(
            model = canteen.imageURL,
            contentDescription = context.resources.getText(R.string.canteens_image).toString(),
            modifier = Modifier
                .fillMaxWidth()
                .background(Color.White)
        )

        Row(
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.Start,
            modifier = Modifier
                .fillMaxWidth()
                .clickable {
                    val coordinateString = "${canteen.coordinates[0]}, ${canteen.coordinates[1]}"
                    val geoLocation = Uri.parse(
                        "geo:${coordinateString}"
                                +"?q=${coordinateString}"
                    )
                    val intent = Intent(Intent.ACTION_VIEW).apply {
                        data = geoLocation
                    }
                    if (intent.resolveActivity(context.packageManager) != null) {
                        context.startActivity(intent)
                    }
                }
        ) {
            Column {
                IconRow(
                    iconId = R.drawable.baseline_map_24,
                    iconDescription = context.resources.getText(R.string.address).toString(),
                    text = listOf(canteen.address)
                )
                IconRow(
                    iconId = R.drawable.baseline_location_pin_24,
                    iconDescription = context.resources.getText(R.string.coordinates)
                        .toString(),
                    text = canteen.coordinates.map { double -> double.toString() }
                )
            }
            Spacer(Modifier.size(10.dp))
            Icon(
                painter = painterResource(id = R.drawable.baseline_open_in_new_24),
                contentDescription = context.resources.getText(R.string.meal_open_maps).toString()
            )
        }

        HorizontalDivider()

        if(canteen.specialOpeningTime.isNotEmpty()) {
            IconRow(
                iconId = R.drawable.baseline_more_time_24,
                iconDescription = context.resources.getText(R.string.canteen_opening_times_changed).toString(),
                text = canteen.specialOpeningTime.map { entry -> "${entry.key}:\n${entry.value.trim()}" }
            )
        }

        if(canteen.regularOpeningTime.isNotEmpty()) {
            IconRow(
                iconId = R.drawable.baseline_access_time_24,
                iconDescription = context.resources.getText(R.string.canteen_opening_times).toString(),
                text = canteen.regularOpeningTime.map { entry -> "${entry.key}:\n${entry.value.trim()}" }
            )
        }

        HorizontalDivider()

        Row(
            horizontalArrangement = Arrangement.Start,
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier
                .fillMaxWidth()
                .clickable {
                    val webpage: Uri = Uri.parse(canteen.detailURL)
                    val intent = Intent(Intent.ACTION_VIEW, webpage)
                    if (intent.resolveActivity(context.packageManager) != null) {
                        context.startActivity(intent)
                    }
                }
        ){
            IconRow(
                iconId = R.drawable.baseline_web_24,
                iconDescription = context.resources.getText(R.string.canteen_website).toString(),
                text = listOf(context.resources.getText(R.string.canteen_details_link).toString())
            )
            Spacer(Modifier.size(10.dp))
            Icon(
                painter = painterResource(id = R.drawable.baseline_open_in_new_24),
                contentDescription = context.resources.getText(R.string.canteen_link_to_website)
                    .toString()
            )
        }
    }
}