package de.ciluvien.mensen.ui.components

import android.content.Intent
import android.net.Uri
import androidx.compose.animation.core.animateDpAsState
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import de.ciluvien.mensen.R
import de.ciluvien.mensen.data.local.relations.CanteenAndPosition
import de.ciluvien.mensen.ui.components.core.IconRow
import sh.calvin.reorderable.ReorderableCollectionItemScope
import sh.calvin.reorderable.ReorderableItem
import sh.calvin.reorderable.ReorderableLazyListState
import java.time.LocalDate

@OptIn(ExperimentalMaterial3Api::class, ExperimentalFoundationApi::class)
@Composable
fun CanteenList(
    onNavigateMenu: (Int, LocalDate) -> Unit,
    scrollBehavior: TopAppBarScrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(),
    listState: LazyListState,
    reorderableLazyListState: ReorderableLazyListState,
    canteenList: List<CanteenAndPosition>,
    bookmarkedCanteens: List<Int>
) {
    Box(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection)
    ){
        LazyColumn(
            modifier = Modifier
                .fillMaxSize()
                .padding(horizontal = 5.dp),
            state = listState,
            verticalArrangement = Arrangement.spacedBy(10.dp)
        ) {
            // Spacer item to avoid reorderable list bug concerning the first item not animating correctly
            item {}

            items(
                items = canteenList,
                key = { item -> item.canteen.id }
            ) {item ->
                ReorderableItem(
                    state = reorderableLazyListState,
                    key = item.canteen.id
                ) {isDragging ->
                    val elevation = animateDpAsState(if (isDragging) 4.dp else 0.dp, label = "")
                    Surface(shadowElevation = elevation.value) {
                        CanteenListItem(
                            onNavigateMenu = onNavigateMenu,
                            canteenAndPosition = item,
                            reorderableCollectionItemScope = this,
                            hasBookmark = bookmarkedCanteens.contains(item.canteen.id)
                        )
                    }
                }
            }

            item {
                OtherCanteens()
            }

            item {}
        }
    }
}

@Composable
fun CanteenListItem(
    onNavigateMenu: (Int, LocalDate) -> Unit,
    canteenAndPosition: CanteenAndPosition,
    reorderableCollectionItemScope: ReorderableCollectionItemScope,
    hasBookmark: Boolean,
) {
    val context = LocalContext.current
    val interactionSource = remember { MutableInteractionSource() }

    Card(
        colors = CardDefaults.cardColors(
            containerColor = MaterialTheme.colorScheme.surfaceContainer
        ),
        border = BorderStroke(2.dp, MaterialTheme.colorScheme.surfaceContainer),
        modifier = Modifier.fillMaxSize(),
        onClick = { onNavigateMenu(canteenAndPosition.position.position, LocalDate.now()) },
        interactionSource = interactionSource
    ) {
        Row(
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.SpaceBetween,
            modifier = Modifier
                .fillMaxWidth()
                .padding(2.dp)
        ) {
            Row(
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.Start
            ) {
                AsyncImage(
                    model = canteenAndPosition.canteen.imageURL,
                    contentDescription = context.resources.getText(R.string.canteens_image).toString(),
                    contentScale = ContentScale.Crop,
                    modifier = Modifier
                        .size(120.dp, 80.dp)
                        .clip(RoundedCornerShape(10.dp))
                        .fillMaxHeight()
                )
                Spacer(modifier = Modifier.size(12.dp))
                Box(Modifier.fillMaxWidth(0.7f)) {
                    Text(
                        text = canteenAndPosition.canteen.name,
                        style = MaterialTheme.typography.titleLarge,
                    )
                }
                if (hasBookmark) {
                    Icon(
                        painter = painterResource(R.drawable.baseline_bookmark_24),
                        contentDescription = context.resources.getText(R.string.menu_bookmarked).toString()
                    )
                }
            }
            IconButton(
                modifier = with(reorderableCollectionItemScope) { Modifier.draggableHandle(interactionSource = interactionSource) },
                onClick = {},
            ) {
                Icon(
                    painter = painterResource(id = R.drawable.baseline_drag_indicator_24),
                    contentDescription = context.resources.getText(R.string.canteen_reorder).toString()
                )
            }
        }
    }
}

@Composable
fun OtherCanteens() {
    val context = LocalContext.current
    val canteensURL = "https://www.studentenwerk-dresden.de/mensen/mensen_cafeterien.html"

    Card(
        colors = CardDefaults.cardColors(
            containerColor = MaterialTheme.colorScheme.surfaceContainer
        ),
        border = BorderStroke(2.dp, MaterialTheme.colorScheme.surfaceContainer),
        modifier = Modifier
            .fillMaxWidth()
            .padding(2.dp),
        onClick = {
            val webpage: Uri = Uri.parse(canteensURL)
            val intent = Intent(Intent.ACTION_VIEW, webpage)
            if (intent.resolveActivity(context.packageManager) != null) {
                context.startActivity(intent)
            }
        },
    ) {
        Row(
            modifier = Modifier
                .fillMaxSize()
                .padding(5.dp),
            horizontalArrangement = Arrangement.Center,
            verticalAlignment = Alignment.CenterVertically
        ) {
            IconRow(
                iconId = R.drawable.baseline_web_24,
                iconDescription = context.getString(R.string.canteens_other_link),
                text = listOf(context.getText(R.string.canteens_others).toString()),
            )
        }
    }
}