package de.ciluvien.mensen.ui.viewmodels

import androidx.compose.runtime.mutableStateOf
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import de.ciluvien.mensen.AppModule
import de.ciluvien.mensen.data.local.Canteen
import de.ciluvien.mensen.data.local.Meal
import de.ciluvien.mensen.ui.states.MenuState
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import java.math.BigDecimal
import java.time.LocalDate

class MenuViewModel(
        appModule: AppModule
): ViewModel() {
    private val dao = appModule.canteenDao
    private val api = appModule.openMensaApi
    private val dataService = appModule.dataService
    private val settings = appModule.settings

    private val _state = mutableStateOf(MenuState())
    val state = _state

    private suspend fun refreshAvailableDates(canteenId: Int) {
        dataService.refreshDailyMenus(canteenId)
    }


    private suspend fun refreshMeals(canteenId: Int, date: LocalDate) {
        dataService.refreshMeals(canteenId, date)
    }

    private suspend fun setMeals(position: Int, canteenId: Int, date: LocalDate) {
        val meals = dao.getDailyMenuWithMeals(canteenId, date)

        val newMap = _state.value.meals.toMutableMap()
        newMap[position] = meals.filterMeals().sortMeals()

        _state.value = _state.value.copy(
            meals = newMap,
        )
    }

    private suspend fun setBookmarks(date: LocalDate) {
        _state.value = _state.value.copy(
            todaysBookmarks = dao.getBookmarkList().filter {it.date == date}
        )
    }

    private suspend fun setAllergensVisible() {
        _state.value = _state.value.copy(
            allergensVisibleSetting = settings.areAllergensVisibleFlow.first()
        )
    }

    private suspend fun List<Meal>.filterMeals(): List<Meal> {
        return this.filter { meal: Meal ->
            var result = true
            val isVegan = settings.isVeganFlow.first()
            val isVegetarian = settings.isVegetarianFlow.first()
            val isPescetarian = settings.isPescetarianFlow.first()
            if (isPescetarian) {
                result = meal.isPescetarian
            } else if (isVegetarian) {
                result = meal.isVegetarian
            } else if (isVegan) {
                result = meal.isVegan
            }
            result
        }
    }

    private fun List<Meal>.sortMeals(): List<Meal> {
        return this.sortedWith(
            compareBy(
                { it.isEveningOffer},
                { !state.value.todaysBookmarks.map { b -> b.mealId }.contains(it.id) },
                { it.isSoldOut },
                { it.prices["Studierende"]?:BigDecimal.ZERO }
            )
        )
    }

    private suspend fun areMealsStale(canteenId: Int, date: LocalDate): Boolean {
        return dao.getDailyMenu(canteenId, date)?.areMealsStale()
                ?: true
    }

    private suspend fun isMenuStale(canteenId: Int): Boolean {
        return dao.getCanteen(canteenId)?.isMenuStale()
            ?: true
    }

    private suspend fun setCanteen(canteenId: Int) {
        val canteen = dao.getCanteen(canteenId)
            ?: Canteen()
        _state.value = _state.value.copy(
            canteenName = canteen.name,
            canteenId = canteenId
        )
    }

    private suspend fun getCanteenId(position: Int): Int? {
        return dao.getPositionAndCanteen(position)?.canteenId
    }

    private suspend fun setPositionCount() {
        var count = dao.getPositionCount()
        if (count == 0) count = 1
        _state.value = _state.value.copy(
            positionCount = count,
        )
    }

    private suspend fun setAvailableDates(canteenId: Int) {
        _state.value = _state.value.copy(
            availableDates = dao.getCanteenDailyMenus(canteenId)
                .filter { menu -> !menu.closed}
                .map { menu -> menu.date }
        )
    }

    fun refreshState(position: Int, date: LocalDate) {
        viewModelScope.launch {
            setPositionCount()
            val canteenId = getCanteenId(position) ?: return@launch
            setCanteen(canteenId)
            setBookmarks(date)
            setAllergensVisible()

            setMeals(position, canteenId, date)

            if(isMenuStale(canteenId)){
                refreshAvailableDates(canteenId)
                refreshMeals(canteenId, date)
            }
            setAvailableDates(canteenId)
            if(areMealsStale(canteenId, date)){
                refreshMeals(canteenId, date)
            }
            setMeals(position, canteenId, date)
        }
    }

    suspend fun forceRemoteRefreshSuspend(position: Int, date: LocalDate) {
        onForceRefresh(position, date)
    }

    fun forceRemoteRefresh(position: Int, date: LocalDate) {
        viewModelScope.launch {
            onForceRefresh(position, date)
        }
    }

    private suspend fun onForceRefresh(position: Int, date: LocalDate){
        _state.value = _state.value.copy(
            isRefreshing = true
        )
        setPositionCount()
        val canteenId = getCanteenId(position) ?: return
        setCanteen(canteenId)
        setBookmarks(date)
        setAllergensVisible()

        refreshAvailableDates(canteenId)
        setAvailableDates(canteenId)

        refreshMeals(canteenId, date)
        setMeals(position, canteenId, date)
        _state.value = _state.value.copy(
            isRefreshing = false
        )
    }
}