package de.ciluvien.mensen.ui.viewmodels

import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.toMutableStateList
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import de.ciluvien.mensen.AppModule
import de.ciluvien.mensen.data.local.Organisation
import de.ciluvien.mensen.data.local.Position
import de.ciluvien.mensen.ui.states.CanteenListState
import kotlinx.coroutines.launch
import java.time.LocalDate

private const val DEFAULT_ORGANISATION = "Studentenwerk Dresden"

class CanteenListViewModel(
    appModule: AppModule
): ViewModel() {
    private val dao = appModule.canteenDao
    private val api = appModule.openMensaApi
    private val dataService = appModule.dataService

    private val _state = mutableStateOf(CanteenListState())
    val state = _state

    private suspend fun setCanteens(organisationName: String = DEFAULT_ORGANISATION) {
        val canteenPositions = dao.getCanteensAndPositionsOrdered()
        _state.value = _state.value.copy(canteensAndPositions = canteenPositions)
    }

    private suspend fun setBookmarkedCanteens() {
        val today = LocalDate.now()
        val bookmarkedCanteens = dao.getBookmarkList()
            .filter { b -> b.date == today }
            .mapNotNull { b -> dao.getMeal(b.mealId, today)?.canteenId }

        _state.value = _state.value.copy(
            bookmarkedCanteens = bookmarkedCanteens
        )
    }

    fun changePosition(from: Int, to: Int){
        _state.value = _state.value.copy(
            canteensAndPositions = _state.value.canteensAndPositions.toMutableStateList().apply {
                add(to - 1, removeAt(from - 1)) // account for spacer in reorderable list
            }
        )
        val list = _state.value.canteensAndPositions.mapIndexed { index, canteenAndPosition ->
            Position(index, canteenAndPosition.canteen.id)
        }
        viewModelScope.launch {
            dao.upsertPositionList(list)
            setCanteens()
        }
    }

    fun refreshState(organisationName: String = DEFAULT_ORGANISATION){
        viewModelScope.launch {
            setCanteens(organisationName)
            setBookmarkedCanteens()
            var organisation = dao.getOrganisation(organisationName)

            if (organisation == null) {
                organisation = Organisation(organisationName)
                dao.insertOrganisation(organisation)
            }

            if (state.value.canteensAndPositions.isEmpty() || organisation.areCanteensStale()) {
                dataService.refreshCanteens(organisationName)
                setCanteens(organisationName)
            }
        }
    }

    suspend fun forceRemoteRefreshSuspend() {
        onForceRefresh()
    }

    fun forceRemoteRefresh() {
        viewModelScope.launch {
            onForceRefresh()
        }
    }

    private suspend fun onForceRefresh(organisationName: String = DEFAULT_ORGANISATION) {
        _state.value = _state.value.copy(
            isRefreshing = true
        )
        dataService.refreshCanteens(organisationName)
        setCanteens(organisationName)
        setBookmarkedCanteens()
        _state.value = _state.value.copy(
            isRefreshing = false
        )
    }
}

