package de.ciluvien.mensen.ui.screens

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.KeyboardArrowLeft
import androidx.compose.material.icons.automirrored.filled.KeyboardArrowRight
import androidx.compose.material.icons.outlined.Info
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DatePickerDialog
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SelectableDates
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.pulltorefresh.PullToRefreshBox
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.font.FontWeight
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavController
import de.ciluvien.mensen.MensenApp
import de.ciluvien.mensen.R
import de.ciluvien.mensen.ui.components.MenuList
import de.ciluvien.mensen.ui.components.core.BottomNavigationBar
import de.ciluvien.mensen.ui.components.core.SettingsTopAppBar
import de.ciluvien.mensen.ui.components.core.CustomTextIcon
import de.ciluvien.mensen.ui.components.core.NothingFoundRefresh
import de.ciluvien.mensen.ui.viewModelFactory
import de.ciluvien.mensen.ui.viewmodels.MenuViewModel
import java.time.Instant
import java.time.LocalDate
import java.time.ZoneId
import java.time.ZoneOffset
import java.time.format.DateTimeFormatter

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun MenuScreen(
    navController: NavController,
    initialPosition: Int,
    date: LocalDate,
    onNavigateSettings: () -> Unit,
    onNavigateMenu: (Int, LocalDate) -> Unit,
    onNavigateMealDetails: (Int, String, String, String) -> Unit,
    onNavigateCanteenDetails: (Int) -> Unit,
    onNavigateCanteenList: () -> Unit,
) {
    val context = LocalContext.current
    val menuViewModel = viewModel<MenuViewModel>(
        factory = viewModelFactory {
            MenuViewModel(MensenApp.appModule)
        }
    )

    val pagerState = rememberPagerState(initialPage = initialPosition, pageCount = { menuViewModel.state.value.positionCount })

    LaunchedEffect(key1 = pagerState.currentPage) {
        menuViewModel.refreshState(pagerState.currentPage, date)
    }

    val selectableDates = remember {
        object : SelectableDates {
            override fun isSelectableDate(utcTimeMillis: Long): Boolean {
                val dates = menuViewModel.state.value.availableDates.sorted()
                if (dates.isEmpty()) {
                    return false
                }
                val isAvailable = dates.contains(
                    LocalDate.from(Instant.ofEpochMilli(utcTimeMillis).atZone(ZoneOffset.UTC))
                )

                return super.isSelectableDate(utcTimeMillis) && isAvailable
            }
        }
    }
    val datePickerState = rememberDatePickerState(
        initialSelectedDateMillis = date.atStartOfDay(ZoneOffset.UTC).toInstant().toEpochMilli(),
        selectableDates = selectableDates
    )
    val showDatePicker = remember { mutableStateOf(false) }
    if (showDatePicker.value){
        DatePickerDialog(
            onDismissRequest = {
                showDatePicker.value = false
                datePickerState.selectedDateMillis = date.atStartOfDay(ZoneOffset.UTC).toInstant().toEpochMilli()
            },
            confirmButton = {
                val selected = datePickerState.selectedDateMillis
                if(selected != null && datePickerState.selectableDates.isSelectableDate(selected)) {
                    TextButton(onClick = {
                        val selectedDate = Instant.ofEpochMilli(datePickerState.selectedDateMillis!!).atZone(
                            ZoneId.of("Z")).toLocalDate()
                        showDatePicker.value = false
                        onNavigateMenu(pagerState.currentPage, selectedDate)
                    }) {
                        Text(text = context.resources.getText(R.string.menu_date_select_confirm).toString())
                    }
                }
            }
        ) {
            DatePicker(state = datePickerState)
        }
    }

    val isMenuErroneous = menuViewModel.state.value.meals[pagerState.currentPage] == null

    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior()
    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        topBar = {
            val leftIcon = if (!isMenuErroneous) {
                CustomTextIcon(
                    onClick = { onNavigateCanteenDetails(menuViewModel.state.value.canteenId) },
                    icon = Icons.Outlined.Info,
                    description = context.resources.getText(R.string.menu_information).toString()
                )
            } else {
                null
            }
            SettingsTopAppBar(
                onNavigateSettings = onNavigateSettings,
                title = menuViewModel.state.value.canteenName,
                scrollBehavior = scrollBehavior,
                leftIcon = leftIcon
            )
        },
        bottomBar = { BottomNavigationBar(navController) },
        floatingActionButton = {
            if (!isMenuErroneous) {
                FloatingActionButton(
                    onClick = { showDatePicker.value = true},
                    content = {
                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            horizontalArrangement = Arrangement.SpaceBetween
                        ) {
                            IconButton(
                                onClick = {
                                    val dayBefore = selectDayBefore(
                                        date,
                                        menuViewModel.state.value.availableDates
                                    )
                                    if (dayBefore != null) {
                                        onNavigateMenu(
                                            pagerState.currentPage,
                                            dayBefore
                                        )
                                    }
                                }
                            ) {
                                Icon(
                                    imageVector = Icons.AutoMirrored.Filled.KeyboardArrowLeft,
                                    contentDescription = context.resources.getText(R.string.menu_previous_day).toString()
                                )
                            }
                            Text(
                                text = date.format(DateTimeFormatter.ofPattern("dd.MM.")),
                                style = MaterialTheme.typography.titleLarge,
                                fontWeight = FontWeight.Bold
                            )
                            IconButton(
                                onClick = {
                                    val dayAfter = selectDayAfter(
                                        date,
                                        menuViewModel.state.value.availableDates
                                    )
                                    if (dayAfter != null) {
                                        onNavigateMenu(
                                            pagerState.currentPage,
                                            dayAfter
                                        )
                                    }
                                }
                            ) {
                                Icon(
                                    imageVector = Icons.AutoMirrored.Filled.KeyboardArrowRight,
                                    contentDescription = context.resources.getText(R.string.menu_next_day).toString()
                                )
                            }
                        }
                    }
                )
            }
        }
    ) { padding ->
        Box(modifier = Modifier.padding(padding)){
            HorizontalPager(state = pagerState) { position ->
                val menu = menuViewModel.state.value.meals[position]
                if (menu == null) {
                    NothingFoundRefresh(
                        text = context.resources.getText(R.string.menu_missing_canteen).toString(),
                        onClickRefresh = onNavigateCanteenList
                    )
                } else if (menu.isEmpty()) {
                    NothingFoundRefresh(
                        text = context.resources.getText(R.string.menu_no_meals).toString(),
                        onClickRefresh = {
                            menuViewModel.forceRemoteRefresh(position, date)
                        }
                    )
                } else {
                    PullToRefreshBox(
                        isRefreshing = menuViewModel.state.value.isRefreshing,
                        onRefresh = { menuViewModel.forceRemoteRefresh(pagerState.currentPage, date) }
                    ) {
                        MenuList(
                            onNavigateMealDetails = onNavigateMealDetails,
                            scrollBehavior = scrollBehavior,
                            menu = menu,
                            bookmarkList = menuViewModel.state.value.todaysBookmarks,
                            canteenName = menuViewModel.state.value.canteenName,
                            allergensVisibleSetting = menuViewModel.state.value.allergensVisibleSetting
                        )
                    }
                }
            }
        }
    }
}

fun selectDayAfter(currentDate: LocalDate, availableDates: List<LocalDate>): LocalDate? {
    return availableDates.sorted().find { it.isAfter(currentDate) }
}

fun selectDayBefore(currentDate: LocalDate, availableDates: List<LocalDate>): LocalDate? {
    return availableDates.sorted().findLast { it.isBefore(currentDate) }
}
