package de.ciluvien.mensen.ui.screens

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.nfc.NfcAdapter
import android.nfc.TagLostException
import android.nfc.tech.IsoDep
import android.util.Log
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.ListItem
import androidx.compose.material3.ListItemDefaults.containerColor
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.core.content.ContextCompat
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavController
import de.ciluvien.mensen.MensenApp
import de.ciluvien.mensen.R
import de.ciluvien.mensen.data.local.CardBalance
import de.ciluvien.mensen.data.nfc.NfcBroadcastReceiver
import de.ciluvien.mensen.ui.components.core.BottomNavigationBar
import de.ciluvien.mensen.ui.components.core.SettingsTopAppBar
import de.ciluvien.mensen.ui.viewModelFactory
import de.ciluvien.mensen.ui.viewmodels.CardViewModel
import java.io.IOException
import java.math.BigDecimal
import java.text.NumberFormat
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import java.util.Currency
import java.util.Locale
import java.util.TimeZone


val SELECT_APP: ByteArray = byteArrayOf(
    0x90.toByte(), // CLA
    0x5A.toByte(), // SELECT APP
    0x00.toByte(), // P1
    0x00.toByte(), // P2
    0x03.toByte(), // Data Length
    0x5F.toByte(), 0x84.toByte(), 0x15.toByte(), // AppId
    0x00.toByte(), // Maximum data answer length
)

val READ_VALUE: ByteArray = byteArrayOf(
    0x90.toByte(), // CLA
    0x6C.toByte(), // READ VALUE
    0x00.toByte(), // P1
    0x00.toByte(), // P2
    0x01.toByte(), // Data Length
    0x01.toByte(), // File
    0x00.toByte(), // Maximum data answer length
)

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun CardScreen(
    navController: NavController,
    onNavigateSettings: () -> Unit
) {
    val context = LocalContext.current
    val viewModel = viewModel<CardViewModel>(
        factory = viewModelFactory {
            CardViewModel(MensenApp.appModule)
        }
    )

    LaunchedEffect(key1 = true) {
        viewModel.setBalanceList()
    }

    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior()
    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        topBar = {
            SettingsTopAppBar(
                onNavigateSettings = onNavigateSettings,
                scrollBehavior = scrollBehavior,
                title = context.resources.getText(R.string.card_title).toString()
            )
        },
        bottomBar = { BottomNavigationBar(navController = navController) }
    ) {padding ->
        Box(modifier = Modifier.padding(padding)) {
            NfcBroadcastReceiver { genericTag ->
                val tag = IsoDep.get(genericTag)
                try {
                    tag.connect()
                    if (tag.isConnected) {
                        Log.i("nfc", "Selecting app")
                        val appResult = tag.transceive(SELECT_APP)
                        if (appResult[0] != 0x91.toByte()) {
                            Log.w("nfc", "Error selecting app: $appResult")
                            tag.close()
                            return@NfcBroadcastReceiver
                        }

                        Log.i("nfc", "Reading value")
                        val valueResult = tag.transceive(READ_VALUE)
                        if (valueResult.size >= 2 && valueResult[valueResult.size - 2] != 0x91.toByte()) {
                            Log.w("nfc", "Error reading value: $valueResult")
                            tag.close()
                            return@NfcBroadcastReceiver
                        }
                        tag.close()

                        viewModel.setCurrentBalance(
                            valueResult
                                .dropLast(2)
                                .reversed()
                                .toByteArray()
                                .toSingleInt()
                                .toBigDecimal()
                                .divide(BigDecimal.valueOf(1000))
                        )
                    }
                } catch (e: TagLostException) {
                    Log.w("nfc", "Error: Tag lost connection")
                } catch (e: IOException) {
                    Log.w("nfc", "Error: ${e.message}")
                } catch (e: SecurityException) {
                    Log.w("nfc", "Error: ${e.message}")
                } catch (e: Exception) {
                    Log.w("nfc", "Error: ${e.message}")
                }
            }

            val nfcAdapter = NfcAdapter.getDefaultAdapter(context.applicationContext)
            val nfcEnabled = remember { mutableStateOf(nfcAdapter?.isEnabled ?: false) }

            DisposableEffect(Unit) {
                val nfcStateReceiver = object : BroadcastReceiver() {
                    override fun onReceive(context: Context?, intent: Intent?) {
                        nfcEnabled.value = context?.let { isNFCEnabled(it) } == true
                    }
                }

                val filter = IntentFilter(NfcAdapter.ACTION_ADAPTER_STATE_CHANGED)
                ContextCompat.registerReceiver(
                    context,
                    nfcStateReceiver,
                    filter,
                    ContextCompat.RECEIVER_EXPORTED
                )

                onDispose {
                    context.unregisterReceiver(nfcStateReceiver)
                }
            }

            Box(
                modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection)
            ) {
                LazyColumn(
                    verticalArrangement = Arrangement.Top,
                    modifier = Modifier.fillMaxSize()
                ) {
                    item{
                        if (nfcAdapter == null) {
                            PlaceholderBalanceCard(
                                context.resources.getText(R.string.card_nfc_not_supported).toString()
                            )
                        } else if (!nfcEnabled.value) {
                            PlaceholderBalanceCard(
                                context.resources.getText(R.string.card_nfc_disabled).toString()
                            )
                        } else if (viewModel.cardState.value.currentBalance == null) {
                            PlaceholderBalanceCard(
                                context.resources.getText(R.string.card_hold_near_phone).toString()
                            )
                        } else {
                            BalanceCard(balance = viewModel.cardState.value.currentBalance!!)
                        }
                    }
                    item{
                        PreviousBalanceList(balanceList = viewModel.cardState.value.balanceList)
                    }
                    item{
                        if (nfcAdapter != null && nfcEnabled.value) {
                            NFCNotice()
                        }
                    }
                }
            }
        }
    }
}

fun isNFCEnabled(context: Context): Boolean {
    val nfcAdapter = NfcAdapter.getDefaultAdapter(context.applicationContext)
    return nfcAdapter?.isEnabled == true
}

@Composable
private fun BalanceCard(balance: BigDecimal) {
    val balanceText = NumberFormat.getCurrencyInstance(Locale.getDefault()).apply { currency = Currency.getInstance("EUR") }.format(balance)
    val context = LocalContext.current

    ElevatedCard(
        colors = CardDefaults.elevatedCardColors(MaterialTheme.colorScheme.primaryContainer),
        modifier = Modifier
            .fillMaxWidth()
            .aspectRatio(1.586f)
            .padding(20.dp)
    ) {
        Box(
            modifier = Modifier
                .fillMaxSize()
                .padding(16.dp)
        ) {
            Text(
                text = context.resources.getText(R.string.card_current_balance).toString(),
                style = MaterialTheme.typography.headlineSmall,
                modifier = Modifier.align(Alignment.TopStart)
            )
            Text(
                text = balanceText,
                color = MaterialTheme.colorScheme.onPrimaryContainer,
                style = MaterialTheme.typography.headlineLarge.copy(
                    fontSize = 45.sp,
                    fontWeight = FontWeight.Bold
                ),
                modifier = Modifier.align(Alignment.BottomEnd)
            )
        }
    }
}

@Composable
private fun PlaceholderBalanceCard(text: String) {
    ElevatedCard(
        colors = CardDefaults.elevatedCardColors(MaterialTheme.colorScheme.primaryContainer),
        modifier = Modifier
            .fillMaxWidth()
            .aspectRatio(1.586f)
            .padding(20.dp)
    ) {
        Box(
            modifier = Modifier
                .fillMaxSize()
                .padding(16.dp),
            contentAlignment = Alignment.Center,
        ) {
            Text(
                text = text,
                color = MaterialTheme.colorScheme.onPrimaryContainer,
                style = MaterialTheme.typography.headlineSmall,
                textAlign = TextAlign.Center,
                fontStyle = FontStyle.Italic
            )
        }
    }
}

@Composable
fun PreviousBalanceList(balanceList: List<CardBalance>) {
    if (balanceList.isEmpty()) return
    val context = LocalContext.current

    Column(
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxWidth()
            .padding(20.dp)
    ) {
        Text(
            text = context.resources.getText(R.string.card_balance_list).toString(),
            color = MaterialTheme.colorScheme.onSurface,
            style = MaterialTheme.typography.headlineSmall,
            textAlign = TextAlign.Center
        )
        Spacer(modifier = Modifier.size(10.dp))
        Card()
        {
            balanceList.forEach {
                val time = it.dateTime.atZone(TimeZone.getDefault().toZoneId()).format(DateTimeFormatter.ofLocalizedDateTime(FormatStyle.SHORT))
                val balance = NumberFormat.getCurrencyInstance(Locale.getDefault()).apply { currency = Currency.getInstance("EUR") }.format(it.balance)
                ListItem(
                    tonalElevation = 3.dp,
                    modifier = Modifier
                        .background(
                            color = containerColor,
                            shape = RoundedCornerShape(16.dp)
                        ),
                    headlineContent = {
                        Text(
                            text = balance,
                            style = MaterialTheme.typography.bodyLarge
                        )
                    },
                    trailingContent = {
                        Text(
                            text = time,
                            style = MaterialTheme.typography.bodyMedium
                        )
                    }
                )
            }
        }
    }
}

@Composable
fun NFCNotice() {
    val context = LocalContext.current

    Column(
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .fillMaxWidth()
            .padding(10.dp)
    ) {
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(8.dp),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.Center
        ) {
            Icon(
                painter = painterResource(id = R.drawable.baseline_info_24),
                contentDescription = context.resources.getText(R.string.card_reader_advice_icon).toString(),
                tint = MaterialTheme.colorScheme.secondary,
                modifier = Modifier.size(24.dp)
            )
            Text(
                text = context.resources.getText(R.string.card_reader_advice).toString(),
                style = MaterialTheme.typography.bodyLarge,
                modifier = Modifier.padding(start = 8.dp, end = 8.dp)
            )
        }
    }
}
fun ByteArray.toSingleInt(): Int {
    return this.fold(0) { acc, byte ->
        (acc shl 8) or (byte.toInt() and 0xFF)
    }
}
