package de.ciluvien.mensen.ui.components

import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.scale
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import de.ciluvien.mensen.R
import de.ciluvien.mensen.data.local.Bookmark
import de.ciluvien.mensen.data.local.Meal
import java.text.NumberFormat
import java.util.Currency
import java.util.Locale

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun MenuList(
    onNavigateMealDetails: (Int, String, String, String) -> Unit,
    scrollBehavior: TopAppBarScrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(),
    menu: List<Meal>,
    bookmarkList: List<Bookmark>,
    canteenName: String,
    allergensVisibleSetting: Boolean
) {
    val context = LocalContext.current
    val listState = rememberLazyListState()
    Box(modifier = Modifier
        .nestedScroll(scrollBehavior.nestedScrollConnection)
    ){
        LazyColumn(
            state = listState,
            modifier = Modifier
                .fillMaxSize()
                .padding(horizontal = 5.dp),
            verticalArrangement = Arrangement.spacedBy(10.dp)
        ) {
            if (!menu.containsEveningOffer()) {
                items(
                    items = menu,
                    key = {meal ->
                        meal.id
                    }
                ) { meal ->
                    MenuListItem(
                        onNavigateMealDetails = onNavigateMealDetails,
                        meal = meal,
                        mealIDList = menu.map {it.id},
                        canteenName = canteenName,
                        bookmarked = isMealBookmarked(meal.id, bookmarkList),
                        allergensVisible = allergensVisibleSetting
                    )
                }
            } else {
                val lunch = menu.filter { !it.isEveningOffer }
                val dinner = menu.filter { it.isEveningOffer }
                item {
                    MenuListText(text = context.resources.getText(R.string.menu_lunch).toString())
                }
                items(
                    items = lunch,
                    key = {meal ->
                        meal.id
                    }
                ) { meal ->
                    MenuListItem(
                        onNavigateMealDetails = onNavigateMealDetails,
                        meal = meal,
                        mealIDList = menu.map{ it.id },
                        canteenName = canteenName,
                        bookmarked = isMealBookmarked(meal.id, bookmarkList),
                        allergensVisible = allergensVisibleSetting
                    )
                }
                item {
                    MenuListText(text = context.resources.getText(R.string.menu_evening).toString())
                }
                items(
                    items = dinner,
                    key = {meal ->
                        meal.id
                    }
                ) { meal ->
                    MenuListItem(
                        onNavigateMealDetails = onNavigateMealDetails,
                        meal = meal,
                        mealIDList = menu.map {it.id},
                        canteenName = canteenName,
                        bookmarked = isMealBookmarked(meal.id, bookmarkList),
                        allergensVisible = allergensVisibleSetting
                    )
                }
            }
            item{Spacer(modifier = Modifier.size(80.dp))}
        }
    }
}

private fun List<Meal>.containsEveningOffer(): Boolean {
    return this.any{ meal -> meal.isEveningOffer}
}

private fun isMealBookmarked(mealId: Int, bookmarkList: List<Bookmark>): Boolean {
    return bookmarkList.map { bookmark -> bookmark.mealId }.contains(mealId)
}

@Composable
fun MenuListText(text: String) {
    Row(
        horizontalArrangement = Arrangement.Start,
        verticalAlignment = Alignment.CenterVertically,
        modifier = Modifier
            .fillMaxWidth()
            .padding(start = 12.dp)
    ) {
        Text(
            text = text,
            style = MaterialTheme.typography.titleLarge,
            fontWeight = FontWeight.Bold
        )
    }
}

@Composable
fun MenuListItem(
    onNavigateMealDetails: (Int, String, String, String) -> Unit,
    meal: Meal,
    mealIDList: List<Int>,
    canteenName: String,
    bookmarked: Boolean,
    allergensVisible: Boolean
) {
    val context = LocalContext.current

    val position = mealIDList.indexOf(meal.id)

    Card(
        colors = CardDefaults.cardColors(
            containerColor = MaterialTheme.colorScheme.surfaceContainer
        ),
        border = BorderStroke(2.dp, MaterialTheme.colorScheme.surfaceContainer),
        onClick = {
            onNavigateMealDetails(
                position,
                mealIDList.joinToString(","),
                meal.menuDate.toString(),
                canteenName
            )
        }
    ) {
        Row(
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.SpaceEvenly,
            modifier = Modifier
                .padding(2.dp)
                .padding(end = 5.dp)
                .height(IntrinsicSize.Min)
        ) {
            val isGenericImage = meal.imageULR.contains("lieber-mensen")
            if (isGenericImage) {
                Column(
                    verticalArrangement = Arrangement.Center,
                    modifier = Modifier
                        .fillMaxHeight()
                        .clip(RoundedCornerShape(10.dp))
                        .background(Color.White)
                ) {
                    AsyncImage(
                        model = meal.imageULR,
                        contentDescription = context.resources.getText(R.string.meal_image).toString(),
                        contentScale = ContentScale.Crop,
                        modifier = Modifier
                            .size(120.dp, 80.dp)
                    )
                }
            } else {
                AsyncImage(
                    model = meal.imageULR,
                    contentDescription = context.resources.getText(R.string.meal_image).toString(),
                    contentScale = ContentScale.Crop,
                    modifier = Modifier
                        .size(120.dp, 80.dp)
                        .clip(RoundedCornerShape(10.dp))
                )

            }

            Spacer(modifier = Modifier.size(5.dp))

            Column(
                modifier = Modifier
                    .padding(5.dp)
                    .fillMaxHeight(),
                verticalArrangement = Arrangement.SpaceBetween
            ) {
                Text(
                    text = if (!allergensVisible) meal.name.replace(Regex("\\s\\([A-Z0-9, ]*\\)"), "") else meal.name,
                    style = MaterialTheme.typography.bodyLarge
                )

                Spacer(Modifier.size(5.dp))

                Text(
                    text = meal.category,
                    style = MaterialTheme.typography.labelLarge,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )

                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.SpaceBetween,
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    if (meal.isSoldOut == true) {
                        Text(
                            text = context.resources.getText(R.string.meal_sold_out).toString(),
                            style = MaterialTheme.typography.labelLarge,
                            color = MaterialTheme.colorScheme.onErrorContainer
                        )
                    } else {
                        Text(
                            text = remember {
                                if (meal.prices.isNotEmpty()) {
                                    meal.prices.values.sorted().joinToString(separator = " | ") {
                                        NumberFormat.getCurrencyInstance(Locale.getDefault())
                                            .apply { currency = Currency.getInstance("EUR") }
                                            .format(it)
                                    }
                                } else {
                                    ""
                                }
                            },
                            style = MaterialTheme.typography.labelLarge,
                            color = MaterialTheme.colorScheme.onSurfaceVariant
                        )
                    }

                    Row {
                        if (meal.isVegan) {
                            Icon(
                                painter = painterResource(id = R.drawable.leaf_24),
                                contentDescription = context.resources.getText(R.string.vegan)
                                    .toString(),
                                modifier = Modifier.scale(0.75f)
                            )
                        } else if (meal.isVegetarian) {
                            Icon(
                                painter = painterResource(id = R.drawable.carrot_24),
                                contentDescription = context.resources.getText(R.string.vegetarian)
                                    .toString(),
                                modifier = Modifier.scale(0.75f)
                            )
                        } else if (meal.isPescetarian) {
                            Icon(
                                painter = painterResource(id = R.drawable.fish_24),
                                contentDescription = context.resources.getText(R.string.pescetarian)
                                    .toString(),
                                modifier = Modifier.scale(0.75f)
                            )
                        }
                        if (meal.generalNotes.contains("Menü ist klimafreundlich")) {
                            Icon(
                                painter = painterResource(R.drawable.earth_africa),
                                contentDescription = context.resources.getText(R.string.meal_climate_friendly).toString(),
                                modifier = Modifier.scale(0.75f)
                            )
                        }
                        if (bookmarked) {
                            Icon(
                                painter = painterResource(R.drawable.baseline_bookmark_24),
                                contentDescription = context.resources.getText(R.string.menu_bookmarked).toString()
                            )
                        }
                    }
                }
            }
        }
    }
}
