package de.ciluvien.mensen.ui.components

import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import de.ciluvien.mensen.R
import de.ciluvien.mensen.data.local.Meal
import de.ciluvien.mensen.ui.components.core.IconRow
import java.net.URLEncoder
import java.text.NumberFormat
import java.time.format.DateTimeFormatter
import java.util.Currency
import java.util.Locale
import java.util.stream.Collectors

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun MealDetails(
    meal: Meal,
    scrollBehavior: TopAppBarScrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(),
    onNavigateImage: (String) -> Unit
) {
    val context = LocalContext.current
    val prices = meal.prices.entries.stream().map { entry ->
        entry.key
            .plus(": ")
            .plus(
            NumberFormat.getCurrencyInstance(Locale.getDefault())
                .apply { currency = Currency.getInstance("EUR") }
                .format(entry.value)
            )
    }.collect(Collectors.toList())

    Column(
        Modifier
            .verticalScroll(rememberScrollState())
            .nestedScroll(scrollBehavior.nestedScrollConnection)
    ) {
        val encodedImageURL = URLEncoder.encode(meal.imageULR, "UTF-8")
        val backgroundColor = if (meal.imageULR.contains("lieber-mensen")) {
            Color.White
        } else {
            MaterialTheme.colorScheme.background
        }
        AsyncImage(
            model = meal.imageULR,
            contentDescription = context.resources.getText(R.string.meal_image).toString(),
            modifier = Modifier
                .background(backgroundColor)
                .fillMaxWidth()
                .clickable {
                    onNavigateImage(encodedImageURL)
                }
        )
        Row(
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier
                .fillMaxWidth()
                .padding(10.dp)
        ) {
            Text(
                text = meal.name,
                style = MaterialTheme.typography.headlineSmall
            )
        }
        HorizontalDivider()
        Row(
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.Top,
            modifier = Modifier.fillMaxWidth()
        ) {
            Column(
                modifier = Modifier.fillMaxWidth(0.45f)
            ) {
                IconRow(
                    iconId = R.drawable.baseline_storefront_24,
                    iconDescription = context.resources.getText(R.string.meal_category).toString(),
                    text = listOf(meal.category)
                )
                IconRow(
                    iconId = R.drawable.baseline_calendar_today_24,
                    iconDescription = context.resources.getText(R.string.meal_date).toString(),
                    text = listOf(meal.menuDate.format(DateTimeFormatter.ofPattern("d. MMMM")))
                )
            }

            if(meal.isSoldOut == true) {
                IconRow(
                    iconId = R.drawable.baseline_payment_24,
                    iconDescription = context.resources.getText(R.string.meal_prices).toString(),
                    text = listOf(context.resources.getText(R.string.meal_sold_out).toString())
                )
            } else if (meal.prices.isNotEmpty()) {
                IconRow(
                    iconId = R.drawable.baseline_payment_24,
                    iconDescription = context.resources.getText(R.string.meal_prices).toString(),
                    text = prices
                )
            }
        }
        HorizontalDivider()
        Column {
            val generalNotes = meal.generalNotes
            val additives = meal.additives
            val allergens = meal.allergens

            if (generalNotes.isNotEmpty()){
                IconRow(
                    iconId = R.drawable.baseline_info_24,
                    iconDescription = context.resources.getText(R.string.meal_general_notes).toString(),
                    text = generalNotes
                )
            }

            if (additives.isNotEmpty()) {
                IconRow(
                    iconId = R.drawable.baseline_add_circle_outline_24,
                    iconDescription = context.resources.getText(R.string.meal_additives).toString(),
                    text = additives
                )
            }

            if (allergens.isNotEmpty()) {
                IconRow(
                    iconId = R.drawable.baseline_medical_information_24,
                    iconDescription = context.resources.getText(R.string.meal_allergens).toString(),
                    text = allergens
                )
            }
        }
    }

}
