package de.ciluvien.mensen.ui

import androidx.compose.runtime.Composable
import androidx.navigation.NavHostController
import androidx.navigation.NavType
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.navArgument
import de.ciluvien.mensen.ui.screens.CanteenDetailsScreen
import de.ciluvien.mensen.ui.screens.CanteensScreen
import de.ciluvien.mensen.ui.screens.ImageScreen
import de.ciluvien.mensen.ui.screens.CardScreen
import de.ciluvien.mensen.ui.screens.Libraries
import de.ciluvien.mensen.ui.screens.MapScreen
import de.ciluvien.mensen.ui.screens.MealDetailsScreen
import de.ciluvien.mensen.ui.screens.MenuScreen
import de.ciluvien.mensen.ui.screens.SettingsScreen
import java.time.LocalDate

@Composable
fun Navigation(
    navController: NavHostController,
    startDestination: String = Screen.CanteenScreen.route
) {
    val onNavigateMenu: (Int, LocalDate) -> Unit = { position, date ->
        navController.navigate("${Screen.MenuScreen.route}?position=${position}?date=${date}")
    }
    val onNavigateSettings: () -> Unit = {
        navController.navigate(Screen.SettingsScreen.route)
    }
    val onNavigateImage: (String) -> Unit = { encodedImageURL ->
        navController.navigate("${Screen.ImageScreen.route}/${encodedImageURL}")
    }
    val onNavigateLibraries: () -> Unit = {
        navController.navigate(Screen.LibrariesScreen.route)
    }
    val onNavigateMealDetails: (Int, String, String, String) -> Unit = { position, mealIDs, date ,canteenName ->
        navController.navigate("${Screen.MealDetailsScreen.route}/${canteenName}/${date}/${mealIDs}/${position}")
    }
    val onNavigateCanteenDetails: (Int) -> Unit = { id ->
        navController.navigate("${Screen.CanteenDetailsScreen.route}/${id}")
    }
    val onNavigateCanteenList: () -> Unit = {
        navController.navigate(Screen.CanteenScreen.route) {
            launchSingleTop = true
        }
    }
    val onNavigateBack: () -> Unit = {
        navController.navigateUp()
    }
    NavHost(navController = navController, startDestination = startDestination){
        composable(route = Screen.CanteenScreen.route) {
            CanteensScreen(
                navController = navController,
                onNavigateSettings = onNavigateSettings,
                onNavigateMenu = onNavigateMenu
            )
        }
        composable(
            route = Screen.MenuScreen.route + "?position={position}?date={date}",
            arguments = listOf(
                navArgument("position") {
                    type = NavType.IntType
                    defaultValue = 0
                },
                navArgument("date") {
                    type = NavType.StringType
                    defaultValue = LocalDate.now().toString()
                }
            )
        ) { entry ->
                MenuScreen(
                    navController = navController,
                    initialPosition = entry.arguments?.getInt("position")!!,
                    date = LocalDate.parse(entry.arguments?.getString("date")),
                    onNavigateMenu = onNavigateMenu,
                    onNavigateSettings = onNavigateSettings,
                    onNavigateMealDetails = onNavigateMealDetails,
                    onNavigateCanteenDetails = onNavigateCanteenDetails,
                    onNavigateCanteenList = onNavigateCanteenList
                )
        }
        composable(
            route = Screen.MealDetailsScreen.route + "/{canteenName}/{date}/{mealIds}/{position}",
            arguments = listOf(
                navArgument("canteenName") {
                    type = NavType.StringType
                },
                navArgument("date") {
                    type = NavType.StringType
                },
                navArgument("mealIds") {
                    type = NavType.StringType
                },
                navArgument("position") {
                    type = NavType.IntType
                    defaultValue = 0
                }
            )
        ){entry ->
            MealDetailsScreen(
                onNavigateBack = onNavigateBack,
                onNavigateImage = onNavigateImage,
                canteenName = entry.arguments?.getString("canteenName")!!,
                date = LocalDate.parse(entry.arguments?.getString("date")!!),
                mealIds = entry.arguments?.getString("mealIds")!!.split(",").map { it.toInt() },
                initialPosition = entry.arguments?.getInt("position")!!
            )

        }
        composable(
            route = Screen.CanteenDetailsScreen.route + "/{canteenId}",
            arguments = listOf(
                navArgument("canteenId") {
                    type = NavType.IntType
                }
            )
        ) {entry ->
            CanteenDetailsScreen(
                onNavigateBack = onNavigateBack,
                canteenId = entry.arguments?.getInt("canteenId")!!
            )
        }
        composable(
            route = Screen.MapScreen.route
        ) {
            MapScreen(
                navController = navController,
                onNavigateMenu = onNavigateMenu
            )
        }
        composable(
            route = Screen.SettingsScreen.route
        ) {
            SettingsScreen(onNavigateBack, onNavigateLibraries)
        }
        composable(
            route = Screen.LibrariesScreen.route
        ) {
            Libraries(onNavigateBack)
        }
        composable(
            route = Screen.ImageScreen.route + "/{imageURL}",
            arguments = listOf(
                navArgument("imageURL") {
                    type = NavType.StringType
                }
            )
        ) { entry ->
            ImageScreen(
                onNavigateBack = onNavigateBack,
                imageURL = entry.arguments?.getString("imageURL") ?: ""
            )
        }
        composable(
            route = Screen.CardScreen.route
        ) {
            CardScreen (
                navController = navController,
                onNavigateSettings = onNavigateSettings
            )
        }
    }
}