package de.ciluvien.mensen.ui.screens

import android.content.Intent
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.PageSize
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.outlined.ArrowBack
import androidx.compose.material.icons.outlined.Share
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Scaffold
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.vectorResource
import androidx.lifecycle.viewmodel.compose.viewModel
import de.ciluvien.mensen.MensenApp
import de.ciluvien.mensen.R
import de.ciluvien.mensen.ui.components.MealDetails
import de.ciluvien.mensen.ui.components.core.CustomTextIcon
import de.ciluvien.mensen.ui.components.core.MultiActionTopAppBar
import de.ciluvien.mensen.ui.viewModelFactory
import de.ciluvien.mensen.ui.viewmodels.MealDetailsViewModel
import java.time.LocalDate

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun MealDetailsScreen(
    onNavigateBack: () -> Unit,
    onNavigateImage: (String) -> Unit,
    canteenName: String,
    date: LocalDate,
    mealIds: List<Int>,
    initialPosition: Int
) {
    val context = LocalContext.current
    val viewModel = viewModel<MealDetailsViewModel>(
        factory = viewModelFactory {
            MealDetailsViewModel(MensenApp.appModule)
        }
    )

    val pagerState = rememberPagerState(initialPage = initialPosition, pageCount = { mealIds.count() })
    LaunchedEffect(key1 = pagerState.currentPage) {
        viewModel.getMeals(mealIds, date)
        viewModel.getBookmarkState(mealIds[pagerState.currentPage], date)
    }


    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior()
    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        topBar = {
            val bookmarkIcon = if (viewModel.bookmarkState.value) {
                CustomTextIcon(
                    onClick = { viewModel.setBookmarkState(false, mealIds[pagerState.currentPage], date) },
                    icon = ImageVector.vectorResource(R.drawable.baseline_bookmark_24),
                    description = context.resources.getText(R.string.meal_remove_bookmark).toString()
                )
            } else {
                CustomTextIcon(
                    onClick = { viewModel.setBookmarkState(true, mealIds[pagerState.currentPage],date) },
                    icon = ImageVector.vectorResource(R.drawable.baseline_bookmark_border_24),
                    description = context.resources.getText(R.string.meal_set_bookmark).toString()
                )
            }

            val shareIcon = CustomTextIcon(
                onClick = {
                    val sendIntent: Intent = Intent().apply {
                        action = Intent.ACTION_SEND
                        putExtra(
                            Intent.EXTRA_TEXT,
                            "https://www.studentenwerk-dresden.de/mensen/speiseplan/details-${mealIds[pagerState.currentPage]}.html"
                        )
                        type = "text/plain"
                    }
                    val shareIntent = Intent.createChooser(sendIntent, null)
                    context.startActivity(shareIntent)
                },
                icon = Icons.Outlined.Share,
                description = context.getText(R.string.meal_share).toString()
            )

            val backIcon = CustomTextIcon(
                onNavigateBack,
                Icons.AutoMirrored.Outlined.ArrowBack,
                context.resources.getText(R.string.meal_back).toString()
            )

            MultiActionTopAppBar(
                canteenName,
                scrollBehavior,
                backIcon,
                listOf(shareIcon, bookmarkIcon)
            )
        }
    ) {padding ->
        Box(modifier = Modifier.padding(padding)) {
            HorizontalPager(
                state = pagerState,
                verticalAlignment = Alignment.Top,
                modifier = Modifier.fillMaxHeight()
            ) { position ->
                val menu = viewModel.meals.value
                if (menu.isNotEmpty() && position < menu.count()) {
                    MealDetails(
                        meal = menu[position],
                        scrollBehavior = scrollBehavior,
                        onNavigateImage = onNavigateImage
                    )
                }
            }
        }
    }
}