package de.ciluvien.mensen.data.mapper

import android.util.Log
import de.ciluvien.mensen.data.local.Canteen
import de.ciluvien.mensen.data.local.DailyMenu
import de.ciluvien.mensen.data.local.Meal
import de.ciluvien.mensen.data.parser.CanteenDetailParser
import de.ciluvien.mensen.data.parser.MenuDetailParser
import de.ciluvien.mensen.data.remote.CanteenResponse
import de.ciluvien.mensen.data.remote.DayResponse
import de.ciluvien.mensen.data.remote.MealResponse
import java.time.LocalDate

class Mapper {
    private val tag: String = "Mapper"
    suspend fun mapCanteen(canteenResponse: CanteenResponse): Canteen{
        val canteenParser = CanteenDetailParser(canteenResponse.url)
        canteenParser.setHTMLDoc()
        return Canteen(
            id = canteenResponse.id,
            name = canteenResponse.name,
            address = canteenResponse.address,
            coordinates = canteenResponse.coordinates,
            organisation = "Studentenwerk Dresden",
            detailURL = canteenResponse.url,
            imageURL = canteenParser.getCanteenLogoURL(),
            menuURL = canteenResponse.menu,
            menuRefreshDateTime = null,
            regularOpeningTime = canteenParser.getRegularOpeningTimes(),
            specialOpeningTime = canteenParser.getSpecialOpeningTimes()
        )
    }

    fun mapDailyMenu(canteenId: Int, dayResponse: DayResponse): DailyMenu {
        return DailyMenu(
            date = LocalDate.parse(dayResponse.date),
            closed = dayResponse.closed,
            canteenId = canteenId,
            mealRefreshDateTime = null
        )
    }

    fun mapMeal(canteenId: Int, date: LocalDate, mealResponse: MealResponse, menuParser: MenuDetailParser): Meal {
        val isSoldOut = menuParser.isMealSoldOut(mealResponse.id)
        val prices = try {
            menuParser.getAllPrices(mealResponse.id)
                ?: mealResponse.prices
                    .filterValues { !it.isNullOrBlank() }
                    .mapValues { entry -> entry.value!!.toBigDecimal() }
        } catch (e: NumberFormatException) {
            Log.e(tag, "Failed to convert api provided prices\n${e.message}")
            emptyMap()
        }

        var generalNotes = mealResponse.getGeneralNotes()
        if (menuParser.isClimatePlate(mealResponse.id) == true) {
            generalNotes = generalNotes.plus("Menü ist klimafreundlich")
        }

        return Meal(
            id = mealResponse.id,
            name = mealResponse.name,
            category = mealResponse.category,
            generalNotes = generalNotes,
            allergens = mealResponse.getAllergens(),
            additives = mealResponse.getAdditives(),
            prices = prices,
            imageULR = "https:${mealResponse.image}",
            menuDate = date,
            canteenId = canteenId,
            isSoldOut = isSoldOut,
            isVegan = mealResponse.isVegan(),
            isVegetarian = mealResponse.isVegetarian(),
            isPescetarian = mealResponse.isPescetarian(),
            isEveningOffer = mealResponse.isEveningOffer()
        )
    }
}