package de.ciluvien.mensen.data

import de.ciluvien.mensen.data.local.Canteen
import de.ciluvien.mensen.data.local.DailyMenu
import de.ciluvien.mensen.data.local.Meal
import de.ciluvien.mensen.data.local.Position
import de.ciluvien.mensen.data.mapper.Mapper
import de.ciluvien.mensen.data.parser.MenuDetailParser
import de.ciluvien.mensen.data.remote.OpenMensaService
import java.time.DayOfWeek
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.temporal.TemporalAdjusters

class DataService(private val dao: CanteenDao, private val api: OpenMensaService) {
    private val mapper = Mapper()

    suspend fun refreshCanteens(organisationName: String = "Studentenwerk Dresden") {
        val oldCanteens = dao.getOrganisationCanteens(organisationName)
        val newCanteens = mutableListOf<Canteen>()

        api.getCanteens().ifEmpty { return }.forEach {
            response -> newCanteens.add(mapper.mapCanteen(response))
        }

        // Selectively delete old canteens
        val newIds = newCanteens.map { it.id }
        oldCanteens.filter { !newIds.contains(it.id) }.forEach {
            dao.deleteCanteen(it)
        }

        // Delete the positions used by old canteens
        val positionList = dao.getPositionList()
        val deletedPositions = positionList.filter { !newIds.contains(it.canteenId) }
        deletedPositions.forEach {
            dao.deletePosition(it)
        }

        // Close the gaps in the list of positions
        val orderedPositionList = dao.getCanteensAndPositionsOrdered()
        orderedPositionList.forEachIndexed{ index, canteenAndPosition ->
            if (canteenAndPosition.position.position != index) {
                dao.insertPosition(Position(index, canteenAndPosition.canteen.id))
            }
        }

        // Assign a position to each new canteen
        var positionCount = dao.getPositionCount()
        newCanteens.forEach {
            dao.insertCanteen(it)
            if (dao.getCanteenAndPosition(it.id) == null) {
                dao.insertPosition(Position(positionCount, it.id))
                positionCount += 1
            }
        }

        val refreshedOrganisation = dao.getOrganisation(organisationName)
        if (refreshedOrganisation != null) {
            refreshedOrganisation.canteenRefreshDateTime = LocalDateTime.now()
            dao.insertOrganisation(refreshedOrganisation)
        }
    }

    suspend fun refreshDailyMenus(canteenId: Int) {
        val days = api.getDays(canteenId)
        val oldMenus = dao.getCanteenDailyMenus(canteenId)
        val newMenus = mutableListOf<DailyMenu>()

        days.forEach{
            response -> newMenus.add(mapper.mapDailyMenu(canteenId, response))
        }

        if (newMenus.isEmpty()) return

        // Remove old menus and their meals
        val newDates = newMenus.map { it.date }
        oldMenus.filter { !newDates.contains(it.date) }.forEach {
            dao.getDailyMenuWithMeals(it.canteenId, it.date).forEach {meal ->
                dao.deleteMeal(meal)
            }
            dao.deleteDailyMenu(it)
        }

        newMenus.forEach {
            dao.insertDailyMenu(it)
        }

        val refreshedCanteen = dao.getCanteen(canteenId)
        if (refreshedCanteen != null) {
            refreshedCanteen.menuRefreshDateTime = LocalDateTime.now()
            dao.insertCanteen(refreshedCanteen)
        }
    }

    private fun LocalDate.isInCurrentWeek(): Boolean {
        val currentDate = LocalDate.now()
        val startOfWeek = currentDate.with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY))
        val endOfWeek = startOfWeek.plusDays(6)
        return this.isAfter(startOfWeek) && this.isBefore(endOfWeek) || this.isEqual(startOfWeek) || this.isEqual(endOfWeek)
    }

    suspend fun refreshMeals(canteenId: Int, date: LocalDate) {
        val canteen = dao.getCanteen(canteenId) ?: return
        val menuParser = MenuDetailParser(canteen.menuURL)
        val oldMeals = dao.getDailyMenuWithMeals(canteenId, date)
        val newMeals = mutableListOf<Meal>()
        val bookmarks = dao.getBookmarkList().filter { it.date == date }.map { b -> b.mealId }
        val today = LocalDate.now()

        if (date.isInCurrentWeek()) {
            menuParser.setHTMLDoc()
        }
        api.getMeals(canteenId, date).forEach{
                response -> newMeals.add(mapper.mapMeal(canteenId, date, response, menuParser))
        }

        if (newMeals.isNotEmpty()) {
            val newIds = newMeals.map { it.id }
            oldMeals
                .filter { !newIds.contains(it.id) }
                .forEach {
                    if (!(bookmarks.contains(it.id) && date >= today)) {
                        dao.deleteMeal(it)
                    }
                }

            newMeals.forEach {
                dao.insertMeal(it)
            }
        }

        val refreshedMenu = dao.getDailyMenu(canteenId, date)
            ?: return
        refreshedMenu.mealRefreshDateTime = LocalDateTime.now()
        dao.insertDailyMenu(refreshedMenu)
    }

    suspend fun refreshDailyMenu(canteenId: Int, date: LocalDate) {
        val dayResponse = api.getDay(canteenId, date)
        if(dayResponse == null) {
            dao.deleteDailyMenuByCanteenAndDate(canteenId, date)
            return
        }

        dao.insertDailyMenu(mapper.mapDailyMenu(canteenId, dayResponse))
    }

    suspend fun refreshMeal(canteenId: Int, date: LocalDate, mealId: Int) {
        val canteen = dao.getCanteen(canteenId) ?: return
        val menuParser = MenuDetailParser(canteen.menuURL)
        menuParser.setHTMLDoc()
        val response = api.getMeal(canteenId, date, mealId)
        if(response == null){
            dao.deleteMealById(mealId)
            return
        }

        dao.insertMeal(mapper.mapMeal(canteenId, date, response, menuParser))
    }
}