package de.ciluvien.mensen.data

import androidx.room.Dao
import androidx.room.Delete
import androidx.room.Insert
import androidx.room.OnConflictStrategy
import androidx.room.Query
import androidx.room.RewriteQueriesToDropUnusedColumns
import androidx.room.Transaction
import androidx.room.Upsert
import de.ciluvien.mensen.data.local.Bookmark
import de.ciluvien.mensen.data.local.Canteen
import de.ciluvien.mensen.data.local.CardBalance
import de.ciluvien.mensen.data.local.Position
import de.ciluvien.mensen.data.local.DailyMenu
import de.ciluvien.mensen.data.local.Organisation
import de.ciluvien.mensen.data.local.Meal
import de.ciluvien.mensen.data.local.relations.CanteenAndPosition
import java.time.LocalDate

@Dao
interface CanteenDao {
    @Upsert
    suspend fun insertOrganisation(organisation: Organisation)

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun insertPosition(position: Position)

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    @Transaction
    suspend fun upsertPositionList(positionList: List<Position>)

    @Upsert
    suspend fun insertCanteen(canteen: Canteen)

    @Upsert
    suspend fun insertDailyMenu(dailyMenu: DailyMenu)

    @Upsert
    suspend fun insertMeal(meal: Meal)

    @Insert
    suspend fun insertCardBalance(balance: CardBalance)

    @Upsert
    suspend fun insertBookmark(bookmark: Bookmark)

    @Delete
    suspend fun deleteOrganisation(organisation: Organisation)

    @Delete
    suspend fun deleteCanteen(canteen: Canteen)

    @Delete
    suspend fun deletePosition(position: Position)

    @Delete
    suspend fun deleteDailyMenu(dailyMenu: DailyMenu)

    @Delete
    suspend fun deleteMeal(meal: Meal)

    @Delete
    suspend fun deleteCardBalance(cardBalance: CardBalance)

    @Delete
    suspend fun deleteBookmark(bookmark: Bookmark)

    @Query("DELETE FROM dailymenu WHERE date = :date AND canteenId = :canteenId")
    suspend fun deleteDailyMenuByCanteenAndDate(canteenId: Int, date: LocalDate)

    @Query("DELETE FROM meal WHERE id = :mealId")
    suspend fun deleteMealById(mealId: Int)

    @Query("SELECT * FROM organisation WHERE name=:organisation")
    suspend fun getOrganisation(organisation: String): Organisation?

    @Transaction
    @Query("SELECT * FROM organisation, canteen WHERE organisation.name = :organisation AND canteen.organisation = organisation.name")
    suspend fun getOrganisationCanteens(organisation: String): List<Canteen>

    @Transaction
    @Query("SELECT dailymenu.* FROM canteen, dailymenu WHERE canteen.id = :canteenId AND canteen.id = dailymenu.canteenId")
    suspend fun getCanteenDailyMenus(canteenId: Int): List<DailyMenu>

    @Transaction
    @Query("SELECT meal.* " +
            "FROM dailymenu " +
            "JOIN  meal ON dailymenu.date = meal.menuDate AND dailymenu.date = :date" +
            " AND dailymenu.canteenId = meal.canteenId  AND dailymenu.canteenId = :canteenId"
    )
    suspend fun getDailyMenuWithMeals(canteenId: Int, date: LocalDate): List<Meal>

    @Transaction
    @RewriteQueriesToDropUnusedColumns
    @Query("SELECT * FROM canteen JOIN position ON canteen.id = position.canteenId ORDER BY position.position ASC")
    suspend fun getCanteensAndPositionsOrdered(): List<CanteenAndPosition>

    @Transaction
    @Query("SELECT * FROM position WHERE canteenId = :canteenId")
    suspend fun getCanteenAndPosition(canteenId: Int): Position?

    @Transaction
    @Query("SELECT * FROM position WHERE position = :position")
    suspend fun getPositionAndCanteen(position: Int): Position?

    @Transaction
    @Query("SELECT * FROM canteen WHERE id = :id")
    suspend fun getCanteen(id: Int): Canteen?

    @Query("SELECT * FROM meal WHERE id = :mealId AND menuDate = :date")
    suspend fun getMeal(mealId: Int, date: LocalDate): Meal?

    @Query("SELECT * FROM dailymenu WHERE date = :date AND canteenId = :canteenId")
    suspend fun getDailyMenu(canteenId: Int, date: LocalDate): DailyMenu?

    @Transaction
    @Query("SELECT COUNT(*) FROM position")
    suspend fun getPositionCount(): Int

    @Query("SELECT * FROM position")
    suspend fun getPositionList(): List<Position>

    @Query("SELECT * FROM CardBalance ORDER BY dateTime DESC")
    suspend fun getCardBalanceList(): List<CardBalance>

    @Query("SELECT * FROM Bookmark WHERE mealId = :mealId AND date = :date")
    suspend fun getBookmark(mealId: Int, date: LocalDate): Bookmark?

    @Query("SELECT * FROM Bookmark")
    suspend fun getBookmarkList(): List<Bookmark>
}