package de.ciluvien.mensen

import android.app.ActivityOptions
import android.app.PendingIntent
import android.content.Intent
import android.content.IntentFilter
import android.nfc.NfcAdapter
import android.nfc.Tag
import android.nfc.tech.IsoDep
import android.os.Build
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.ui.Modifier
import androidx.core.content.IntentCompat
import androidx.navigation.NavHostController
import androidx.navigation.compose.rememberNavController
import de.ciluvien.mensen.data.nfc.INTENT_ACTION_NFC_READ
import de.ciluvien.mensen.ui.Navigation
import de.ciluvien.mensen.ui.Screen
import de.ciluvien.mensen.ui.theme.MensenTheme
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

class MainActivity : ComponentActivity() {
    private var nfcAdapter : NfcAdapter? = null
    private var navHostController: NavHostController? = null

    private var pendingIntent: PendingIntent? = null
    private var nfcIntentFilter: Array<IntentFilter>? = null
    private var techList: Array<Array<String>>? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val launchedByNFC = intent.action == NfcAdapter.ACTION_TAG_DISCOVERED ||
                intent.action == NfcAdapter.ACTION_TECH_DISCOVERED

        nfcAdapter = NfcAdapter.getDefaultAdapter(this)

        // Create new PendingIntent that can launch the app for foreground dispatch
        val intentStartApp = Intent(this, javaClass).apply { addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP) }
        val options = ActivityOptions.makeBasic()

        if (Build.VERSION.SDK_INT == Build.VERSION_CODES.UPSIDE_DOWN_CAKE
            || Build.VERSION.SDK_INT == Build.VERSION_CODES.VANILLA_ICE_CREAM) {
            // API 34-35 require allowing PendingIntents to start background activities
            options.pendingIntentCreatorBackgroundActivityStartMode = ActivityOptions.MODE_BACKGROUND_ACTIVITY_START_ALLOWED
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.BAKLAVA) {
            // API 36 requires allowing PendingIntents to start background activities as well, but different
            options.pendingIntentCreatorBackgroundActivityStartMode = ActivityOptions.MODE_BACKGROUND_ACTIVITY_START_ALLOW_ALWAYS
        }
        pendingIntent = PendingIntent.getActivity(
            this,
            0,
            intentStartApp,
            PendingIntent.FLAG_MUTABLE,
            options.toBundle()
        )

        nfcIntentFilter = arrayOf(
            // Alternatively filter for ACTION_TECH_DISCOVERED
            IntentFilter(NfcAdapter.ACTION_TAG_DISCOVERED)
        )

        // techList should not be necessary when filtering for TAG_DISCOVERED and not TECH_DISCOVERED
        techList = arrayOf(arrayOf(IsoDep::class.java.name))

        enableEdgeToEdge()
        setContent {
            LaunchedEffect(key1 = navHostController) {
                if(launchedByNFC) {
                    useNFCIntent(intent)
                }
            }
            MensenTheme {
                navHostController = rememberNavController()
                Surface(
                    modifier = Modifier.fillMaxSize(),
                    color = MaterialTheme.colorScheme.background
                ) {
                    navHostController?.let {
                        if(launchedByNFC) {
                            Log.i("nfc","Launched app using nfc")
                            Navigation(navController = it, startDestination = Screen.CardScreen.route)
                        } else {
                            Navigation(it)
                        }
                    }
                }
            }
        }
    }

    override fun onResume() {
        super.onResume()
        nfcAdapter?.enableForegroundDispatch(this, pendingIntent, nfcIntentFilter, techList)
    }

    override fun onPause() {
        super.onPause()
        nfcAdapter?.disableForegroundDispatch(this)
    }

    override fun onNewIntent(intent: Intent) {
        super.onNewIntent(intent)

        val launchedByNFC = intent.action == NfcAdapter.ACTION_TAG_DISCOVERED
                || intent.action == NfcAdapter.ACTION_TECH_DISCOVERED
        if (launchedByNFC) {
            navHostController?.navigate(Screen.CardScreen.route) {
                launchSingleTop = true
                restoreState = true
            }
            useNFCIntent(intent)
        }
    }

    private fun useNFCIntent(intent: Intent) {
        CoroutineScope(Dispatchers.Main).launch {
            delay(250)
            sendBroadcast(Intent(INTENT_ACTION_NFC_READ).apply {
                Log.i("nfc", "Sending Broadcast")
                putExtra(
                    NfcAdapter.EXTRA_TAG,
                    IntentCompat.getParcelableExtra(intent, NfcAdapter.EXTRA_TAG, Tag::class.java)
                )
            })
        }
    }
}
