/*
 * Copyright by the original author or authors.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package de.ccc.events.badge.card10.time

import android.bluetooth.BluetoothGattCharacteristic
import android.bluetooth.BluetoothGattService
import android.util.Log
import de.ccc.events.badge.card10.*
import de.ccc.events.badge.card10.common.ConnectionService
import java.nio.ByteBuffer

private const val TAG = "Card10Service"

class Card10Service(
    service: BluetoothGattService
) {
    private val timeCharacteristic = service.getCharacteristic(TIME_CHARACTERISTIC_UUID)
    private val rocketsCharacteristic = service.getCharacteristic(ROCKETS_CHARACTERISTIC_UUID)
    private val ledsAboveCharacteristic = service.getCharacteristic(LEDS_ABOVE_CHARACTERISTIC_UUID)
    private val ledBLCharacteristic = service.getCharacteristic(BG_LED_BOTTOM_LEFT_CHARACTERISTIC_UUID)
    private val ledBRCharacteristic = service.getCharacteristic(BG_LED_BOTTOM_RIGHT_CHARACTERISTIC_UUID)
    private val ledTRCharacteristic = service.getCharacteristic(BG_LED_TOP_RIGHT_CHARACTERISTIC_UUID)
    private val ledTLCharacteristic = service.getCharacteristic(BG_LED_TOP_LEFT_CHARACTERISTIC_UUID)
    private val personalStateCharacteristic = service.getCharacteristic(PERSONAL_STATE_CHARACTERISTIC_UUID)


    init {
        timeCharacteristic.writeType = BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
        rocketsCharacteristic.writeType = BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
        ledsAboveCharacteristic.writeType = BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
        ledBLCharacteristic.writeType = BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
        ledBRCharacteristic.writeType = BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
        ledTRCharacteristic.writeType = BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
        ledTLCharacteristic.writeType = BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
        personalStateCharacteristic.writeType = BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE
    }

    fun setTime() {
        val buffer = ByteBuffer.allocate(8)
        buffer.putLong(System.currentTimeMillis())
        timeCharacteristic.value = buffer.array()
        ConnectionService.writeCharacteristic(timeCharacteristic)
    }

    fun setRocketValue(value: ByteArray) {
        rocketsCharacteristic.value = value
        ConnectionService.writeCharacteristic(rocketsCharacteristic)
    }

    fun setLeds(value: ByteArray) {
        ledsAboveCharacteristic.value = value
        ConnectionService.writeCharacteristic(ledsAboveCharacteristic)
    }

    fun setBLLed(value: ByteArray) {
        ledBLCharacteristic.value = value
        ConnectionService.writeCharacteristic(ledBLCharacteristic)
    }

    fun setBRLed(value: ByteArray) {
        ledBRCharacteristic.value = value
        ConnectionService.writeCharacteristic(ledBRCharacteristic)
    }

    fun setTRLed(value: ByteArray) {
        ledTRCharacteristic.value = value
        ConnectionService.writeCharacteristic(ledTRCharacteristic)
    }

    fun setTLLed(value: ByteArray) {
        ledTLCharacteristic.value = value
        ConnectionService.writeCharacteristic(ledTLCharacteristic)
    }

    fun setPersonalState(value: PersonalState) {
        personalStateCharacteristic.value = ByteArray(2)
        personalStateCharacteristic.value[1] = value.v.toByte()
        personalStateCharacteristic.value[0] = (value.v shr 8).toByte()
        Log.d(TAG,"Writing personal State array : ${personalStateCharacteristic.value[0]},${personalStateCharacteristic.value[1]} ")
        ConnectionService.writeCharacteristic(personalStateCharacteristic)
    }
}