/*
 *    Copyright 2024, Petr Laštovička as Lasta apps, All rights reserved
 *
 *     This file is part of Menza.
 *
 *     Menza is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Menza is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Menza.  If not, see <https://www.gnu.org/licenses/>.
 */

@file:Suppress("SpellCheckingInspection")

package cz.lastaapps.menza.ui.components

import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLayoutResult
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.style.TextDecoration
import kotlinx.collections.immutable.ImmutableList
import kotlinx.collections.immutable.mutate
import kotlinx.collections.immutable.persistentListOf
import java.util.regex.Pattern

// https://stackoverflow.com/questions/66130513/linkify-with-compose-text
@Composable
fun LinkifyText(
    text: String,
    modifier: Modifier = Modifier,
    linkColor: Color = LocalContentColor.current,
) {
    val uriHandler = LocalUriHandler.current
    val layoutResult =
        remember {
            mutableStateOf<TextLayoutResult?>(null)
        }
    val linksList = remember(text) { extractUrls(text) }
    val annotatedString =
        buildAnnotatedString {
            append(text)
            linksList.forEach {
                addStyle(
                    style =
                        SpanStyle(
                            color = linkColor,
                            textDecoration = TextDecoration.Underline,
                        ),
                    start = it.start,
                    end = it.end,
                )
                addStringAnnotation(
                    tag = "URL",
                    annotation = it.url,
                    start = it.start,
                    end = it.end,
                )
            }
        }
    Text(
        text = annotatedString,
        modifier =
            modifier.pointerInput(Unit) {
                detectTapGestures { offsetPosition ->
                    layoutResult.value?.let {
                        val position = it.getOffsetForPosition(offsetPosition)
                        annotatedString
                            .getStringAnnotations(position, position)
                            .firstOrNull()
                            ?.let { result ->
                                if (result.tag == "URL") {
                                    uriHandler.openUri(result.item)
                                }
                            }
                    }
                }
            },
        onTextLayout = { layoutResult.value = it },
    )
}

private val urlPattern: Pattern =
    Pattern.compile(
        "(?:^|[\\W])((ht|f)tp(s?)://|www\\.)" +
            "(([\\w\\-]+\\.)+?([\\w\\-.~]+/?)*" +
            "[\\p{Alnum}.,%_=?&#\\-+()\\[\\]*$~@!:/{};']*)",
        Pattern.CASE_INSENSITIVE or Pattern.MULTILINE or Pattern.DOTALL,
    )

private fun extractUrls(text: String): ImmutableList<LinkInfo> =
    persistentListOf<LinkInfo>().mutate { links ->
        val matcher = urlPattern.matcher(text)
        var matchStart: Int
        var matchEnd: Int

        while (matcher.find()) {
            matchStart = matcher.start(1)
            matchEnd = matcher.end()

            var url = text.substring(matchStart, matchEnd)
            if (!url.startsWith("http://") && !url.startsWith("https://")) {
                url = "https://$url"
            }

            links.add(LinkInfo(url, matchStart, matchEnd))
        }
    }

private data class LinkInfo(
    val url: String,
    val start: Int,
    val end: Int,
)
