/*
 *    Copyright 2025, Petr Laštovička as Lasta apps, All rights reserved
 *
 *     This file is part of Menza.
 *
 *     Menza is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Menza is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Menza.  If not, see <https://www.gnu.org/licenses/>.
 */

package cz.lastaapps.menza.features.settings.ui.screens

import android.content.Context
import android.content.Intent
import androidx.annotation.StringRes
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ColumnScope
import androidx.compose.foundation.layout.ExperimentalLayoutApi
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ReceiptLong
import androidx.compose.material.icons.filled.BugReport
import androidx.compose.material.icons.filled.Refresh
import androidx.compose.material.icons.filled.Security
import androidx.compose.material.icons.filled.Share
import androidx.compose.material.icons.filled.Star
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.platform.UriHandler
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import cz.lastaapps.api.core.domain.model.Menza
import cz.lastaapps.menza.R
import cz.lastaapps.menza.features.other.ui.dialog.ReportDialog
import cz.lastaapps.menza.features.other.ui.dialog.sendReport
import cz.lastaapps.menza.features.settings.domain.model.AppSettings
import cz.lastaapps.menza.features.settings.domain.model.AppThemeType
import cz.lastaapps.menza.features.settings.domain.model.AppThemeType.Agata
import cz.lastaapps.menza.features.settings.domain.model.Currency
import cz.lastaapps.menza.features.settings.domain.model.InitialSelectionBehaviour
import cz.lastaapps.menza.features.settings.domain.model.PriceType
import cz.lastaapps.menza.features.settings.ui.util.name
import cz.lastaapps.menza.features.settings.ui.widget.BalanceThresholdSlider
import cz.lastaapps.menza.features.settings.ui.widget.FullReloadDialog
import cz.lastaapps.menza.features.settings.ui.widget.SettingsItem
import cz.lastaapps.menza.features.settings.ui.widget.SettingsSwitch
import cz.lastaapps.menza.ui.components.ChooseFromDialog
import cz.lastaapps.menza.ui.theme.Padding
import cz.lastaapps.menza.ui.util.PreviewWrapper
import kotlinx.collections.immutable.ImmutableList
import kotlinx.collections.immutable.persistentListOf
import kotlinx.collections.immutable.toImmutableList

@Composable
@Suppress("ktlint:compose:parameter-naming")
internal fun SettingsScreen(
    appSettings: AppSettings,
    appTheme: AppThemeType,
    preferredMenza: Menza?,
    onChooseTheme: () -> Unit,
    onChooseDishLanguage: () -> Unit,
    onDiscounterPrices: (PriceType) -> Unit,
    onCurrency: (Currency) -> Unit,
    onImagesOnMetered: (Boolean) -> Unit,
    onAlternativeNavigation: (Boolean) -> Unit,
    onBalanceThreshold: (Int) -> Unit,
    onInitialMenzaBehaviour: (InitialSelectionBehaviour) -> Unit,
    menzaList: ImmutableList<Menza>,
    onSelectedMenza: (Menza) -> Unit,
    showAbout: Boolean,
    onAboutClick: () -> Unit,
    onPrivacyPolicy: () -> Unit,
    onFullRefresh: () -> Unit,
    onCrashesDialog: () -> Unit,
    modifier: Modifier = Modifier,
) {
    Column(
        modifier = modifier.verticalScroll(rememberScrollState()),
        verticalArrangement = Arrangement.spacedBy(Padding.None),
    ) {
//        Text(
//            text = stringResource(id = R.string.settings_title),
//            style = MaterialTheme.typography.displaySmall,
//            modifier = Modifier.padding(horizontal = SettingsTokens.itemPadding),
//        )

        // App theme
        SettingsItem(
            title = stringResource(id = R.string.settings_theme_title),
            subtitle =
                buildString {
                    append(appTheme.name())
                    append(", ")
                    append(appSettings.darkMode.name())
                },
            onClick = onChooseTheme,
        )

        // Discounted prices
        SettingsSwitch(
            title = stringResource(id = R.string.settings_switch_price),
            subtitle = appSettings.priceType.name(),
            isChecked = appSettings.priceType == PriceType.Discounted,
            onCheck = { onDiscounterPrices(appSettings.priceType.other()) },
        )

        // Dish language
        SettingsItem(
            title = stringResource(id = R.string.settings_language_title),
            subtitle = stringResource(id = R.string.settings_language_subtitle),
            onClick = onChooseDishLanguage,
        )

        // Currency
        CurrencySelector(
            currentCurrency = appSettings.currency,
            onCurrency = onCurrency,
        )

        // Metered networks
        SettingsSwitch(
            title = stringResource(id = R.string.settings_switch_metered_title),
            subtitle = stringResource(id = R.string.settings_switch_metered_subtitle),
            isChecked = appSettings.imagesOnMetered,
            onCheck = onImagesOnMetered,
        )

        // Alternative navigation
        SettingsSwitch(
            title = stringResource(id = R.string.settings_alternative_navigation_title),
            subtitle = stringResource(id = R.string.settings_alternative_navigation_subtitle),
            isChecked = appSettings.alternativeNavigation,
            onCheck = onAlternativeNavigation,
        )

        // Balance warning threshold
        BalanceThresholdSlider(
            title = stringResource(id = R.string.settings_balance_threshold_title),
            threshold = appSettings.balanceWarningThreshold,
            onThreshold = onBalanceThreshold,
        )

        // Behaviour at startup
        InitialBehaviourSelector(
            initialMenzaBehaviour = appSettings.initialMenzaMode,
            onInitialMenzaBehaviour = onInitialMenzaBehaviour,
            menzaList = menzaList,
            selectedMenza = preferredMenza,
            onSelectedMenza = onSelectedMenza,
        )

        // About
        if (showAbout) {
            SettingsItem(
                title = stringResource(id = R.string.settings_about_title),
                subtitle = stringResource(id = R.string.settings_about_subtitle),
                onClick = onAboutClick,
            )
        }

        Spacer(Modifier.height(Padding.Medium))

        Buttons(
            onPrivacyPolicy = onPrivacyPolicy,
            onFullRefresh = onFullRefresh,
            onCrashesDialog = onCrashesDialog,
            modifier = Modifier.fillMaxWidth(),
        )
    }
}

@Composable
@Suppress("UnusedReceiverParameter")
private fun ColumnScope.InitialBehaviourSelector(
    initialMenzaBehaviour: InitialSelectionBehaviour,
    onInitialMenzaBehaviour: (InitialSelectionBehaviour) -> Unit,
    menzaList: ImmutableList<Menza>,
    selectedMenza: Menza?,
    onSelectedMenza: (Menza) -> Unit,
) {
    var initDialogVisible by remember { mutableStateOf(false) }

    // Behaviour
    SettingsItem(
        title = stringResource(id = R.string.settings_init_menza_title),
        subtitle = initialMenzaBehaviour.name(),
        onClick = { initDialogVisible = true },
    )

    if (initDialogVisible) {
        val items =
            InitialSelectionBehaviour.entries
                .map { it to it.name() }
                .toImmutableList()

        ChooseFromDialog(
            title = stringResource(id = R.string.settings_init_menza_title),
            items = items,
            onItemSelect = { onInitialMenzaBehaviour(it.first) },
            onDismiss = { initDialogVisible = false },
            toString = Pair<InitialSelectionBehaviour, String>::second,
        )
    }

    // Specific menza
    var selectDialogVisible by remember { mutableStateOf(false) }

    if (initialMenzaBehaviour == InitialSelectionBehaviour.Specific) {
        SettingsItem(
            title = stringResource(id = R.string.settings_init_menza_select_title),
            subtitle =
                selectedMenza?.name
                    ?: stringResource(id = R.string.settings_init_menza_select_placeholder),
            onClick = { selectDialogVisible = true },
        )
    }

    if (selectDialogVisible) {
        ChooseFromDialog(
            title = stringResource(id = R.string.settings_init_menza_title),
            items = menzaList,
            onItemSelect = { onSelectedMenza(it) },
            onDismiss = { selectDialogVisible = false },
            toString = Menza::name,
        )
    }
}

@Composable
@Suppress("UnusedReceiverParameter")
private fun ColumnScope.CurrencySelector(
    currentCurrency: Currency,
    onCurrency: (Currency) -> Unit,
) {
    var dialogVisible by remember { mutableStateOf(false) }

    // Behaviour
    SettingsItem(
        title = stringResource(id = R.string.settings_currency_title),
        subtitle =
            currentCurrency
                .name()
                .let { name ->
                    currentCurrency.disclaimer()?.let { "$name – $it" } ?: name
                },
        onClick = { dialogVisible = true },
    )

    if (dialogVisible) {
        val items =
            Currency.entries
                .drop(1)
                .map { it to it.name() }
                .toImmutableList()

        ChooseFromDialog(
            title = stringResource(id = R.string.settings_currency_dialog_title),
            items = items,
            onItemSelect = { onCurrency(it.first) },
            onDismiss = { dialogVisible = false },
            toString = Pair<Currency, String>::second,
        )
    }
}

@Composable
private fun Currency.name() =
    when (this) {
        Currency.NONE -> ""
        Currency.CZK -> stringResource(id = R.string.settings_currency_name_czech)
        Currency.BEER -> stringResource(id = R.string.settings_currency_name_beer)
        Currency.EUR -> stringResource(id = R.string.settings_currency_name_eur)
        Currency.USD -> stringResource(id = R.string.settings_currency_name_usd)
    }

@Composable
private fun Currency.disclaimer() =
    when (this) {
        Currency.NONE,
        Currency.CZK,
        -> null

        Currency.BEER,
        -> stringResource(id = R.string.settings_currency_disclaimer_desitka)

        Currency.EUR,
        Currency.USD,
        -> stringResource(id = R.string.settings_currency_disclaimer_conversion)
    }

@OptIn(ExperimentalLayoutApi::class)
@Composable
private fun Buttons(
    onPrivacyPolicy: () -> Unit,
    onFullRefresh: () -> Unit,
    onCrashesDialog: () -> Unit,
    modifier: Modifier = Modifier,
    context: Context = LocalContext.current,
    uriHandler: UriHandler = LocalUriHandler.current,
) {
    FlowRow(
        modifier = modifier,
        horizontalArrangement =
            Arrangement.spacedBy(
                Padding.Small,
                Alignment.CenterHorizontally,
            ),
        verticalArrangement = Arrangement.Center,
    ) {
        val shareText = stringResource(R.string.settings_button_share_text)
        OutlinedButton(
            onClick = {
                val sendIntent: Intent =
                    Intent().apply {
                        action = Intent.ACTION_SEND
                        putExtra(Intent.EXTRA_TEXT, shareText)
                        type = "text/plain"
                    }
                context.startActivity(Intent.createChooser(sendIntent, null))
            },
        ) { IconAndText(Icons.Default.Share, R.string.settings_button_share) }

        OutlinedButton(
            onClick = { uriHandler.openUri("https://play.google.com/store/apps/details?id=cz.lastaapps.menza") },
        ) { IconAndText(Icons.Default.Star, R.string.settings_button_rate) }

        OutlinedButton(
            onClick = onPrivacyPolicy,
        ) { IconAndText(Icons.Default.Security, R.string.settings_button_privacy_policy) }

        ReportButton()

        CrashesButton(
            onCrashesDialog = onCrashesDialog,
        )

        @Suppress("ConstantConditionIf")
        if (false) {
            FullDataReload(
                onFullRefresh = onFullRefresh,
            )
        }
    }
}

@Composable
private fun FullDataReload(
    onFullRefresh: () -> Unit,
    modifier: Modifier = Modifier,
) {
    var showFullReload by rememberSaveable { mutableStateOf(false) }
    if (showFullReload) {
        FullReloadDialog(
            onDismissRequest = { showFullReload = false },
            onConfirm = onFullRefresh,
        )
    }

    OutlinedButton(onClick = { showFullReload = true }, modifier) {
        IconAndText(Icons.Default.Refresh, R.string.settings_button_reload)
    }
}

@Composable
private fun ReportButton(modifier: Modifier = Modifier) {
    var shown by rememberSaveable { mutableStateOf(false) }

    OutlinedButton(onClick = { shown = true }, modifier) {
        IconAndText(Icons.Default.BugReport, R.string.settings_button_report)
    }

    val context = LocalContext.current
    ReportDialog(shown, false, { shown = false }) {
        sendReport(context, it, "User feedback", null, null)
        shown = false
    }
}

@Composable
fun CrashesButton(
    onCrashesDialog: () -> Unit,
    modifier: Modifier = Modifier,
) {
    OutlinedButton(onClick = onCrashesDialog, modifier) {
        IconAndText(
            Icons.AutoMirrored.Default.ReceiptLong,
            stringResource(R.string.settings_button_crashes),
        )
    }
}

@Composable
private fun IconAndText(
    icon: ImageVector,
    @StringRes textId: Int,
) = IconAndText(icon, stringResource(textId))

@Composable
private fun IconAndText(
    icon: ImageVector,
    text: String,
) {
    Row(
        horizontalArrangement = Arrangement.spacedBy(4.dp, Alignment.CenterHorizontally),
        verticalAlignment = Alignment.CenterVertically,
    ) {
        Icon(icon, null)
        Text(text, textAlign = TextAlign.Center)
    }
}

@Preview
@Composable
private fun SettingsScreenPreview() =
    PreviewWrapper {
        SettingsScreen(
            appSettings = AppSettings.default,
            appTheme = Agata,
            onChooseTheme = {},
            onChooseDishLanguage = {},
            onDiscounterPrices = {},
            onCurrency = {},
            onImagesOnMetered = {},
            onAlternativeNavigation = {},
            onBalanceThreshold = {},
            onInitialMenzaBehaviour = {},
            menzaList = persistentListOf(),
            preferredMenza = null,
            onSelectedMenza = {},
            showAbout = true,
            onAboutClick = {},
            onFullRefresh = {},
            onPrivacyPolicy = {},
            onCrashesDialog = {},
        )
    }
