package com.ztftrue.music.utils

import android.appwidget.AppWidgetManager
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.pm.ResolveInfo
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.media.MediaMetadataRetriever
import android.net.Uri
import android.os.Bundle
import android.provider.OpenableColumns
import android.text.TextUtils
import android.util.Log
import android.util.TypedValue
import android.widget.Toast
import androidx.annotation.OptIn
import androidx.compose.ui.unit.Dp
import androidx.core.content.ContextCompat
import androidx.core.content.FileProvider
import androidx.core.graphics.scale
import androidx.core.net.toUri
import androidx.lifecycle.viewModelScope
import androidx.media3.common.C
import androidx.media3.common.MediaItem
import androidx.media3.common.util.UnstableApi
import androidx.media3.session.LibraryResult
import androidx.media3.session.SessionResult
import com.google.common.collect.ImmutableList
import com.google.common.util.concurrent.ListenableFuture
import com.google.common.util.concurrent.MoreExecutors
import com.ztftrue.music.BuildConfig
import com.ztftrue.music.ImageSource
import com.ztftrue.music.MainActivity
import com.ztftrue.music.MusicViewModel
import com.ztftrue.music.R
import com.ztftrue.music.play.manager.MediaCommands
import com.ztftrue.music.play.MediaItemUtils
import com.ztftrue.music.sqlData.model.DictionaryApp
import com.ztftrue.music.sqlData.model.MusicItem
import com.ztftrue.music.ui.play.Lyrics
import com.ztftrue.music.utils.model.AnyListBase
import com.ztftrue.music.utils.trackManager.PlaylistManager
import com.ztftrue.music.utils.trackManager.SongsUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException
import java.util.Locale

enum class OperateTypeInActivity {
    DeletePlayList,
    InsertTrackToPlaylist,
    ModifyTrackFromPlayList,//sort and remove tracks
    RenamePlaylist,
    RemoveTrackFromStorage,
    EditTrackInfo,
}

enum class PlayListType {
    Songs,
    PlayLists,
    Queue,
    Albums,
    Artists,
    Genres,
    Folders,
    None
}

fun enumToStringForPlayListType(myEnum: PlayListType): String {
    return myEnum.name
}

fun stringToEnumForPlayListType(enumString: String): PlayListType {
    return try {
        PlayListType.valueOf(enumString)
    } catch (e: IllegalArgumentException) {
        Log.e("Utils", "stringToEnumForPlayListType: $enumString", e)
        PlayListType.Songs
    }
}


enum class OperateType {
    AddToQueue,
    RemoveDuplicate,
    RemoveFromQueue,
    PlayNext,
    AddToPlaylist,
    RemoveFromPlaylist,
    AddToFavorite,
    Artist,
    Album,
    EditMusicInfo,
    DeleteFromStorage,
    No,
    RenamePlayList,
    DeletePlayList,
    ShowArtist,
    ClearQueue,
    SaveQueueToPlayList,
    IgnoreFolder
}

enum class ScrollDirectionType {
    GRID_HORIZONTAL,
    GRID_VERTICAL,
    LIST_VERTICAL,
}

data class CheckLyricsData(val path: String, val type: LyricsType)

@Suppress("deprecation")
object Utils {
    val translateMap = HashMap<String, Int>().apply {
        put("Songs", R.string.tab_songs)
        put("PlayLists", R.string.tab_playLists)
        put("Queue", R.string.tab_queue)
        put("Albums", R.string.tab_albums)
        put("Artists", R.string.tab_artists)
        put("Genres", R.string.tab_genres)
        put("Folders", R.string.tab_folders)
        put("Cover", R.string.tab_cover)
        put("Lyrics", R.string.tab_lyrics)
        put("Equalizer", R.string.tab_equalizer)
        put("Effect", R.string.effect)
        put("Custom", R.string.custom)
    }

    val items = listOf(
        R.string.theme_follow_system,
        R.string.theme_light,
        R.string.theme_dark,
        R.string.theme_follow_music_cover,
        R.string.theme_material_you,
    )
    var bandsCenter = floatArrayOf(
//        31.0, 62.0, 125.0, 250.0, 500.0, 1000.0, 2000.0, 4000.0, 8000.0, 16000.0
        60.0f, 170.0f, 310.0f, 600.0f, 1000.0f, 3000.0f, 6000.0f, 12000.0f, 14000.0f, 16000.0f
    )
    const val Q = 3.2f

    var equalizerMax = 13
    var equalizerMin = -13
    var custom = "Custom"
    var eqPreset = LinkedHashMap<String, IntArray>().apply {
        put("Zero", intArrayOf(0, 0, 0, 0, 0, 0, 0, 0, 0, 0))
        put("Classical", intArrayOf(0, 0, 0, 0, 0, 0, -9, -9, -9, -12))
        put("Club", intArrayOf(0, 0, 2, 3, 3, 3, 2, 0, 0, 0))
        put("Dance", intArrayOf(6, 4, 1, 0, 0, -7, -9, -9, 0, 0))
        put("FullBass", intArrayOf(8, 8, 8, 4, 2, -10, -12, -13, -13, -13))
        put("FullTreble", intArrayOf(-12, -12, -12, -6, 1, 6, 9, 9, 9, 10))
        put("FullBass+Treble", intArrayOf(4, 3, 0, -9, -6, 1, 5, 6, 7, 7))
        put("Laptop/Headphones", intArrayOf(3, 6, 3, -4, 0, -7, -9, -9, 0, 0))
        put("LargeHall", intArrayOf(6, 6, 3, 3, 0, -6, -6, -6, 0, 0))
        put("Live", intArrayOf(-6, 0, 2, 3, 3, 3, 2, 1, 1, 1))
        put("Party", intArrayOf(4, 4, 0, 0, 0, 0, 0, 0, 4, 4))
        put("Pop", intArrayOf(-2, 3, 4, 4, 3, -1, -3, -3, -2, -2))
        put("Reggae", intArrayOf(0, 0, -1, -7, 0, -8, -8, 0, 0, 0))
        put("Rock", intArrayOf(4, 3, -7, -9, -4, 2, 5, 6, 6, 6))
        put("Soft", intArrayOf(3, 1, -1, -3, -1, 2, 5, 6, 6, 7))
        put("Ska", intArrayOf(-3, -6, -6, -1, 2, 3, 5, 6, 6, 6))
        put("SoftRock", intArrayOf(2, 2, 1, -1, -6, -7, -4, -1, 1, 5))
        put("Techno", intArrayOf(4, 3, 0, -7, -6, 0, 4, 6, 6, 5))
    }

    fun initSettingsData(musicViewModel: MusicViewModel, context: Context) {
        CoroutineScope(Dispatchers.IO).launch {
            musicViewModel.showSlideIndicators.value =
                SharedPreferencesUtils.getShowSlideIndicators(context)
            musicViewModel.musicVisualizationEnable.value =
                SharedPreferencesUtils.getEnableMusicVisualization(context)
            musicViewModel.showMusicCover.value =
                SharedPreferencesUtils.getShowMusicCover(context)
            musicViewModel.themeSelected.intValue = context.getSharedPreferences(
                "SelectedTheme",
                Context.MODE_PRIVATE
            ).getInt("SelectedTheme", 0)
            musicViewModel.textAlign.value =
                SharedPreferencesUtils.getDisplayAlign(context)
            musicViewModel.fontSize.intValue = SharedPreferencesUtils.getFontSize(context)
            musicViewModel.autoScroll.value =
                SharedPreferencesUtils.getAutoScroll(context)
            musicViewModel.autoHighLight.value =
                SharedPreferencesUtils.getAutoHighLight(context)
            val dicApps = musicViewModel.getDb(context).DictionaryAppDao().findAllDictionaryApp()
            if (dicApps.isEmpty()) {
                val list = ArrayList<DictionaryApp>()
                getAllDictionaryActivity(context)
                    .forEachIndexed { index, it ->
                        list.add(
                            DictionaryApp(
                                index,
                                it.activityInfo.name,
                                it.activityInfo.packageName,
                                it.loadLabel(context.packageManager).toString(),
                                isShow = true,
                                autoGo = false
                            )
                        )
                    }
                musicViewModel.dictionaryAppList.addAll(list)
            } else {
                musicViewModel.dictionaryAppList.addAll(dicApps)
            }
        }
    }

    fun formatTime(millis: Long): String {
        val totalSeconds = millis / 1000
        val seconds = totalSeconds % 60
        var minutes = totalSeconds / 60
        val hours = minutes / 60
        minutes %= 60
        return if (hours > 0) String.format(
            Locale.getDefault(),
            "%02d:%02d:%02d",
            hours,
            minutes,
            seconds
        ) else String.format(Locale.getDefault(), "%02d:%02d", minutes, seconds)
    }

    fun formatTimeWithUnit(millis: Long): String {
        val totalSeconds = millis / 1000
        val seconds = totalSeconds % 60
        var minutes = totalSeconds / 60
        val hours = minutes / 60
        minutes %= 60
        return if (hours > 0) String.format(
            Locale.getDefault(),
            "%02dh %02dm %02ds",
            hours,
            minutes,
            seconds
        ) else String.format(Locale.getDefault(), "%02dm %02ds", minutes, seconds)
    }

    private fun openFile(path: String, minType: String = "text/plain", context: Context) {
        val fileUri: Uri
        val outputImage = File(path)
        fileUri =
            FileProvider.getUriForFile(
                context,
                "${BuildConfig.APPLICATION_ID}.fileprovider",
                outputImage
            )
        val intent = Intent(Intent.ACTION_VIEW)
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION or Intent.FLAG_GRANT_WRITE_URI_PERMISSION)
        if (TextUtils.isEmpty(minType)) {
            intent.data = fileUri
        } else {
            intent.setDataAndType(fileUri, minType)
        }
        context.startActivity(intent)
    }

    private val retriever = MediaMetadataRetriever()
    fun getCover(
        musicViewModel: MusicViewModel?,
        context: Context,
        musicId: Long,
        path: String
    ): ImageSource {
//        try {
//            var uri = MediaStore.Audio.Media.getContentUri(MediaStore.VOLUME_EXTERNAL_PRIMARY)
//            uri = ContentUris.withAppendedId(uri, musicId)
//            val thumbnail =
//                context.contentResolver.loadThumbnail(uri, Size(512, 512), null);
//            return thumbnail
//        } catch (_: IOException) {
//        }
        try {
            retriever.setDataSource(path)
            val coverT = retriever.embeddedPicture
            if (coverT != null) {
                return ImageSource.BitmapFile(
                    BitmapFactory.decodeByteArray(
                        coverT,
                        0,
                        coverT.size
                    ).scale(512, 512, false)
                )
            }
        } catch (_: Exception) {
        }
        when (val defaultCoverResId = musicViewModel?.customMusicCover?.value) {
            null -> {
                return ImageSource.Resource(R.drawable.songs_thumbnail_cover)
            }

            is Int -> {
                return ImageSource.Resource(defaultCoverResId)
            }

            is String -> {
                val sourceFile = File(defaultCoverResId)
                return if (sourceFile.exists()) {
                    ImageSource.FilePath(defaultCoverResId)
                } else {
                    musicViewModel.customMusicCover.value =
                        ImageSource.Resource(R.drawable.songs_thumbnail_cover)
                    SharedPreferencesUtils.setTrackCoverData(context, "")
                    ImageSource.Resource(R.drawable.songs_thumbnail_cover)
                }
            }

            else -> return ImageSource.Resource(R.drawable.songs_thumbnail_cover)
        }
    }

    /**
     * for edit track page
     */
    fun getCoverNoFallback(path: String): Bitmap? {
        try {
            retriever.setDataSource(path)
            val coverT = retriever.embeddedPicture
            if (coverT != null) {
                return BitmapFactory.decodeByteArray(
                    coverT,
                    0,
                    coverT.size
                ).scale(512, 512, false)
            }
        } catch (_: Exception) {
        }
        return null
    }

    /**
     * for play music widget
     */
    fun getCoverBitmap(
        context: Context,
        path: String
    ): Bitmap {
        try {
            retriever.setDataSource(path)
            val coverT = retriever.embeddedPicture
            if (coverT != null) {
                return BitmapFactory.decodeByteArray(
                    coverT,
                    0,
                    coverT.size
                ).scale(512, 512, false)
            }
        } catch (_: Exception) {
        }
        val defaultCoverResId = SharedPreferencesUtils.getTrackCoverData(
            context
        )
        if (defaultCoverResId == null) {
            return BitmapFactory.decodeResource(context.resources, R.drawable.songs_thumbnail_cover)
                .scale(512, 512, false)
        } else {
            val sourceFile = File(defaultCoverResId)
            return if (sourceFile.exists()) {
                BitmapFactory.decodeFile(sourceFile.absolutePath).scale(512, 512, false)
            } else {
                SharedPreferencesUtils.setTrackCoverData(context, "")
                BitmapFactory.decodeResource(context.resources, R.drawable.songs_thumbnail_cover)
                    .scale(512, 512, false)
            }
        }
    }

    @Suppress("unused")
    fun copyFile(sourceFile: File, destinationFile: File) {
        try {
            val inputStream = FileInputStream(sourceFile)
            val outputStream = FileOutputStream(destinationFile)

            val buffer = ByteArray(1024)
            var length: Int

            while (inputStream.read(buffer).also { length = it } > 0) {
                outputStream.write(buffer, 0, length)
            }

            inputStream.close()
            outputStream.close()
        } catch (e: IOException) {
            e.printStackTrace()
        }
    }

    fun openBrowser(link: String, context: Context) {
        val intent = Intent(Intent.ACTION_VIEW, link.toUri())
        context.startActivity(intent)
    }

    fun sendEmail(recipient: String, subject: String, context: Context) {
        val intent = Intent(Intent.ACTION_SENDTO).apply {
            data = "mailto:".toUri() // only email apps should handle this
            putExtra(Intent.EXTRA_EMAIL, arrayOf(recipient))
            putExtra(Intent.EXTRA_SUBJECT, subject)
        }
        context.startActivity(intent)
    }

    fun createPlayListAddTracks(
        name: String,
        context: Context,
        type: PlayListType,
        id: Long,
        musicViewModel: MusicViewModel,
        removeDuplicate: Boolean
    ) {
        if (name.isNotEmpty()) {
            val futureResult: ListenableFuture<LibraryResult<ImmutableList<MediaItem>>>? =
                musicViewModel.browser?.getChildren(
                    type.name + "_track_" + id,
                    0,
                    Integer.MAX_VALUE,
                    null
                )
            futureResult?.addListener({
                try {
                    val result: LibraryResult<ImmutableList<MediaItem>>? = futureResult.get()
                    if (result == null || result.resultCode != LibraryResult.RESULT_SUCCESS) {
                        return@addListener
                    }
                    val albumMediaItems: List<MediaItem> = result.value ?: listOf()
                    val tracksList = ArrayList<MusicItem>()
                    albumMediaItems.forEach { mediaItem ->
                        MediaItemUtils.mediaItemToMusicItem(mediaItem)?.let { tracksList.add(it) }
                    }
                    val idPlayList =
                        PlaylistManager.createPlaylist(context, name, tracksList, false)
                    if (idPlayList == null) {
                        Toast.makeText(
                            context,
                            context.getString(R.string.create_failed),
                            Toast.LENGTH_SHORT
                        ).show()
                    } else {
                        SongsUtils.refreshPlaylist(musicViewModel)
                    }
                } catch (e: Exception) {
                    // 处理在获取结果过程中可能发生的异常 (如 ExecutionException)
                    Log.e("Client", "Failed to toggle favorite status", e)
                }
            }, ContextCompat.getMainExecutor(context))
        }
    }

    fun createPlayListAddTrack(
        name: String,
        context: Context,
        item: MusicItem,
        musicViewModel: MusicViewModel,
        removeDuplicate: Boolean
    ) {
        if (name.isNotEmpty()) {
            val tIds = ArrayList<MusicItem>()
            tIds.add(item)
            val idPlayList = PlaylistManager.createPlaylist(context, name, tIds, removeDuplicate)
            if (idPlayList != null) {
                musicViewModel.browser?.sendCustomCommand(
                    MediaCommands.COMMAND_PlAY_LIST_CHANGE,
                    Bundle().apply {},
                )
            } else {
                Toast.makeText(
                    context,
                    context.getString(R.string.create_failed),
                    Toast.LENGTH_SHORT
                )
                    .show()
            }
        }
    }

    fun addTracksToPlayList(
        playListId: Long,
        context: Context,
        type: PlayListType,
        id: Long,
        musicViewModel: MusicViewModel,
        removeDuplicate: Boolean
    ) {
        val futureResult: ListenableFuture<LibraryResult<ImmutableList<MediaItem>>>? =
            musicViewModel.browser?.getChildren(
                type.name + "_track_" + id,
                0,
                Integer.MAX_VALUE,
                null
            )
        futureResult?.addListener({
            try {
                val result: LibraryResult<ImmutableList<MediaItem>>? = futureResult.get()
                if (result == null || result.resultCode != LibraryResult.RESULT_SUCCESS) {
                    return@addListener
                }
                val albumMediaItems: List<MediaItem> = result.value ?: listOf()
                val tracksList = ArrayList<MusicItem>()
                albumMediaItems.forEach { mediaItem ->
                    MediaItemUtils.mediaItemToMusicItem(mediaItem)?.let { tracksList.add(it) }
                }
                val tIds = ArrayList(tracksList.map { item -> item })
                if (PlaylistManager.addMusicsToPlaylist(
                        context,
                        playListId,
                        tIds,
                        removeDuplicate
                    )
                ) {
                    SongsUtils.refreshPlaylist(musicViewModel)
                }
            } catch (e: Exception) {
                // 处理在获取结果过程中可能发生的异常 (如 ExecutionException)
                Log.e("Client", "Failed to toggle favorite status", e)
            }
        }, ContextCompat.getMainExecutor(context))

    }

    fun operateDialogDeal(
        operateType: OperateType,
        item: AnyListBase,
        musicViewModel: MusicViewModel
    ) {
        when (operateType) {
            OperateType.AddToQueue -> {
                val futureResult: ListenableFuture<LibraryResult<ImmutableList<MediaItem>>>? =
                    musicViewModel.browser?.getChildren(
                        item.type.name + "_track_" + item.id,
                        0,
                        Integer.MAX_VALUE,
                        null
                    )
                futureResult?.addListener({
                    try {
                        val result: LibraryResult<ImmutableList<MediaItem>>? = futureResult.get()
                        if (result == null || result.resultCode != LibraryResult.RESULT_SUCCESS) {
                            return@addListener
                        }
                        val albumMediaItems: List<MediaItem> = result.value ?: listOf()
                        val tracksList = ArrayList<MusicItem>()
                        albumMediaItems.forEach { mediaItem ->
                            MediaItemUtils.mediaItemToMusicItem(mediaItem)
                                ?.let { tracksList.add(it) }
                        }
                        musicViewModel.musicQueue.addAll(tracksList)
                        val mediaItems = tracksList.map { musicItem ->
                            MediaItemUtils.musicItemToMediaItem(musicItem)
                        }
                        musicViewModel.browser?.addMediaItems(
                            mediaItems
                        )
                    } catch (e: Exception) {
                        // 处理在获取结果过程中可能发生的异常 (如 ExecutionException)
                        Log.e("Client", "Failed to toggle favorite status", e)
                    }
                }, MoreExecutors.directExecutor())
            }

            OperateType.PlayNext -> {
                val futureResult: ListenableFuture<LibraryResult<ImmutableList<MediaItem>>>? =
                    musicViewModel.browser?.getChildren(
                        item.type.name + "_track_" + item.id,
                        0,
                        Integer.MAX_VALUE,
                        null
                    )
                futureResult?.addListener({
                    try {
                        val result: LibraryResult<ImmutableList<MediaItem>>? = futureResult.get()
                        if (result == null || result.resultCode != LibraryResult.RESULT_SUCCESS) {
                            return@addListener
                        }
                        val albumMediaItems: List<MediaItem> = result.value ?: listOf()
                        val tracksList = ArrayList<MusicItem>()
                        albumMediaItems.forEach { mediaItem ->
                            MediaItemUtils.mediaItemToMusicItem(mediaItem)
                                ?.let { tracksList.add(it) }
                        }
                        val i = musicViewModel.browser?.currentMediaItemIndex
                        val position = if (i != null) {
                            if (i == C.INDEX_UNSET) {
                                0
                            } else {
                                i + 1
                            }
                        } else {
                            0
                        }
                        addTracksToQueue(
                            musicViewModel,
                            tracksList,
                            position
                        )
                    } catch (e: Exception) {
                        // 处理在获取结果过程中可能发生的异常 (如 ExecutionException)
                        Log.e("Client", "Failed to toggle favorite status", e)
                    }
                }, MoreExecutors.directExecutor())
            }


            else -> {

            }
        }
    }

    fun addTracksToQueue(
        musicViewModel: MusicViewModel,
        tracksList: ArrayList<MusicItem>,
        indexTemp: Int
    ) {
        var index = indexTemp
        if (musicViewModel.musicQueue.isEmpty()) {
            index = 0
        }
        musicViewModel.musicQueue.addAll(
            index,
            tracksList
        )
        val mediaItems = tracksList.map { music ->
            MediaItemUtils.musicItemToMediaItem(music)
        }
        musicViewModel.browser?.addMediaItems(
            index,
            mediaItems
        )
    }

    fun deleteTrackUpdate(musicViewModel: MusicViewModel) {
        musicViewModel.refreshPlayList.value =
            !musicViewModel.refreshPlayList.value
        musicViewModel.refreshAlbum.value =
            !musicViewModel.refreshAlbum.value
        musicViewModel.refreshArtist.value =
            !musicViewModel.refreshArtist.value
        musicViewModel.refreshGenre.value =
            !musicViewModel.refreshGenre.value
        musicViewModel.refreshFolder.value =
            !musicViewModel.refreshFolder.value
        musicViewModel.songsList.clear()
        musicViewModel.playListCurrent.value = null
        musicViewModel.viewModelScope.launch(Dispatchers.Main) {
            val futureResult: ListenableFuture<SessionResult>? =
                musicViewModel.browser?.sendCustomCommand(
                    MediaCommands.COMMAND_REFRESH_ALL,
                    Bundle().apply { },
                )
            futureResult?.addListener({
                try {
                    val sessionResult = futureResult.get()
                    if (sessionResult.resultCode == SessionResult.RESULT_SUCCESS) {
                        musicViewModel.refreshPlayList.value =
                            !musicViewModel.refreshPlayList.value
                        musicViewModel.refreshAlbum.value =
                            !musicViewModel.refreshAlbum.value
                        musicViewModel.refreshArtist.value =
                            !musicViewModel.refreshArtist.value
                        musicViewModel.refreshGenre.value =
                            !musicViewModel.refreshGenre.value
                        musicViewModel.refreshFolder.value =
                            !musicViewModel.refreshFolder.value
                        sessionResult.extras.getParcelableArrayList<MusicItem>(
                            "songsList"
                        )?.also {
                            musicViewModel.songsList.clear()
                            musicViewModel.songsList.addAll(it)
                        }
                    }
                } catch (e: Exception) {
                    Log.e("Client", "Failed to toggle favorite status", e)
                }
            }, MoreExecutors.directExecutor())
        }
//        resultData?.getLong("id", -1)?.also {
//            if (it == -1L) return
//            musicViewModel.musicQueue.removeAll { mIt ->
//                mIt.id == it
//            }
//            if (it == musicViewModel.currentPlay.value?.id) {
//                musicViewModel.currentPlay.value = null
//            }
//        }
    }

    fun getAllDictionaryActivity(context: Context): List<ResolveInfo> {
        val shareIntent = Intent(Intent.ACTION_PROCESS_TEXT)
        shareIntent.type = "text/plain"
        shareIntent.putExtra(Intent.EXTRA_TEXT, "")
        val resolveInfoList: List<ResolveInfo> =
            context.packageManager.queryIntentActivities(shareIntent, 0)
//        for (resolveInfo in resolveInfoList) {
//            val activityInfo = resolveInfo.activityInfo
//            val packageName = activityInfo.packageName
//            val className = activityInfo.name
//            val label = resolveInfo.loadLabel(context.packageManager).toString()
//        }
        return resolveInfoList
    }

    fun clearAlbumCoverCache(context: Context) {
        val folder = File(context.externalCacheDir, "album_cover")
        folder.mkdirs()
        folder.listFiles()?.forEach {
            it.delete()
        }
    }

    @OptIn(UnstableApi::class)
    fun setLyricsFolder(context: Context) {
        val intent = Intent(Intent.ACTION_OPEN_DOCUMENT_TREE)
        (context as MainActivity).folderPickerLauncher.launch(intent)
    }

    @OptIn(UnstableApi::class)
    fun setArtistFolder(context: Context) {
        val intent = Intent(Intent.ACTION_OPEN_DOCUMENT_TREE)
        (context as MainActivity).artistFolderPickerLauncher.launch(intent)
    }

    @OptIn(UnstableApi::class)
    fun setGenreFolder(context: Context) {
        val intent = Intent(Intent.ACTION_OPEN_DOCUMENT_TREE)
        (context as MainActivity).genreFolderPickerLauncher.launch(intent)
    }

    @OptIn(UnstableApi::class)
    fun setLyricsFile(musicViewModel: MusicViewModel, context: Context) {
        if (musicViewModel.currentPlay.value != null) {
            val regexPattern = Regex("[<>\"/~'{}?,+=)(^&*%!@#$]")
            val artistsFolder = musicViewModel.currentPlay.value?.artist
                ?.replace(
                    regexPattern,
                    "_"
                )
            val folderPath = "$Lyrics/$artistsFolder"
            val folder = context.getExternalFilesDir(
                folderPath
            )
            folder?.mkdirs()
            val id =
                musicViewModel.currentPlay.value?.name?.replace(regexPattern, "_")
            val pathLyrics: String =
                context.getExternalFilesDir(folderPath)?.absolutePath + "/$id.lrc"
            val path: String =
                context.getExternalFilesDir(folderPath)?.absolutePath + "/$id.txt"
            val lyrics = File(pathLyrics)
            val text = File(path)
            if (lyrics.exists()) {
                openFile(lyrics.path, context = context)
            } else if (text.exists()) {
                openFile(text.path, context = context)
            } else {
                val tempPath: String =
                    context.getExternalFilesDir(folderPath)?.absolutePath + "/$id."
                (context as MainActivity).openFilePicker(tempPath)
            }
        }
    }

    fun checkLyrics(path: String): CheckLyricsData? {
        if (File("$path.lrc").canRead()) {
            return CheckLyricsData("$path.lrc", LyricsType.LRC)
        } else if (File("$path.txt").canRead()) {
            return CheckLyricsData("$path.txt", LyricsType.TEXT)
        } else if (File("$path.srt").canRead()) {
            return CheckLyricsData("$path.srt", LyricsType.SRT)
        } else if (File("$path.vtt").canRead()) {
            return CheckLyricsData("$path.vtt", LyricsType.VTT)
        }
        return null
    }

    fun Dp.toPx(context: Context): Int {
        val displayMetrics = context.resources.displayMetrics
        return TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, this.value, displayMetrics)
            .toInt()
    }

    @OptIn(UnstableApi::class)
    fun setCoverFile(musicViewModel: MusicViewModel, context: Context) {
        (context as MainActivity).roseImagPicker()
    }

    fun getFileNameFromUri(context: Context, uri: Uri): String? {
        var result: String? = null
        if (uri.scheme == "content") {
            val cursor = context.contentResolver.query(uri, null, null, null, null)
            cursor?.use {
                if (it.moveToFirst()) {
                    val index = it.getColumnIndex(OpenableColumns.DISPLAY_NAME)
                    if (index != -1) {
                        result = it.getString(index)
                    }
                }
            }
        }
        if (result == null) {
            result = uri.path
            val cut = result?.lastIndexOf('/')
            if (cut != null && cut != -1) {
                result = result.substring(cut + 1)
            }
        }
        return result
    }

    /**
     * 检查用户是否已经创建了指定 AppWidgetProvider 的任何小部件实例。
     *
     * @param context 上下文
     * @param widgetProviderClass 你的 AppWidgetProvider 的 Class 对象，例如 `MyWidgetProvider::class.java`
     * @return 如果至少存在一个小部件，则返回 true；否则返回 false。
     */
    fun hasAppWidget(context: Context, widgetProviderClass: Class<*>): Boolean {
        val appWidgetManager = AppWidgetManager.getInstance(context)
        val componentName = ComponentName(context, widgetProviderClass)
        val appWidgetIds: IntArray = appWidgetManager.getAppWidgetIds(componentName)
        return appWidgetIds.isNotEmpty()
    }

}