package com.zhangke.fread.activitypub.app.internal.adapter

import com.zhangke.activitypub.entities.ActivityPubFilterEntity
import com.zhangke.activitypub.entities.ActivityPubFilterResultEntity
import com.zhangke.activitypub.entities.ActivityPubMediaAttachmentEntity
import com.zhangke.activitypub.entities.ActivityPubQuoteApprovalEntity
import com.zhangke.activitypub.entities.ActivityPubStatusEntity
import com.zhangke.framework.date.DateParser
import com.zhangke.framework.ktx.ifNullOrEmpty
import com.zhangke.framework.utils.WebFinger
import com.zhangke.fread.activitypub.app.internal.model.ActivityPubLoggedAccount
import com.zhangke.fread.common.utils.formatDefault
import com.zhangke.fread.status.author.BlogAuthor
import com.zhangke.fread.status.blog.Blog
import com.zhangke.fread.status.blog.BlogEmbed
import com.zhangke.fread.status.blog.BlogMedia
import com.zhangke.fread.status.blog.BlogMediaType
import com.zhangke.fread.status.blog.CurrentUserQuoteApproval
import com.zhangke.fread.status.blog.PostingApplication
import com.zhangke.fread.status.model.BlogFiltered
import com.zhangke.fread.status.model.BlogTranslationUiState
import com.zhangke.fread.status.model.HashtagInStatus
import com.zhangke.fread.status.model.Mention
import com.zhangke.fread.status.model.PlatformLocator
import com.zhangke.fread.status.model.StatusUiState
import com.zhangke.fread.status.model.StatusVisibility
import com.zhangke.fread.status.model.createActivityPubProtocol
import com.zhangke.fread.status.platform.BlogPlatform
import com.zhangke.fread.status.status.model.Status
import me.tatarka.inject.annotations.Inject

class ActivityPubStatusAdapter @Inject constructor(
    private val activityPubAccountEntityAdapter: ActivityPubAccountEntityAdapter,
    private val metaAdapter: ActivityPubBlogMetaAdapter,
    private val pollAdapter: ActivityPubPollAdapter,
    private val emojiEntityAdapter: ActivityPubCustomEmojiEntityAdapter,
) {

    fun toStatusUiState(
        status: Status,
        locator: PlatformLocator,
        logged: Boolean,
        isOwner: Boolean,
    ): StatusUiState {
        return StatusUiState(
            status = status,
            locator = locator,
            logged = logged,
            isOwner = isOwner,
            blogTranslationState = BlogTranslationUiState(
                support = status.intrinsicBlog.supportTranslate,
                translating = false,
                showingTranslation = false,
                blogTranslation = null,
            ),
        )
    }

    fun toStatusUiState(
        status: Status,
        locator: PlatformLocator,
        loggedAccount: ActivityPubLoggedAccount?,
    ): StatusUiState {
        return toStatusUiState(
            status = status,
            locator = locator,
            logged = loggedAccount != null,
            isOwner = loggedAccount?.webFinger?.equalsDomain(status.intrinsicBlog.author.webFinger) == true,
        )
    }

    fun toStatusUiState(
        entity: ActivityPubStatusEntity,
        platform: BlogPlatform,
        locator: PlatformLocator,
        loggedAccount: ActivityPubLoggedAccount?,
    ): StatusUiState {
        val status = toStatus(entity, platform)
        return toStatusUiState(status, locator, loggedAccount)
    }

    fun toStatusUiState(
        entity: ActivityPubStatusEntity,
        platform: BlogPlatform,
        locator: PlatformLocator,
        isOwner: Boolean,
        logged: Boolean,
    ): StatusUiState {
        val status = toStatus(entity, platform)
        return toStatusUiState(status, locator, logged = logged, isOwner = isOwner)
    }

    fun toStatus(
        entity: ActivityPubStatusEntity,
        platform: BlogPlatform,
    ): Status {
        return if (entity.reblog != null) {
            transformReblog(entity, platform)
        } else {
            transformNewBlog(entity, platform)
        }
    }

    private fun transformNewBlog(
        entity: ActivityPubStatusEntity,
        platform: BlogPlatform,
    ): Status.NewBlog {
        val blog = getBlogAndInteractions(entity, platform)
        return Status.NewBlog(blog)
    }

    private fun transformReblog(
        entity: ActivityPubStatusEntity,
        platform: BlogPlatform,
    ): Status.Reblog {
        val blog = getBlogAndInteractions(entity.reblog!!, platform)
        return Status.Reblog(
            author = activityPubAccountEntityAdapter.toAuthor(entity.account),
            id = entity.id,
            createAt = DateParser.parseOrCurrent(entity.createdAt),
            reblog = blog,
        )
    }

    private fun getBlogAndInteractions(
        entity: ActivityPubStatusEntity,
        platform: BlogPlatform
    ): Blog {
        val statusAuthor = activityPubAccountEntityAdapter.toAuthor(entity.account)
        return transformBlog(entity, platform, statusAuthor)
    }

    private fun transformBlog(
        entity: ActivityPubStatusEntity,
        platform: BlogPlatform,
        author: BlogAuthor,
    ): Blog {
        val emojis = entity.emojis.map(emojiEntityAdapter::toEmoji)
        val createAt = DateParser.parseOrCurrent(entity.createdAt)
        return Blog(
            id = entity.id,
            author = author,
            title = null,
            description = null,
            content = entity.content.orEmpty(),
            sensitive = entity.sensitive,
            spoilerText = entity.spoilerText,
            createAt = createAt,
            formattedCreateAt = createAt.formatDefault(),
            url = entity.url.ifNullOrEmpty { entity.uri },
            link = entity.url.ifNullOrEmpty { entity.uri },
            language = entity.language,
            like = Blog.Like(
                support = true,
                liked = entity.favourited,
                likedCount = entity.favouritesCount.toLong()
            ),
            forward = Blog.Forward(
                support = true,
                forward = entity.reblogged,
                forwardCount = entity.reblogsCount.toLong(),
            ),
            bookmark = Blog.Bookmark(
                support = true,
                bookmarked = entity.bookmarked,
            ),
            reply = Blog.Reply(
                support = true,
                repliesCount = entity.repliesCount.toLong(),
            ),
            quote = buildQuote(entity, platform),
            supportEdit = true,
            isReply = !entity.inReplyToId.isNullOrEmpty(),
            platform = platform,
            mediaList = entity.mediaAttachments?.map { it.toBlogMedia() } ?: emptyList(),
            poll = entity.poll?.let(pollAdapter::adapt),
            emojis = emojis,
            pinned = entity.pinned == true,
            facets = emptyList(),
            supportTranslate = true,
            mentions = entity.mentions.mapNotNull { it.toMention() },
            tags = entity.tags.map { it.toTag() },
            visibility = entity.visibility.convertActivityPubVisibility(),
            embeds = buildEmbed(entity, platform),
            editedAt = entity.editedAt?.let { DateParser.parseOrCurrent(it) },
            application = entity.application?.toApplication(),
            filtered = entity.filtered?.map { it.toFiltered() },
        )
    }

    private fun String.convertActivityPubVisibility(): StatusVisibility {
        return when (this) {
            ActivityPubStatusEntity.VISIBILITY_PUBLIC -> StatusVisibility.PUBLIC
            ActivityPubStatusEntity.VISIBILITY_UNLISTED -> StatusVisibility.UNLISTED
            ActivityPubStatusEntity.VISIBILITY_PRIVATE -> StatusVisibility.PRIVATE
            ActivityPubStatusEntity.VISIBILITY_DIRECT -> StatusVisibility.DIRECT
            else -> StatusVisibility.PUBLIC
        }
    }

    private fun ActivityPubMediaAttachmentEntity.toBlogMedia(): BlogMedia {
        val mediaType = convertMediaType(type)
        return BlogMedia(
            id = id,
            url = url.orEmpty(),
            type = mediaType,
            previewUrl = previewUrl,
            remoteUrl = remoteUrl,
            description = description,
            meta = this.meta?.let { metaAdapter.adapt(mediaType, it) },
            blurhash = blurhash,
        )
    }

    private fun ActivityPubStatusEntity.Tag.toTag(): HashtagInStatus {
        return HashtagInStatus(
            name = name,
            url = url,
            protocol = createActivityPubProtocol(),
        )
    }

    private fun convertMediaType(type: String): BlogMediaType {
        return when (type) {
            ActivityPubMediaAttachmentEntity.TYPE_IMAGE -> BlogMediaType.IMAGE
            ActivityPubMediaAttachmentEntity.TYPE_AUDIO -> BlogMediaType.AUDIO
            ActivityPubMediaAttachmentEntity.TYPE_VIDEO -> BlogMediaType.VIDEO
            ActivityPubMediaAttachmentEntity.TYPE_GIFV -> BlogMediaType.GIFV
            ActivityPubMediaAttachmentEntity.TYPE_UNKNOWN -> BlogMediaType.UNKNOWN
            else -> throw IllegalArgumentException("Unsupported media type(${type})!")
        }
    }

    private fun ActivityPubStatusEntity.Mention.toMention(): Mention? {
        val webFinger = WebFinger.create(acct) ?: WebFinger.create(this.url) ?: return null
        return Mention(
            id = id,
            username = username,
            url = url,
            webFinger = webFinger,
            protocol = createActivityPubProtocol(),
        )
    }

    private fun buildEmbed(
        entity: ActivityPubStatusEntity,
        platform: BlogPlatform,
    ): List<BlogEmbed> {
        val list = mutableListOf<BlogEmbed>()
        entity.card?.toEmbed()?.let { list += it }
        if (entity.quote != null) {
            val quotedStatus = entity.quote?.quotedStatus
            if (quotedStatus != null) {
                val statusAuthor = activityPubAccountEntityAdapter.toAuthor(quotedStatus.account)
                val quotedBlog = transformBlog(quotedStatus, platform, statusAuthor)
                list += BlogEmbed.Blog(quotedBlog)
            } else {
                list += BlogEmbed.UnavailableQuote(
                    reason = entity.quote?.state.orEmpty(),
                    blogId = entity.quote?.quotedStatusId,
                )
            }
        }
        return list
    }

    private fun ActivityPubStatusEntity.PreviewCard.toEmbed(): BlogEmbed {
        return BlogEmbed.Link(
            url = url,
            title = title,
            description = description,
            video = type == ActivityPubStatusEntity.PreviewCard.TYPE_VIDEO,
            authorName = authorName,
            authorUrl = authorUrl,
            providerName = providerName,
            providerUrl = providerUrl,
            html = html,
            width = width,
            height = height,
            image = image,
            embedUrl = embedUrl,
            blurhash = blurhash,
        )
    }

    private fun buildQuote(
        entity: ActivityPubStatusEntity,
        blogPlatform: BlogPlatform,
    ): Blog.Quote {
        val currentUserApproval = entity.quoteApproval?.currentUser?.toApproval()
        return Blog.Quote(
            support = blogPlatform.supportsQuotePost == true,
            enabled = currentUserApproval?.quotable ?: false,
            currentUserApproval = currentUserApproval,
        )
    }

    private fun String.toApproval(): CurrentUserQuoteApproval {
        return when (this) {
            ActivityPubQuoteApprovalEntity.CURRENT_USER_AUTOMATIC -> CurrentUserQuoteApproval.AUTOMATIC
            ActivityPubQuoteApprovalEntity.CURRENT_USER_MANUAL -> CurrentUserQuoteApproval.MANUAL
            ActivityPubQuoteApprovalEntity.CURRENT_USER_DENIED -> CurrentUserQuoteApproval.DENIED
            else -> CurrentUserQuoteApproval.UNKNOWN
        }
    }

    private fun ActivityPubStatusEntity.Application.toApplication(): PostingApplication {
        return PostingApplication(
            name = name,
            website = website,
        )
    }

    private fun ActivityPubFilterResultEntity.toFiltered(): BlogFiltered {
        //warn, hide, blur
        return BlogFiltered(
            id = this.filter.id,
            title = this.filter.title,
            action = when (this.filter.filterAction) {
                ActivityPubFilterEntity.FILTER_ACTION_WARN -> BlogFiltered.FilterAction.WARN
                ActivityPubFilterEntity.FILTER_ACTION_KEYWORDS -> BlogFiltered.FilterAction.HIDE
                else -> BlogFiltered.FilterAction.BLUR
            },
            keywordMatches = keywordMatches ?: emptyList(),
        )
    }
}
