package com.zell_mbc.publicartexplorer.screens

import android.content.Context
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.NavigationBar
import androidx.compose.material3.NavigationBarItem
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.SnackbarResult
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.compositionLocalOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import com.zell_mbc.publicartexplorer.DEBUG_MODE_KEY
import com.zell_mbc.publicartexplorer.DebugOverlay
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.Screen
import com.zell_mbc.publicartexplorer.bottomNavItems
import com.zell_mbc.publicartexplorer.data.ViewModel
import kotlinx.coroutines.launch

val LocalSnackbarHostState = compositionLocalOf<SnackbarHostState> {
    error("SnackbarHostState not provided")
}

@Composable
fun NetworkStatusBanner(isOnline: Boolean) {
    AnimatedVisibility(visible = !isOnline, enter = fadeIn(), exit = fadeOut()) {
        Box(modifier = Modifier.fillMaxWidth().background(Color(0xFFFF5555)) // red banner for offline
                .padding(8.dp), contentAlignment = Alignment.Center
        ) {
            Text(text = stringResource(R.string.statusOffline), color = Color.White, style = MaterialTheme.typography.bodyMedium)
        }
    }
}


@Composable
fun MainScaffold(viewModel: ViewModel) {
    val context = LocalContext.current
    val snackbarHostState = remember { SnackbarHostState() }
    val scope = rememberCoroutineScope()
    val isOnline by viewModel.isOnline.collectAsState()

    val navController = rememberNavController()

    LaunchedEffect(viewModel.snackbarMessages) {
        viewModel.snackbarMessages.collect { msg ->
            // showSnackbar is a suspend function
            val result = snackbarHostState.showSnackbar(
                message = msg.message,
                actionLabel = msg.actionLabel
            )

            if (result == SnackbarResult.ActionPerformed && msg.action != null) {
                // launch the suspend action safely
                scope.launch {
                    msg.action.invoke()
                }
            }
        }
    }

    // Provide the LocalSnackbarHostState so children can read it
    CompositionLocalProvider(LocalSnackbarHostState provides snackbarHostState) {
        Scaffold(
            snackbarHost = { SnackbarHost(snackbarHostState) },
            topBar = {
                NetworkStatusBanner(isOnline = isOnline)
            },
            bottomBar = {
                NavigationBar {
                    val navBackStackEntry by navController.currentBackStackEntryAsState()
                    val currentRoute = navBackStackEntry?.destination?.route
                    bottomNavItems.forEach { screen ->
                        NavigationBarItem(
                            icon = {
                                Icon(
                                    painterResource(id = screen.icon),
                                    contentDescription = null
                                )
                            },
                            label = { Text(context.getString(screen.label)) },
                            selected = currentRoute == screen.route,
                            onClick = {
                                if (currentRoute != screen.route) {
                                    navController.navigate(screen.route) {
                                        popUpTo(navController.graph.startDestinationId)
                                        launchSingleTop = true
                                    }
                                }
                            }
                        )
                    }
                }
            }
        ) { innerPadding ->
            NavHost(
                navController,
                startDestination = Screen.Map.route,
                Modifier.padding(innerPadding)
            ) {
                composable(Screen.Map.route) {
                    val preferences = context.getSharedPreferences(
                        "com.zell_mbc.publicartexplorer._preferences",
                        Context.MODE_PRIVATE
                    )
                    viewModel.debugMode = preferences.getBoolean(DEBUG_MODE_KEY, false)

                    if (viewModel.debugMode)
                        DebugOverlay(viewModel) {
                            MapScreen(snackbarHostState, viewModel)
                        }
                    else MapScreen(snackbarHostState, viewModel)
                }
                composable(Screen.Settings.route) { SettingsScreen(viewModel) }
                composable(Screen.Info.route) { InfoScreen(viewModel) }
            }
        }
    }
}
