package com.zell_mbc.publicartexplorer.osm

import android.content.Context
import com.zell_mbc.publicartexplorer.Artwork
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.onboarding.BUILD_KEY
import com.zell_mbc.publicartexplorer.osmApiUrl
import com.zell_mbc.publicartexplorer.sharedOkHttpClient
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.Request
import okhttp3.RequestBody.Companion.toRequestBody
import org.xmlpull.v1.XmlPullParser
import org.xmlpull.v1.XmlPullParserFactory

@Serializable
data class OSMUserResponse(val user: OSMUser)

@Serializable
data class OSMUser(
    val id: Long,
    val display_name: String,
    val account_created: String
)

object OSMApi {
    private val client = sharedOkHttpClient
    private val json = Json { ignoreUnknownKeys = true }

    suspend fun getUserDetails(token: String): OSMUser? {
        return withContext(Dispatchers.IO) {
            val request = Request.Builder()
                .url("$osmApiUrl/api/0.6/user/details.json")
                .header("Authorization", "Bearer $token")
                .build()

            client.newCall(request).execute().use { response ->
                if (!response.isSuccessful) return@withContext null
                val body = response.body.string()
                return@withContext json.decodeFromString<OSMUserResponse>(body).user
            }
        }
    }
}

// Pass modified tags to OSM.
fun createOsmNode(selectedArtwork: Artwork, context: Context, token: String): String {
    val preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
    val build = preferences.getInt(BUILD_KEY,0).toString()
    val appName = context.getString(R.string.appName)
    var message: String
    var changesetId = ""

    if (selectedArtwork.tags.isEmpty())
        return "Upload error: Tag list is empty?"

    try {
        // Step 1: Create changeset XML
        val changesetXml = """
        <osm version="0.6" generator="$appName">
            <changeset>
                <tag k="created_by" v="$appName"/>
                <tag k="comment" v="Updating tags via $appName ($build)"/>
            </changeset>
        </osm>
        """.trimIndent()

        // Create changeset
        val changesetRequest = Request.Builder()
            .url("$osmApiUrl/api/0.6/changeset/create")
            .addHeader("Authorization", "Bearer $token")
            .put(changesetXml.toRequestBody("text/xml; charset=utf-8".toMediaType()))
            .build()

        changesetId = sharedOkHttpClient.newCall(changesetRequest).execute().use { response ->
            if (!response.isSuccessful) throw Exception(context.getString(R.string.failedToCreateChangeset) + ": ${response.code}")
            response.body.string().trim()
        }

        // Tags to xml
        val tagsXml = selectedArtwork.tags
            .filter { it.value.isNotBlank() } // skip empty or blank values
            .entries.joinToString("\n") { (key, value) ->
                """<tag k="$key" v="$value"/>"""
            }


        // 2. Fetch the current node to get its version
        val nodeId = selectedArtwork.id
        var request: Request
        val exceptionText: String
        if (nodeId.isNotEmpty()) { // Update
            val getNodeRequest = Request.Builder()
                .url("$osmApiUrl/api/0.6/node/$nodeId")
                .addHeader("Authorization", "Bearer $token")
                .get()
                .build()

            val nodeXml = sharedOkHttpClient.newCall(getNodeRequest).execute().use { resp ->
                if (!resp.isSuccessful) throw Exception("Failed to fetch node: ${resp.code}")
                resp.body.string()
            }

            // Extract node attributes: version, lat, lon
            val nodeRegex =
                Regex("""<node[^>]*id="$nodeId"[^>]*version="(\d+)"[^>]*lat="([0-9.-]+)"[^>]*lon="([0-9.-]+)"[^>]*>""")
            val match = nodeRegex.find(nodeXml) ?: throw Exception("Node XML parse failed")
            val (version, lat, lon) = match.destructured

            val updatedNodeXml = """
                <osm version="0.6" generator="PublicArtExplorer">
                <node id="$nodeId" changeset="$changesetId" version="$version" lat="$lat" lon="$lon">
                $tagsXml
                </node>
                </osm>
             """.trimIndent()

            request = Request.Builder()
                .url("$osmApiUrl/api/0.6/node/$nodeId") // ✅ for update
                .addHeader("Authorization", "Bearer $token")
                .put(updatedNodeXml.toRequestBody("text/xml; charset=utf-8".toMediaType()))
                .build()

            exceptionText = "Failed to update node"
        } else { // New node
            val lat = selectedArtwork.lat
            val lon = selectedArtwork.lon
            val newNodeXml = """
           <osmChange version="0.6" generator="PublicArtExplorer">
              <create>
                <node id="-1" changeset="$changesetId" lat="$lat" lon="$lon">
                  $tagsXml
                </node>
              </create>
           </osmChange>""".trimIndent()

            request = Request.Builder()
                .url("$osmApiUrl/api/0.6/changeset/$changesetId/upload") // ✅ correct endpoint
                .addHeader("Authorization", "Bearer $token")
                .post(newNodeXml.toRequestBody("text/xml; charset=utf-8".toMediaType()))
                .build()

            exceptionText = "Failed to create node"
        }

        sharedOkHttpClient.newCall(request).execute().use { resp ->
            if (!resp.isSuccessful) throw Exception("$exceptionText: ${resp.code}")
            val body = resp.body.string() ?: throw Exception("Empty response")

            // Extract the new node ID
            val match = Regex("""<node\s+old_id="-1"\s+new_id="(\d+)"""").find(body)
            val newNodeId = match?.groupValues?.get(1)?.toLong() ?: throw Exception("Could not extract new node ID from response")
            selectedArtwork.id = newNodeId.toString()
        }
        message = context.getString(R.string.updateSuccessful)
    } catch(e: Exception) {
        message = e.message.toString()
    }

    // Step 4: Close changeset
   try {
       val closeRequest = Request.Builder()
           .url("$osmApiUrl/api/0.6/changeset/$changesetId/close")
           .addHeader("Authorization", "Bearer $token")
           .put("".toRequestBody())
           .build()

       sharedOkHttpClient.newCall(closeRequest).execute().use { response ->
           if (!response.isSuccessful) throw Exception("Failed to close changeset: ${response.code}")
       }
   } catch(e: Exception) {
       message = e.message.toString()
   }

    return message
}


fun fetchChangesetCount(context: Context, token: String): String {
    val client = sharedOkHttpClient

    val request = Request.Builder()
        .url("$osmApiUrl/api/0.6/user/details")
        .addHeader("Authorization", "Bearer $token")
        .build()

    try {
        client.newCall(request).execute().use { response ->
            if (!response.isSuccessful) throw Exception("Unexpected code $response")

            val inputStream = response.body.byteStream()
            val factory = XmlPullParserFactory.newInstance()
            val parser = factory.newPullParser()
            parser.setInput(inputStream, null)

            var eventType = parser.eventType
            var count = 0

            while (eventType != XmlPullParser.END_DOCUMENT) {
                if (eventType == XmlPullParser.START_TAG && parser.name == "changesets") {
                    val attrValue = parser.getAttributeValue(null, "count")
                    count = attrValue?.toIntOrNull() ?: 0
                    break
                }
                eventType = parser.next()
            }

            return count.toString()
        }
    } catch (e: Exception) {
        return context.getString(R.string.changesetLookupError) + ": ${e.message}"
    }
}
