package com.zell_mbc.publicartexplorer.helpers

import com.zell_mbc.publicartexplorer.DebugLog
import com.zell_mbc.publicartexplorer.sharedOkHttpClient
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.Request
import org.json.JSONObject
import java.net.URLDecoder
import java.net.URLEncoder
import java.nio.charset.StandardCharsets
import java.security.MessageDigest

// Wikipedia "media viewer" URL
// Call with file name (language-agnostic)
suspend fun getWikimediaImageUrl(fileName: String): String = withContext(Dispatchers.IO) {
    val client = sharedOkHttpClient

    // Normalize file name: ignore any language-specific prefix, always prepend "File:"
    val normalizedFileName = "File:" + fileName.substringAfter(":", fileName)
    val apiUrl = "https://commons.wikimedia.org/w/api.php" +
            "?action=query" +
            "&titles=${normalizedFileName.replace(" ", "_")}" +
            "&prop=imageinfo" +
            "&iiprop=url|user|extmetadata" +
            "&format=json"

    val request = Request.Builder()
        .url(apiUrl)
        .build()

    client.newCall(request).execute().use { response ->
        if (!response.isSuccessful) return@withContext "NOTHING_FOUND"

        val body = response.body.string() // ?: return@withContext "NOTHING_FOUND"
        val json = JSONObject(body)

        val pages = json.getJSONObject("query").getJSONObject("pages")
        val page = pages.keys().asSequence().firstOrNull()?.let { pages.getJSONObject(it) }
        val imageInfo = page?.optJSONArray("imageinfo")

        val imageObj = imageInfo?.optJSONObject(0) // get the first object in the array
        val uploader = imageObj?.optString("user")
        val extMetadata = imageObj?.optJSONObject("extmetadata")
        val license = extMetadata?.optJSONObject("LicenseShortName")?.optString("value")
        val attribution = "$uploader|$license|"
        val url = imageInfo?.getJSONObject(0)?.optString("url")
        "$attribution$url"// Return all relevant values in one string, a bit of a hack
    }
}

suspend fun resolveImageUrl(inputUrl: String): String = withContext(Dispatchers.IO) {
    val trimmed = inputUrl.trim()

    // Heuristic: if we got 'http" and a second colon, treat as a file page needing normalization
    val needsNormalization = trimmed.count { it == ':' } > 1
    //val needsNormalization = if (trimmed.contains("http")) trimmed.count { it == ':' } > 1 // https://wikimedia_commons/?File:Museum_Biberach,_Statue_von_Johann_Baptist_Pflug.jpg
    //    else trimmed.count { it == ':' } > 0 // File:Museum_Biberach,_Statue_von_Johann_Baptist_Pflug.jpg

    if (needsNormalization) {
        val normalizedFileName = "File:" + trimmed.substringAfterLast(":") // Make it work with different languages

        // Strip query params/fragments (e.g., ?uselang=de, #section)
        val trimmed = normalizedFileName.substringBefore("?").substringBefore("#")

        // Call Wikimedia API to get the actual image URL
        getWikimediaImageUrl(trimmed)
    } else
        // Otherwise, assume it's already a direct image URL
        trimmed
}

fun extractWikiWord(input: String): String? {
    val regex = Regex("""\b(wikipedia|wikimedia|wikidata)\b""", RegexOption.IGNORE_CASE)
    val result = regex.find(input)?.value
    return if (result != null) titlecase(result)
    else null
}

fun titlecase(input: String): String {
    return input.replaceFirstChar {
        if (it.isLowerCase()) it.titlecase() else it.toString()
    }
}


// Converts a Q tag to a url
suspend fun wikidataTagToImageUrl(id: String): String = withContext(Dispatchers.IO) {
    val apiUrl = if (id.startsWith("Q")) "https://www.wikidata.org/w/api.php?action=wbgetclaims&property=P18&entity=$id&format=json" else ""

    try {
        // Retrieve filename
        val client = sharedOkHttpClient //OkHttpClient()
        val request = Request.Builder().url(apiUrl).build()
        val response = client.newCall(request).execute()
        if (response.code != 200) return@withContext "NOTHING_FOUND"
        val jsonString = response.body.string()
        val json = JSONObject(jsonString)

        val p18Array = json.optJSONObject("claims")?.optJSONArray("P18") //?: return null
        if (p18Array == null) return@withContext "NOTHING_FOUND" // No image under this id

        p18Array.length().let {
            if (it > 0) {
                val mainsnak = p18Array.getJSONObject(0).optJSONObject("mainsnak")
                val datavalue = mainsnak?.optJSONObject("datavalue")
                val imageFilename = datavalue?.optString("value")

                val cleanedFileName = imageFilename?.replace(" ", "_")
                if (imageFilename != null) {
                    val md5 = MessageDigest.getInstance("MD5")
                        .digest(cleanedFileName!!.toByteArray())
                        .joinToString("") { "%02x".format(it) }
                    val firstChar = md5.substring(0, 1)
                    val firstTwoChars = md5.substring(0, 2)
                    "https://upload.wikimedia.org/wikipedia/commons/$firstChar/$firstTwoChars/$cleanedFileName"
                }
                else "NOTHING_FOUND"
            }
            else "NOTHING_FOUND"
        }
    } catch (e: Exception) {
        //Log.d("getWikidataImageUrl", "$e")
        DebugLog.add("$e")
        "NOTHING_FOUND"
    }
}

// Lookup Wikimedia category and return first x images, if any
suspend fun getImageUrlsFromCategory(category: String, limit: Int = 5): List<String> = withContext(Dispatchers.IO) {

    val client = sharedOkHttpClient

    // 1. Get file titles from category
    val categoryUrl = "https://commons.wikimedia.org/w/api.php" +
            "?action=query" +
            "&list=categorymembers" +
            "&cmtitle=${category.replace(" ", "%20")}" +
            "&cmtype=file" +
            "&cmlimit=$limit" +
            "&format=json"

    val titles = try {
        client.newCall(Request.Builder().url(categoryUrl).build())
            .execute().use { response ->
                val json = JSONObject(response.body.string())
                val members = json.getJSONObject("query").getJSONArray("categorymembers")

                val list = mutableListOf<String>()
                for (i in 0 until members.length().coerceAtMost(limit)) {
                    list.add(members.getJSONObject(i).getString("title"))
                }
                list
            }
    } catch (_: Exception) { return@withContext emptyList() }

    if (titles.isEmpty()) return@withContext emptyList()

    // 2. Convert titles into a pipe-separated string for batch lookup
    val titleParam = titles.joinToString("|") { it.replace(" ", "%20") }

    // 3. Get actual image URLs
    val imageInfoUrl =
        "https://commons.wikimedia.org/w/api.php" +
                "?action=query" +
                "&titles=$titleParam" +
                "&prop=imageinfo" +
                "&iiprop=url" +
                "&format=json"

    return@withContext try {
        client.newCall(Request.Builder().url(imageInfoUrl).build())
            .execute().use { response ->
                val json = JSONObject(response.body.string())
                val pages = json.getJSONObject("query").getJSONObject("pages")

                val urls = mutableListOf<String>()

                val keys = pages.keys()
                while (keys.hasNext()) {
                    val key = keys.next()
                    val page = pages.getJSONObject(key)

                    if (page.has("imageinfo")) {
                        val info = page.getJSONArray("imageinfo").getJSONObject(0)
                        urls.add(info.getString("url")) // actual image URL
                    }
                }

                urls
            }
    } catch (_: Exception) {
        emptyList()
    }
}

// Retrieve up to maxImageCount images from category
// E.g. autoeater in Munich, 48.1402, 11.565
suspend fun getImagesInCategory(category: String, maxImageCount: Int): List<String> = withContext(Dispatchers.IO) {
    val client = sharedOkHttpClient

    // MediaWiki API URL (encoded properly)
    val url = "https://commons.wikimedia.org/w/api.php" +
            "?action=query" +
            "&list=categorymembers" +
            "&cmtitle=${category.replace(" ", "%20")}" +
            "&cmtype=file" +
            "&cmlimit=$maxImageCount" +
            "&format=json"

    val request = Request.Builder()
        .url(url)
        .build()

    try {
        client.newCall(request).execute().use { response ->
            val body = response.body.string()
            val json = JSONObject(body)

            val members = json
                .getJSONObject("query")
                .getJSONArray("categorymembers")

            val results = mutableListOf<String>()

            for (i in 0 until members.length().coerceAtMost(maxImageCount)) {
                val item = members.getJSONObject(i)
                results.add(item.getString("title"))
            }

            return@use results
        }
    }
    catch (_: Exception) { null }!!
}

fun createWikipediaUrl(s: String): String {
    if (s.isEmpty()) return ""

    val (lang, title) = s.split(":", limit = 2)
    val encodedTitle = URLEncoder.encode(title, StandardCharsets.UTF_8.toString())
        .replace("+", "_") // Wikipedia expects underscores for spaces
    return "https://$lang.wikipedia.org/wiki/$encodedTitle"
}

// Convert a encoded url back to Commons notation
fun normalizeCommonsUrl(input: String): String {
    // 1️⃣ Only handle Commons wiki links
    val commonsPrefix = "https://commons.wikimedia.org/wiki/"
    if (!input.startsWith(commonsPrefix, ignoreCase = true)) return input

    // 2️⃣ Extract the "title" part after /wiki/
    val encodedPart = input.substringAfter("/wiki/")

    // 3️⃣ Decode percent-encoding (e.g. %3A -> :, %C3%A4 -> ä)
    val decodedPart = URLDecoder.decode(encodedPart, StandardCharsets.UTF_8.name())

    // 4️⃣ Replace underscores with spaces for display consistency
    return decodedPart.replace('_', ' ')
}
